/*
 * Rahmenwerk-Plug-in "Parametrierung"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.param.editors.attribut;

import java.util.EventObject;

import org.eclipse.swt.events.FocusAdapter;
import org.eclipse.swt.events.FocusEvent;
import org.eclipse.swt.events.VerifyEvent;
import org.eclipse.swt.events.VerifyListener;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Text;

import de.bsvrz.dav.daf.main.Data;
import de.bsvrz.dav.daf.main.config.StringAttributeType;

/**
 * Editor zum Bearbeiten der Daten eines Attributs vom Basistyp "Zeichenkette".
 *
 * @author BitCtrl Systems GmbH, Uwe Peuker
 *
 */
public class TextAttributeEditor extends AbstraktAttributEditor<String> {

	/**
	 * Control zur Eingabe eines Textes.
	 */
	private final Text textFeld;

	/**
	 * Der Text als Java-Objekt.
	 */
	private String text = "";

	/**
	 * Maximal zulässige Länge des Textes.
	 */
	private final int maximaleLaenge;

	/**
	 * Konstruktor.
	 *
	 * @param parent
	 *            das umgebende Element
	 * @param stil
	 *            Stil-Flags für das Element
	 * @param editierbar
	 *            der Inhalt des Editors kann bearbeitet werden
	 * @param typ
	 *            der Typ des Attributs, dessen Wert bearbeitet werden soll
	 */
	public TextAttributeEditor(final Composite parent, final int stil, final boolean editierbar,
			final StringAttributeType typ) {
		super(parent);

		maximaleLaenge = typ.getMaxLength();

		final GridLayout layout = new GridLayout(1, false);
		layout.marginWidth = 0;
		layout.marginHeight = 0;
		layout.marginBottom = 0;
		layout.marginLeft = 0;
		layout.marginRight = 0;
		layout.marginTop = 0;

		setLayout(layout);

		textFeld = new Text(this, stil);
		textFeld.setText("");
		textFeld.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		textFeld.setEditable(editierbar);
		if (editierbar) {
			textFeld.addVerifyListener(new VerifyListener() {

				@Override
				public void verifyText(final VerifyEvent event) {
					final StringBuffer derText = new StringBuffer(textFeld.getText());
					if (event.start == event.end) {
						derText.insert(event.start, event.text);
					} else {
						derText.replace(event.start, event.end, event.text);
					}
					if (maximaleLaenge > 0) {
						if (derText.length() > maximaleLaenge) {
							event.doit = false;
						}
					}
					if (event.doit) {
						text = derText.toString();
					}
				}
			});

			textFeld.addFocusListener(new FocusAdapter() {

				/** wird aufgerufen, wenn das Textfeld den Focus verliert. */
				@Override
				public void focusLost(final FocusEvent event) {
					fireAenderungsListener(new EventObject(TextAttributeEditor.this));
				}
			});
		}
	}

	@Override
	public String getWert() {
		return text;
	}

	@Override
	public void getWert(final Data daten) {
		daten.asTextValue().setText(text);
	}

	@Override
	public void setWert(final Data daten) {
		text = daten.asTextValue().getText();
		textFeld.setText(text);
	}

	@Override
	public void setWert(final String wert) {
		this.text = wert;
	}
}
