/*
 * Rahmenwerk-Plug-in "Parametrierung"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.param.editors.helper;

import java.util.Iterator;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;

import com.bitctrl.Constants;

import de.bsvrz.buv.plugin.param.internal.ParamPlugin;
import de.bsvrz.dav.daf.main.Data;
import de.bsvrz.dav.daf.main.Data.Array;
import de.bsvrz.dav.daf.main.Data.NumberValue;
import de.bsvrz.dav.daf.main.Data.ReferenceValue;
import de.bsvrz.dav.daf.main.Data.TimeValue;
import de.bsvrz.dav.daf.main.config.AttributeType;
import de.bsvrz.dav.daf.main.config.DoubleAttributeType;
import de.bsvrz.dav.daf.main.config.IntegerAttributeType;
import de.bsvrz.dav.daf.main.config.IntegerValueRange;
import de.bsvrz.dav.daf.main.config.IntegerValueState;
import de.bsvrz.dav.daf.main.config.ReferenceAttributeType;
import de.bsvrz.dav.daf.main.config.StringAttributeType;
import de.bsvrz.dav.daf.main.config.TimeAttributeType;

/**
 * performs changes within an Datenverteiler-array object. Note that the added
 * fields might not always be initialized with <b><i>null</i></b> because the
 * Datenverteiler doesn't support that on any position. So if the added object
 * isn't initialized with <b><i>null</i></b> it's initialized either with some
 * matching values (in order not to violate the limitations of the object
 * itself) or it is just a copy of another item in the array (depending on the
 * insert-position and the type of object).
 *
 * <b>Der Typ BitfiledAttributeType wird nicht unterstützt !</b>
 *
 * AU: Es gab viele catch (Exception) Statements, die scheinbar aus
 * DaV2-Legacy-Tagen stammten. Diese wurden nur auskommentiert, um die Historie
 * besser zu verstehen.
 *
 * @author BitCtrl Systems GmbH, Thomas Thierfelder
 *
 */
public class ArrayObjectTransformer {

	/**
	 * Das Data-Objekt, dass den Array enthält.
	 */
	private Data data;

	/**
	 * der eigentliche Array.
	 */
	private Array array;

	/**
	 * Standardkonstruktor.
	 */
	public ArrayObjectTransformer() {
		super();
	}

	/**
	 * Konstruktor.
	 *
	 * @param data
	 *            - the data object to be transformed
	 */
	public ArrayObjectTransformer(final Data data) {
		setData(data);
	}

	/**
	 * Asks for a matching int value which doesn't violate the range of an
	 * IntegerAttributeType-object.
	 *
	 * @param valData
	 *            die auszuwertenden Daten
	 *
	 * @return a matching int value which doesn't violate the range of an
	 *         IntegerAttributeType-object
	 */
	private int getProperIntValue(final Data valData) {
		int min = 0;
		final AttributeType att = valData.getAttributeType();
		final IntegerValueRange ivr = ((IntegerAttributeType) att).getRange();
		if (ivr != null) {
			min = (int) ivr.getMinimum();
		} else {
			final List<IntegerValueState> ivStates = ((IntegerAttributeType) att).getStates();
			if (null != ivStates && ivStates.size() > 0) {
				final IntegerValueState state = ivStates.get(0);
				min = (int) state.getValue();
			}
		}
		return min;
	}

	/**
	 * adds one field to the array above the location-index.
	 *
	 * @param location
	 *            - Einfügeort
	 * @return the transformed array or null if the operation failed
	 */
	public Data addLineAbove(final int location) {
		if (data == null) {
			return null;
		}
		if ((array.getLength() + 1 > array.getMaxCount()) && (array.getMaxCount() != 0)) {
			return null;
		}

		array.setLength(array.getLength() + 1);
		for (int i = array.getLength() - 1; i >= (location == 0 ? location + 1 : location); i--) {
			copyData(array.getItem(i - 1), array.getItem(i));
		}
		// only try to initialize whith <<null>>
		copyData(null, array.getItem(location));

		return data;
	}

	/**
	 * adds one field to the array underneath the location-index.
	 *
	 * @param location
	 *            - Einfügeort
	 * @return the transformed array or null if the operation failed
	 */
	public Data addLineBelow(final int location) {
		if (data == null) {
			return null;
		}
		if ((array.getLength() + 1 > array.getMaxCount()) && (array.getMaxCount() != 0)) {
			return null;
		}

		array.setLength(array.getLength() + 1);
		for (int i = array.getLength() - 1; i > location; i--) {
			copyData(array.getItem(i - 1), array.getItem(i));
		}
		// only try to initialize whith <<null>>
		copyData(null, array.getItem(location + 1));

		return data;
	}

	/**
	 * Adds lines to the end of the array.
	 *
	 * @param size
	 *            - the number of lines to be added
	 * @parem number of objects to be added
	 * @return the transformed array or null if the operation failed
	 */
	public Data addLinesEnd(final int size) {
		if (data == null) {
			return null;
		}
		if ((array.getLength() + size > array.getMaxCount()) && (array.getMaxCount() != 0)) {
			return null;
		}

		final int len = array.getLength();
		array.setLength(array.getLength() + size);
		// only try to initialize whith <<null>>
		for (int i = len; i < array.getLength(); i++) {
			copyData(null, array.getItem(i));
		}

		return data;
	}

	/**
	 * Adds lines to the begin of the array.
	 *
	 * @param size
	 *            - how many lines to be added
	 * @parem number of objects to be added
	 * @return the transformed array or null if the operation failed
	 */
	public Data addLinesBegin(final int size) {
		if (data == null) {
			return null;
		}
		if ((array.getLength() + size > array.getMaxCount()) && (array.getMaxCount() != 0)) {
			return null;
		}

		final int len = array.getLength();
		int i;
		array.setLength(array.getLength() + size);
		if (len == 0) {
			copyData(null, array.getItem(0));
			return data;
		}

		for (i = len - 1; i >= 0; i--) {
			copyData(array.getItem(i), array.getItem(i + size));
		}

		// only try to initialize with <<null>>
		for (i = 0; i < size; i++) {
			copyData(null, array.getItem(i));
		}

		return data;
	}

	/**
	 * erases the entire data array.
	 *
	 * @return the transformed Data object or null if the operation failed
	 */
	public Data removeAllLines() {
		if (array == null) {
			return null;
		}
		array.setLength(0);
		return data;
	}

	/**
	 * removes one line.
	 *
	 * @param location
	 *            - the location of the object to be removed in the array
	 * @return the transformed Data object or null if the operation failed
	 */
	public Data removeSingleLine(final int location) {
		if (data == null) {
			return null;
		}

		for (int i = location + 1; i < array.getLength(); i++) {
			copyData(array.getItem(i), array.getItem(i - 1));
		}

		array.setLength(array.getLength() - 1);
		return data;
	}

	/**
	 * Erfragt the transformed Data object.
	 *
	 * @return the transformed Data object
	 */
	public Data getTransformedData() {
		return data;
	}

	/**
	 * references the data object to be transformed.
	 *
	 * @param data
	 *            die Daten
	 */
	public void setData(final Data data) {
		this.data = data;
		if (data != null) {
			this.array = data.asArray();
		}
	}

	/**
	 * copies the content of one data object into another one or erases the content
	 * of the target object as far as possible when you hand over <i>null</i> to
	 * source.
	 *
	 * @param data1
	 *            the source
	 * @param data2
	 *            the target
	 */
	private void copyData(final Data data1, final Data data2) {
		if (data2.isArray()) {
			copyArrayData(data1 == null ? null : data1.asArray(), data2.asArray());
		} else if (data2.isList()) {
			copyAttributListData(data1, data2);
		} else if (data2.getAttributeType() instanceof StringAttributeType) {
			copyTextData(data1, data2);
		} else if (data2.getAttributeType() instanceof IntegerAttributeType) {
			copyIntegerData(data1 == null ? null : data1, data2);
		} else if (data2.getAttributeType() instanceof DoubleAttributeType) {
			copyDoubleData(data1 == null ? null : data1.asUnscaledValue(), data2.asUnscaledValue());
		} else if (data2.getAttributeType() instanceof TimeAttributeType) {
			copyTimeData(data1 == null ? null : data1.asTimeValue(), data2.asTimeValue());
		} else if (data2.getAttributeType() instanceof ReferenceAttributeType) {
			copyReferenceData(data1 == null ? null : data1.asReferenceValue(), data2.asReferenceValue());
		} else {
			Logger.getLogger(ArrayObjectTransformer.class.getName()).log(Level.INFO,
					"Fehler in ArrayObjectTransformer"); //$NON-NLS-1$
		}
	}

	/**
	 * Kopiert Attributlisten.
	 *
	 * @param data1
	 *            - Quelle
	 * @param data2
	 *            - Ziel
	 */
	private void copyAttributListData(final Data data1, final Data data2) {
		if (data1 == null) {
			final Iterator<?> it2 = data2.iterator();

			while (it2.hasNext()) {
				copyData(null, (Data) it2.next());
			}
		} else {
			// try {
			final Iterator<?> it1 = data1.iterator();
			final Iterator<?> it2 = data2.iterator();

			while (it1.hasNext()) {
				copyData((Data) it1.next(), (Data) it2.next());
			}
			// } catch (final Exception ex) {
			// final Iterator<?> it2 = data2.iterator();
			//
			// while (it2.hasNext()) {
			// copyData(null, (Data) it2.next());
			// }
			// }
		}
	}

	/**
	 * Kopiert Objekte vom Typ IntegerValueType abhängig davon, ob es sich um einen
	 * Zustandstyp oder einen "normalen" Int handelt.
	 *
	 * @param d1
	 *            - Quelle
	 * @param d2
	 *            - Ziel
	 */
	private void copyIntegerData(final Data d1, final Data d2) {
		NumberValue data1 = null;
		NumberValue data2 = null;

		if (d1 != null) {
			data1 = d1.asUnscaledValue();
		}
		if (d2 != null) {
			data2 = d2.asUnscaledValue();
		}

		if (data2 != null) {
			if (data1 == null) {
				if (!data2.isState()) {
					final int i = getProperIntValue(d2);
					data2.set(i);
				}
			} else {
				try {
					if (data1.isState()) {
						data2.setState(data1.getState());
					} else {
						data2.set(data1.longValue());
					}
				} catch (final UnsupportedOperationException e) {
					if (!data2.isState()) {
						data2.set(getProperIntValue(d2));
					}
					ParamPlugin.getDefault().getLogger().warning("Konvertierungsproblem in DAF-API#1704", e);
				}
			}
		}
	}

	/**
	 * Kopiert Objekte vom Typ NumberValueType.
	 *
	 * @param data1
	 *            - Quelle
	 * @param data2
	 *            - Ziel
	 */
	private void copyDoubleData(final NumberValue data1, final NumberValue data2) {
		if (data1 == null) {
			data2.set(0);
		} else {
			// try {
			data2.set(data1.doubleValue());
			data2.set(data1.floatValue());
			// } catch (final Exception ex) {
			// data2.set(0);
			// }
		}
	}

	/**
	 * Kopiert Objekte vom Typ TextValueType.
	 *
	 * @param data1
	 *            - Quelle. Kann <code>null</code> sein.
	 * @param data2
	 *            - Ziel. Darf nicht <code>null</code> sein.
	 */
	private void copyTextData(final Data data1, final Data data2) {
		if (null == data1) {
			final StringAttributeType strAttrType = (StringAttributeType) data2.getAttributeType();
			String defaultValue = strAttrType.getDefaultAttributeValue();
			if (null == defaultValue) {
				defaultValue = Constants.EMPTY_STRING;
			}
			data2.asTextValue().setText(defaultValue);
		} else {
			data2.asTextValue().setText(data1.asTextValue().getText());
		}
	}

	/**
	 * Kopiert ArrayObjekte.
	 *
	 * @param data1
	 *            - Quelle
	 * @param data2
	 *            - Ziel
	 */
	private void copyArrayData(final Array data1, final Array data2) {
		if (data1 == null) {
			data2.setLength(0);
		} else {
			// try {
			data2.setLength(data1.getLength());
			for (int i = 0; i < data1.getLength(); i++) {
				copyData(data1.getItem(i), data2.getItem(i));
			}
			// } catch (final Exception ex) {
			// data2.setLength(0);
			// }
		}
	}

	/**
	 * Kopiert Objekte vom Typ TimeValueType.
	 *
	 * @param data1
	 *            - Quelle
	 * @param data2
	 *            - Ziel
	 */
	private void copyTimeData(final TimeValue data1, final TimeValue data2) {
		if (data1 == null) {
			data2.setMillis(0);
			data2.setSeconds(0);
		} else {
			// try {
			data2.setMillis(data1.getMillis());
			data2.setSeconds(data1.getSeconds());
			data2.setText(data1.getText());
			// } catch (final Exception ex) {
			// data2.setMillis(0);
			// data2.setSeconds(0);
			// }
		}
	}

	/**
	 * Kopiert Objekte vom Typ ReferenceValueType.
	 *
	 * @param data1
	 *            - Quelle
	 * @param data2
	 *            - Ziel
	 */
	private void copyReferenceData(final ReferenceValue data1, final ReferenceValue data2) {
		if (data1 == null) {
			return;
		}
		// try {
		data2.setSystemObject(data1.getSystemObject());
		data2.setText(data1.getText());
		// } catch (final Exception ex) {
		// Logger.getLogger(getClass().getName()).log(Level.FINE, "", ex);
		// }
	}
}
