/*
 * Rahmenwerk-Plug-in "Parametrierung"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.param.editors.helper;

import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.Font;
import org.eclipse.swt.graphics.FontData;
import org.eclipse.swt.widgets.Display;

/**
 * Objekt, welches einen String durch Attribute wie Farbe oder Font näher
 * beschreibt. Dadurch kann man z.B. bei einem Methodenaufruf sowohl den Text
 * selbst als auch dessen Attribute in einem einzigen Parameter übergeben.
 *
 * @author BitCtrl Systems GmbH, Albrecht Uhlmann
 *
 */
public class DecoratableString {

	/**
	 * der anzuzeigende Text.
	 */
	private String text;

	/**
	 * der Schriftstil, möglich sind SWTNORMAL, SWT.ITALIC und SWT.BOLD, oder eine
	 * beliebige VerODERung derselben.
	 */
	private int style;

	/**
	 * die Höhe der Schrift in Pixeln, z.B. 8 oder 10.
	 */
	private int height;

	/**
	 * der Name der Schriftart, z.B. Arial.
	 */
	private String fontName;

	/**
	 * SWT.COLOR_*-Objekt für die Vordergrundfarbe.
	 */
	private int foreground;

	/**
	 * SWT.COLOR_*-Objekt für die Hintergrundfarbe.
	 */
	private int background;

	/**
	 * der erzeugte Font.
	 */
	private Font font;

	/**
	 * Konstruktor übernimmt nur die Proeprties.
	 *
	 * @param text
	 *            der anzuzeigende Text
	 * @param style
	 *            der Schriftstil, möglich sind SWTNORMAL, SWT.ITALIC und SWT.BOLD,
	 *            oder eine beliebige VerODERung derselben.
	 * @param height
	 *            die Höhe der Schrift in Pixeln, z.B. 8 oder 10
	 * @param fontName
	 *            der Name der Schriftart, z.B. Arial
	 * @param foreground
	 *            SWT.COLOR_*-Objekt für die Vordergrundfarbe
	 * @param background
	 *            SWT.COLOR_*-Objekt für die Hintergrundfarbe
	 */
	public DecoratableString(final String text, final int style, final int height, final String fontName,
			final int foreground, final int background) {
		super();
		this.text = text;
		this.style = style;
		this.height = height;
		this.fontName = fontName;
		this.foreground = foreground;
		this.background = background;
	}

	/**
	 * Liefert ein System-Color-Objekt aus dem int-Member.
	 *
	 * @return das Color-Objekt für die Hintergrundfarbe
	 */
	public final Color getBackgroundColor() {
		return Display.getDefault().getSystemColor(background);
	}

	/**
	 * Erzeugt ein SWT-Font-Objekt aus den entsprechenden Membern, falls es noch
	 * nicht existiert, und liefert es zurück.
	 *
	 * @return das SWT-Font-Objekt
	 */
	public final Font getFont() {
		if (font == null || font.isDisposed()) {
			final FontData fd = new FontData(fontName, height, style);
			font = new Font(Display.getDefault(), fd);
		}
		return font;
	}

	/**
	 * Liefert ein System-Color-Objekt aus dem int-Member.
	 *
	 * @return das Color-Objekt für die Vordergrundfarbe
	 */
	public final Color getForegroundColor() {
		return Display.getDefault().getSystemColor(foreground);
	}

	/**
	 * Sammelmethode, um die erzeugten SWT-Grafikobjekte zu dispose.
	 *
	 * Bitte diese Methode benutzen und nicht getFont().dispose().
	 *
	 */
	public void dispose() {
		if (font != null && !font.isDisposed()) {
			font.dispose();
			font = null;
		}
	}

	/**
	 * Liefert ein SWT.COLOR_*-Objekt für die Hintergrundfarbe.
	 *
	 * @return die Farbcodierung
	 */
	public final int getBackground() {
		return background;
	}

	/**
	 * Setzt SWT.COLOR_*-Objekt für die Hintergrundfarbe.
	 *
	 * @param background
	 *            Festzulegende Farbcodierung
	 */
	public final void setBackground(final int background) {
		this.background = background;
	}

	/**
	 * Liefert den Namen des Font.
	 *
	 * @return der Fontname
	 */
	public final String getFontName() {
		return fontName;
	}

	/**
	 * Setzt den Namen des Font.
	 *
	 * @param fontName
	 *            Festzulegender Fontname
	 */
	public final void setFontName(final String fontName) {
		this.fontName = fontName;
	}

	/**
	 * Liefert ein SWT.COLOR_*-Objekt für die Vordergrundfarbe.
	 *
	 * @return die Farbcodierung
	 */
	public final int getForeground() {
		return foreground;
	}

	/**
	 * Setzt ein SWT.COLOR_*-Objekt für die Vordergrundfarbe.
	 *
	 * @param foreground
	 *            Festzulegende Farbcodierung
	 */
	public final void setForeground(final int foreground) {
		this.foreground = foreground;
	}

	/**
	 * Setzt die Höhe der Schrift in Pixeln, z.B. 8 oder 10.
	 *
	 * @return die Höhe
	 */
	public final int getHeight() {
		return height;
	}

	/**
	 * Liefert die Höhe der Schrift in Pixeln, z.B. 8 oder 10.
	 *
	 * @param height
	 *            Festzulegende Höhe
	 */
	public final void setHeight(final int height) {
		this.height = height;
	}

	/**
	 * Liefert den Schriftstil, möglich sind SWTNORMAL, SWT.ITALIC und SWT.BOLD,
	 * oder eine beliebige VerODERung derselben.
	 *
	 * @return der Stil.
	 */
	public final int getStyle() {
		return style;
	}

	/**
	 * Setzt den Schriftstil, möglich sind SWTNORMAL, SWT.ITALIC und SWT.BOLD, oder
	 * eine beliebige VerODERung derselben.
	 *
	 * @param style
	 *            Festzulegender Stil
	 */
	public final void setStyle(final int style) {
		this.style = style;
	}

	/**
	 * Liefert den anzuzeigenden Text.
	 *
	 * @return der Text
	 */
	public final String getText() {
		return text;
	}

	/**
	 * Setzt den anzuzeigenden Text.
	 *
	 * @param text
	 *            Festzulegender Text
	 */
	public final void setText(final String text) {
		this.text = text;
	}

}
