/*
 * Rahmenwerk-Plug-in "Parametrierung"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.param.editors.helper;

import java.util.ArrayList;
import java.util.List;

import de.bsvrz.dav.daf.main.Data;
import de.bsvrz.dav.daf.main.config.AttributeType;
import de.bsvrz.dav.daf.main.config.DoubleAttributeType;
import de.bsvrz.dav.daf.main.config.IntegerAttributeType;
import de.bsvrz.dav.daf.main.config.ReferenceAttributeType;
import de.bsvrz.dav.daf.main.config.StringAttributeType;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.dav.daf.main.config.TimeAttributeType;
import de.bsvrz.sys.funclib.debug.Debug;

/**
 * dummy-object in which the ResultData-tree is mirrored and where every
 * attribute comes whith its own data object (even if this doesn't even exists)
 * Note: The class is still used to provide input-data for the Contentproviders
 * even though only the data object is the only one used. This is due to the
 * fact, that its possible to create an entire empty Data structure according to
 * ResultData-object when it's data-object is null
 *
 * @author BitCtrl Systems GmbH, Thomas Thierfelder
 *
 */
public class MyTreeObject implements Comparable<MyTreeObject> {

	/**
	 * Debug-Logger für alle MyTreeObject-Instanzen. Ein Logger pro Instanz lohnt
	 * nicht.
	 */
	private static final Debug DEBUG = Debug.getLogger();

	/**
	 * das Wurzelobjekt.
	 */
	private MyTreeObject root;

	/**
	 * ein Attribut.
	 */
	private Object attribute;

	/**
	 * die Daten.
	 */
	private Object data;

	/**
	 * Originalwert. Kann <code>null</code> sein, wenn nämlich {@link #data} null
	 * oder nicht plain ist. Andernfalls ergibt sich der Typ aus dem
	 * {@link AttributeType} von {@link #data}
	 */
	private Object originalValue;

	/**
	 * die Kinder.
	 */
	private final List<MyTreeObject> children = new ArrayList<>();

	/**
	 * die ID.
	 */
	private final long id;

	/**
	 * Konstruktor.
	 */
	public MyTreeObject() {
		this.id = IDCounter.getId();
	}

	/**
	 * Konstruktor.
	 *
	 * @param attribute
	 *            das Attribut
	 * @param data
	 *            die Daten
	 */
	public MyTreeObject(final Object attribute, final Object data) {
		this.attribute = attribute;
		this.data = data;
		if ((data instanceof Data) && ((Data) data).isPlain()) {
			final AttributeType attributeType = ((Data) data).getAttributeType();
			if (attributeType instanceof IntegerAttributeType) {
				originalValue = Long.valueOf(((Data) data).asUnscaledValue().longValue());
			} else if (attributeType instanceof DoubleAttributeType) {
				originalValue = Double.valueOf(((Data) data).asUnscaledValue().doubleValue());
			} else if (attributeType instanceof StringAttributeType) {
				originalValue = ((Data) data).asTextValue().getText();
			} else if (attributeType instanceof TimeAttributeType) {
				originalValue = Long.valueOf(((Data) data).asTimeValue().getMillis());
			} else if (attributeType instanceof ReferenceAttributeType) {
				originalValue = ((Data) data).asReferenceValue().getSystemObject();
			}
		}
		this.id = IDCounter.getId();
		MyTreeObject.DEBUG.finer("Neues " + getClass().getSimpleName() + "-Objekt erzeugt", this);
	}

	/**
	 * fügt dem ELement ein Kind hinzu.
	 *
	 * @param child
	 *            das Kind
	 */
	public void add(final MyTreeObject child) {
		children.add(child);
		child.setRoot(this);
	}

	/**
	 * liefert die Liste der Kinder.
	 *
	 * @return die Kinder
	 */
	public List<MyTreeObject> getChildren() {
		return children;
	}

	/**
	 * setzt das Wurzelobjekt.
	 *
	 * @param root
	 *            das Objekt
	 */
	public void setRoot(final MyTreeObject root) {
		this.root = root;
	}

	/**
	 * liefert das Wurzelobjekt.
	 *
	 * @return die Wurzel
	 */
	public MyTreeObject getRoot() {
		return this.root;
	}

	/**
	 * liefert das Attribut.
	 *
	 * @return das Attribut
	 */
	public Object getAttribute() {
		return this.attribute;
	}

	/**
	 * liefert die Daten.
	 *
	 * @return die Daten
	 */
	public Object getData() {
		return this.data;
	}

	/**
	 * setzt die Daten des Objekts.
	 *
	 * @param data
	 *            die Daten
	 */
	public void setData(final Object data) {
		this.data = data;
	}

	/**
	 * setzt das Attribut.
	 *
	 * @param attribute
	 *            das Attribut
	 */
	public void setAttribute(final Object attribute) {
		this.attribute = attribute;
	}

	/**
	 * ermittelt, ob das Objekt Kinder hat.
	 *
	 * @return Kinder vorhanden ?
	 */
	public boolean hasChildren() {
		return !children.isEmpty();
	}

	/**
	 * zeigt das Objekt an.
	 *
	 * @param nest
	 *            Ebene des Objekts innerhalb des Baums
	 */
	public void showTree(final int nest) {
		if (hasChildren()) {
			for (final MyTreeObject obj : getChildren()) {
				obj.showTree(nest + 1);
			}
		}
	}

	/**
	 * Liefert den vollständigen Pfad des in diesem Objekt verkapselten Datums
	 * innerhalb des kompletten Data-Baums. Die einzelnen Elemente werden durch
	 * Punkt getrennt.
	 *
	 * @return der Pfad. Kann leer, aber nicht null sein.
	 */
	public String getFullPath() {
		String path = "";
		if (getData() instanceof Data) {
			path = ((Data) getData()).getName();
			MyTreeObject rootTreeObject = getRoot();
			while ((rootTreeObject != null) && (rootTreeObject.getData() instanceof Data)) {
				final Data rootData = (Data) rootTreeObject.getData();
				path = rootData.getName() + '.' + path;
				rootTreeObject = rootTreeObject.getRoot();
			}
		}
		return path;
	}

	/*
	 * (non-Javadoc)
	 *
	 * @see java.lang.Object#toString()
	 */
	@Override
	public String toString() {
		final StringBuilder b = new StringBuilder("MyTreeObject Id=");
		b.append(id);
		b.append(", Attribut: ");
		if (attribute != null) {
			b.append(attribute.toString());
			b.append(" (Typ: ");
			if (data instanceof Data) {
				b.append(((Data) data).getAttributeType().toString());
			} else {
				b.append(attribute.getClass().getName());
			}
			b.append(")");
		} else {
			b.append("unbekannt");
		}
		b.append(", Aktuelle Daten: ");
		if (data != null) {
			b.append(data.toString());
		} else {
			b.append("Keine");
		}
		b.append(", Originale Daten: ");
		if (originalValue != null) {
			b.append(originalValue.toString());
		} else {
			b.append("Keine");
		}
		return b.toString();
	}

	/**
	 * Liefert den originalen Wert des zugrunde liegenden Objekts.
	 *
	 * @return der Originalwert oder <code>null</code>, wenn nämlich data null oder
	 *         nicht plain ist. Andernfalls ergibt sich der Typ aus dem
	 *         AttributeTypevon data. (gleiches Prinzip wie bei
	 *         {@link IParaEditCompositeContainer#valueModified(ParaEditComposite, MyTreeObject, Object, Object, AttributeType)}
	 *         )
	 */
	public Object getOriginalValue() {
		return originalValue;
	}

	/**
	 * Für den Fall, dass dieses Objekt ein Blatt des Data-Baums repräsentiert, wird
	 * der enthaltene Einzelwert mit dem Originalwert verglichen und bei
	 * Ungleichheit true geliefert. Falls das enthaltene Datum komplex ist, werden
	 * die Kinder gefragt. Sobald auch nur eines true liefert, wird die Rekursion
	 * abgebrochen und true zurückgeliefert..
	 *
	 * @return true - Objekt oder eines der Kinder modifiziert
	 */
	public boolean isDirty() {
		boolean result = false;
		if (data instanceof Data) {
			final Data currentData = (Data) data;
			if (currentData.isPlain() && (getOriginalValue() != null)) {
				Object currentValue = null;
				final AttributeType attributeType = currentData.getAttributeType();
				if (attributeType instanceof IntegerAttributeType) {
					currentValue = Long.valueOf(currentData.asUnscaledValue().longValue());
				} else if (attributeType instanceof DoubleAttributeType) {
					currentValue = Double.valueOf(currentData.asUnscaledValue().doubleValue());
				} else if (attributeType instanceof StringAttributeType) {
					currentValue = currentData.asTextValue().getText();
				} else if (attributeType instanceof TimeAttributeType) {
					currentValue = Long.valueOf(currentData.asTimeValue().getMillis());
				} else if (attributeType instanceof ReferenceAttributeType) {
					currentValue = currentData.asReferenceValue().getSystemObject();
				}
				if ((null == currentValue) || !getOriginalValue().equals(currentValue)) {
					result = true;
				}
			} else {
				for (final MyTreeObject child : getChildren()) {
					if (child.isDirty()) {
						result = true;
						break;
					}
				}
			}
		}
		return result;
	}

	/**
	 * Überträgt die Werte im data-Objekt in originalValue. Ruft diese Methode auch
	 * rekursiv an den Kindern auf.
	 */
	public void commit() {
		if ((data instanceof Data) && ((Data) data).isPlain()) {
			final AttributeType attributeType = ((Data) data).getAttributeType();
			if (attributeType instanceof IntegerAttributeType) {
				originalValue = Long.valueOf(((Data) data).asUnscaledValue().longValue());
			} else if (attributeType instanceof DoubleAttributeType) {
				originalValue = Double.valueOf(((Data) data).asUnscaledValue().doubleValue());
			} else if (attributeType instanceof StringAttributeType) {
				originalValue = ((Data) data).asTextValue().getText();
			} else if (attributeType instanceof TimeAttributeType) {
				originalValue = Long.valueOf(((Data) data).asTimeValue().getMillis());
			} else if (attributeType instanceof ReferenceAttributeType) {
				originalValue = ((Data) data).asReferenceValue().getSystemObject();
			}
		}
		for (final MyTreeObject child : getChildren()) {
			child.commit();
		}
	}

	/**
	 * Überträgt die Werte in originalValue in das data -Objekt. Ruft diese Methode
	 * auch rekursiv an den Kindern auf.
	 */
	public void revert() {
		if ((data instanceof Data) && ((Data) data).isPlain() && (originalValue != null)) {
			final AttributeType attributeType = ((Data) data).getAttributeType();
			if (attributeType instanceof IntegerAttributeType) {
				((Data) data).asUnscaledValue().set((Long) originalValue);
			} else if (attributeType instanceof DoubleAttributeType) {
				((Data) data).asUnscaledValue().set((Double) originalValue);
			} else if (attributeType instanceof StringAttributeType) {
				((Data) data).asTextValue().setText(String.valueOf(originalValue));
			} else if (attributeType instanceof TimeAttributeType) {
				((Data) data).asTimeValue().setMillis((Long) originalValue);
			} else if (attributeType instanceof ReferenceAttributeType) {
				((Data) data).asReferenceValue().setSystemObject((SystemObject) originalValue);
			}
		}
		for (final MyTreeObject child : getChildren()) {
			child.revert();
		}
	}

	@Override
	public int compareTo(final MyTreeObject o) {
		int result = 0;
		if ((data instanceof Data) && (o != null) && (o.getData() instanceof Data)) {
			final Data currentData = (Data) data;
			final Data otherData = (Data) o.getData();
			if (currentData.isPlain() && otherData.isPlain()) {
				Object currentValue = null;
				Object otherValue = null;
				final AttributeType attributeType = currentData.getAttributeType();
				if (attributeType instanceof IntegerAttributeType) {
					currentValue = Long.valueOf(currentData.asUnscaledValue().longValue());
					otherValue = Long.valueOf(otherData.asUnscaledValue().longValue());
				} else if (attributeType instanceof DoubleAttributeType) {
					currentValue = new Double(currentData.asUnscaledValue().doubleValue());
					otherValue = Double.valueOf(otherData.asUnscaledValue().doubleValue());
				} else if (attributeType instanceof StringAttributeType) {
					currentValue = currentData.asTextValue().getText();
					otherValue = otherData.asTextValue().getText();
				} else if (attributeType instanceof TimeAttributeType) {
					currentValue = Long.valueOf(currentData.asTimeValue().getMillis());
					otherValue = Long.valueOf(otherData.asTimeValue().getMillis());
				} else if (attributeType instanceof ReferenceAttributeType) {
					currentValue = currentData.asReferenceValue().getSystemObject();
					otherValue = otherData.asReferenceValue().getSystemObject();
				}
				if ((null == currentValue) || !currentValue.equals(otherValue)) {
					result = 1;
				}
			}
			/*
			 * else { for (MyTreeObject child : getChildren()) { if (child.compareTo(o) !=
			 * 0) { result = 1; break; } } }
			 */
		}
		return result;
	}

	/**
	 * Liefert die ID.
	 *
	 * @return die ID.
	 */
	public long getId() {
		return id;
	}
}
