/*
 * Rahmenwerk-Plug-in "Parametrierung"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.param.editors.helper;

import java.util.ArrayList;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.eclipse.jface.action.GroupMarker;
import org.eclipse.jface.action.IMenuListener;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.viewers.CellEditor;
import org.eclipse.jface.viewers.ComboBoxCellEditor;
import org.eclipse.jface.viewers.ICellModifier;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.TextCellEditor;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.swt.widgets.MessageBox;
import org.eclipse.swt.widgets.Text;
import org.eclipse.swt.widgets.Tree;
import org.eclipse.swt.widgets.TreeColumn;
import org.eclipse.swt.widgets.TreeItem;
import org.eclipse.ui.IWorkbenchActionConstants;
import org.eclipse.ui.forms.AbstractFormPart;
import org.eclipse.ui.forms.widgets.FormToolkit;
import org.eclipse.ui.forms.widgets.Section;
import org.eclipse.ui.forms.widgets.SharedScrolledComposite;

import com.bitctrl.Constants;

import de.bsvrz.buv.plugin.param.Zeichenketten;
import de.bsvrz.buv.plugin.param.internal.ParamPlugin;
import de.bsvrz.buv.plugin.param.internal.RahmenwerkService;
import de.bsvrz.buv.rw.bitctrl.eclipse.SystemObjectComparatorAlphabetical;
import de.bsvrz.dav.daf.main.ClientDavInterface;
import de.bsvrz.dav.daf.main.Data;
import de.bsvrz.dav.daf.main.config.AttributeListDefinition;
import de.bsvrz.dav.daf.main.config.AttributeType;
import de.bsvrz.dav.daf.main.config.DoubleAttributeType;
import de.bsvrz.dav.daf.main.config.IntegerAttributeType;
import de.bsvrz.dav.daf.main.config.IntegerValueRange;
import de.bsvrz.dav.daf.main.config.IntegerValueState;
import de.bsvrz.dav.daf.main.config.ReferenceAttributeType;
import de.bsvrz.dav.daf.main.config.StringAttributeType;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.dav.daf.main.config.TimeAttributeType;
import de.bsvrz.puk.param.lib.MethodenBibliothek;
import de.bsvrz.puk.param.lib.Parameter;
import de.bsvrz.puk.param.lib.ParameterClientException;
import de.bsvrz.puk.param.lib.ParameterInfo;
import de.bsvrz.puk.param.lib.ParameterManager;
import de.bsvrz.puk.param.lib.daten.DataWithTime;
import de.bsvrz.sys.funclib.debug.Debug;

/**
 * Editor-composite which enables the user to edit a Data-object (from a
 * Parameter-Object which is to be handed over) in the appropriate way.
 *
 * @author BitCtrl Systems GmbH, Thomas Thierfelder, Albrecht Uhlmann
 *
 *
 */
public class ParaEditComposite {

	/**
	 * Debug-Logger für alle ParaEditComposite-Instanzen. Ein Logger pro Instanz
	 * lohnt nicht.
	 */
	private static final Debug DEBUG = Debug.getLogger();

	/**
	 * Maximale Länge des Textes hinter &quot;Ursache&quot; in {@link #headLine} .
	 */
	private static final int MAX_LEN_URSACHE_TEXT = 100;

	/** letzte Fehlermeldung. */
	private String errorMessage;

	/** das Editierte Objekt. */
	private MyTreeObject dataObject;

	/** Das Zwischenobjekt, an dem die flachen Listen für Exporte hängen. */
	private MyDataObject mediatorDataObject;

	/** Toplevel-Schattenobjekte. */
	private final MyTreeObject[] shadowDataObjects;

	/**
	 * Für die dreispialtige Variante wird hier das Mapping zwischen
	 * korrespondierenden Attributen gehalten.
	 */
	private final Map<MyTreeObject, MyTreeObject[]> shadowDataObjectMap = new LinkedHashMap<>();

	/**
	 * das editierte Objekt befindet sich in Element Nummer 0. Weitere Elemente sind
	 * optional, müssen sich aber auf dieselbe Attributgruppe beziehen.
	 */
	private final Parameter[] parameters;

	/** TableListener. */
	private TableSelectionListener currentTL;

	/** Neue Items. */
	private final List<Integer> newItems = new ArrayList<>();

	/** Warnungsfarbe. */
	private static Color alertColor;

	/** Nachrichtenfarbe. */
	private static Color messageColor;

	/** Statuszeile unten. */
	private Label infoLine;

	/** Überschrift. */
	private Label headLine;

	/** Spaltenköpfe. */
	private final String[] properties;

	/** TreeViewer. */
	private TreeViewer treeViewer;

	/** der Table innerhalb von treeViewer. */
	private Tree tree;

	/** Steuert die Kontextmenüs. */
	private MenuManager contextMenuManager;

	/** Menüteil.1. */
	private MenuManager lineAdd;

	/** Menüteil.2. */
	private MenuManager addOne;

	/** Menüteil.3. */
	private MenuManager addMore;

	/** Editor für DoubleAttrType. */
	private TextCellEditor doubleEdit;

	/** Editor für TextAttrType. */
	private TextCellEditor refEdit;

	/** Editor für IntegerAttrType. */
	private TextCellEditor integerEdit;

	/** Editor für StringAttrType. */
	private TextCellEditor stringEdit;

	/** Editor für TimeAttrType. */
	private TextCellEditor timeEdit;

	/** Editor für IntegerAttrType mit festen Zuständen. */
	private ComboBoxCellEditor stateEdit;

	/** Hier werden die aktuellen Editoren eingespielt. */
	private final CellEditor[] editors;

	/** Aktion zum Einfügen über einem Element. */
	private AddRowAction lineAddAbove;

	/** Aktion zum Einfügen unter einem Element. */
	private AddRowAction lineAddBelow;

	/** Aktion zum Löschen. */
	private LineRemoveAction lineRemove;

	/** Aktion zum Einfügen eines Feldes in Array. */
	private AddRowAction arrayAddOneLast;

	/** Aktion zum Einfügen eines Feldes in Array. */
	private AddRowAction arrayAddOneFirst;

	/** Aktion zum Einfügen mehrerer Felder in Array. */
	private AddRowAction arrayAddMoreLast;

	/** Aktion zum Einfügen mehrerer Felder in Array. */
	private AddRowAction arrayAddMoreFirst;

	/** Aktion zum Löschen aller Elemente eines Arrays. */
	private RemoveAllAction arrayRemoveAll;

	/** Aktueller Expansionsstatus des Baums. */
	private ExpandState treeState;

	/** TextModifier. */
	private final TextModifier textModifier;

	/** StateModifier. */
	private final StateModifier stateModifier = new StateModifier();

	/** ReferenceModifier. */
	private final ReferenceModifier referenceModifier = new ReferenceModifier();

	/**
	 * Alphabetischer Sortierer für die Comboboxen von SystemObjects und
	 * IntegerValueStates.
	 */
	private final SystemObjectComparatorAlphabetical systemObjectComparator = new SystemObjectComparatorAlphabetical(
			true);

	/** Beschreibung eines optionalen zusätzlichen Textes. */
	private final DecoratableString additionalText;

	/** SWT-Label für einen optionalen zusätzlichen Text. */
	private Label additionalTextLabel;

	/** Das Hauptcontrol, welches auch als Sectionclient fungiert. */
	private final Composite sectionClient;

	/** Formtoolkit zu Erzeugen der Controls. */
	private final FormToolkit toolkit;

	/** 'Verändert'-Flag. */
	private boolean dirty;

	/** Der Container, in dem wir uns befinden. Kann null sein. */
	private final IParaEditCompositeContainer container;

	/** Der FormPart, in dem wir uns befinden. */
	private AbstractFormPart formPart;

	/**
	 * Feld von ParameterInfo-Objekten. Muss mindestens ein Element enthalten, das
	 * sind die Vorgabewerte in Spalte 2. Kann auch zwei Elemente enthalten, dann
	 * wird eine dritte Spalte mit den Sollwerten benutzt.
	 */
	private final ParameterInfo[] patameterInfos;

	/** Die Arbeitskopie, auf die die Editoren arbeiten. */
	private DataWithTime workingCopy;

	/**
	 * true, wenn Editieren nicht möglich sein soll. Die Daten können dann nur
	 * betrachtet werden.
	 */
	private final boolean readOnly;

	/**
	 * the standard-constructor: it creates a copy of the data object for later use.
	 *
	 * @param toolkit
	 *            Formtoolkit zum erzeugen neuer Widgets
	 * @param sektion
	 *            Master-Composite
	 * @param parameters
	 *            das zu editierende Objekt. Das Feld muss mindestens ein Element
	 *            enthalten, auf das gearbeitet wird. Jedes weitere Element muss das
	 *            gleiche Objekt und die gleiche Attributgruppe referenzieren. Der
	 *            enthaltene Datensatz wird dann in einer eigenen Spalte
	 *            dargestellt.
	 * @param additionalText
	 *            zusätzlicher Text, der als oberste Zeile(n) angezeigt wird. Wenn
	 *            null übergeben wird, wird auch nichts angezeigt.
	 * @param container
	 *            übergerodnetes Objekt. Darf null sein.
	 * @param readOnly
	 *            true, wenn Editieren nicht möglich sein soll. Die Daten können
	 *            dann nur betrachtet werden
	 */
	public ParaEditComposite(final FormToolkit toolkit, final Section sektion, final Parameter[] parameters,
			final DecoratableString additionalText, final IParaEditCompositeContainer container,
			final boolean readOnly) {
		this.readOnly = readOnly;
		this.toolkit = toolkit;
		this.container = container;
		sectionClient = toolkit.createComposite(sektion);
		sectionClient.setLayout(new GridLayout(1, false));
		final GridData gd = new GridData(SWT.LEFT, SWT.TOP, true, false);
		gd.heightHint = calculateOptimalHeight();
		sectionClient.setLayoutData(gd);
		this.additionalText = additionalText;

		final ClientDavInterface davIf = RahmenwerkService.getService().getObjektFactory().getDav();
		final Data data;
		if (parameters[0].getData() == null) {
			data = davIf.createData(parameters[0].getAtg());
		} else {
			if (isReadOnly()) {
				data = parameters[0].getData();
			} else {
				data = parameters[0].getData().createModifiableCopy();
			}
		}
		workingCopy = new DataWithTime(data, System.currentTimeMillis());
		this.parameters = parameters;
		int infoLoop;
		patameterInfos = new ParameterInfo[this.parameters.length];
		for (infoLoop = 0; infoLoop < this.parameters.length; ++infoLoop) {
			patameterInfos[infoLoop] = new ParameterInfo(this.parameters[infoLoop].getObjekt(),
					this.parameters[infoLoop].getTyp(), this.parameters[infoLoop].getAtg(),
					this.parameters[infoLoop].getSim());

		}
		editors = new CellEditor[1 + parameters.length];
		final List<String> columnPropertiesList = new ArrayList<>();
		columnPropertiesList.add("Attribut");
		if (this.parameters.length > 1) {
			columnPropertiesList.add("Vorgabe");
			columnPropertiesList.add("Soll");
		} else {
			columnPropertiesList.add("Wert");
		}
		properties = columnPropertiesList.toArray(new String[columnPropertiesList.size()]);
		textModifier = new TextModifier(this, properties[0]);
		shadowDataObjects = new MyTreeObject[this.parameters.length - 1];
		createPartControl();
		showData(this.parameters, false);
		sektion.setClient(sectionClient);
	}

	/**
	 * Erfragt die Menge von Kindern dieses Objektes, die <i>null</i> sind.
	 *
	 * @param root
	 *            - die Wurzel
	 * @return die Menge von Kindern
	 */
	int innerCheck(final MyTreeObject root) {
		if (root == null) {
			return 0;
		}
		if (root.hasChildren()) {
			int dummy = 0;
			for (final MyTreeObject treeObj : root.getChildren()) {
				dummy += innerCheck(treeObj);
			}
			return dummy;
		}
		final Data data = (Data) root.getData();
		if (data.valueToString().contains("<<null>>")) {
			return 1;
		}
		return 0;
	}

	/**
	 * Erfragt die Fehlermeldungen.
	 *
	 * @return die Fehlermeldung
	 */
	public String getErrorMessage() {
		return errorMessage;
	}

	/**
	 * Erfragt, ob noch unausgefüllte objekte im Editor stehen.
	 *
	 * @return kann beendet werden ?
	 */
	public boolean isReadyForFinish() {
		return innerCheck(dataObject) == 0;
	}

	/**
	 * Erfragt, ob die nächste Seite des Wizards aufgerufen werden kann.
	 *
	 * @return nächste Seite kann ausgewählt werden ?
	 */
	public boolean isReadyForNext() {
		return innerCheck(dataObject) > 0;
	}

	/**
	 * provides the location of a tree-element within the its direct parent tree.
	 *
	 * @param myTreeObject
	 *            - the tree element
	 * @return the location
	 */
	int getLocation(final MyTreeObject myTreeObject) {
		final MyTreeObject root = myTreeObject.getRoot();
		int i = 0;

		for (final MyTreeObject obj : root.getChildren()) {
			if (obj.getId() == myTreeObject.getId()) {
				return i;
			}
			i++;
		}
		return -1;
	}

	/**
	 * creates the popupmenus and initalizes the different actions for manipulating
	 * the array items.
	 */
	private void createContextMenu() {
		lineAddAbove = new AddRowAction("über diesem Element", ParaEditCompositeConstants.ABOVE); //$NON-NLS-1$
		treeViewer.addSelectionChangedListener(lineAddAbove);
		lineAddBelow = new AddRowAction("unter diesem Element", ParaEditCompositeConstants.BELOW); //$NON-NLS-1$
		treeViewer.addSelectionChangedListener(lineAddBelow);
		lineRemove = new LineRemoveAction("dieses Element löschen"); //$NON-NLS-1$
		treeViewer.addSelectionChangedListener(lineRemove);
		arrayAddOneLast = new AddRowAction("am Ende des Feldes", ParaEditCompositeConstants.ONE_LAST); //$NON-NLS-1$
		treeViewer.addSelectionChangedListener(arrayAddOneLast);
		arrayAddOneFirst = new AddRowAction("am Anfang des Feldes", ParaEditCompositeConstants.ONE_FIRST); //$NON-NLS-1$
		treeViewer.addSelectionChangedListener(arrayAddOneFirst);
		arrayAddMoreLast = new AddRowAction("am Ende des Feldes", ParaEditCompositeConstants.MORE_LAST); //$NON-NLS-1$
		treeViewer.addSelectionChangedListener(arrayAddMoreLast);
		arrayAddMoreFirst = new AddRowAction("am Anfang des Feldes", ParaEditCompositeConstants.MORE_FIRST); //$NON-NLS-1$
		treeViewer.addSelectionChangedListener(arrayAddMoreFirst);
		arrayRemoveAll = new RemoveAllAction("alle Elemente löschen"); //$NON-NLS-1$
		treeViewer.addSelectionChangedListener(arrayRemoveAll);

		lineAdd = new MenuManager("Element einfügen"); //$NON-NLS-1$
		addOne = new MenuManager("ein Element einfügen"); //$NON-NLS-1$
		addMore = new MenuManager("mehrere Elemente einfügen"); //$NON-NLS-1$

		lineAdd.add(lineAddAbove);
		lineAdd.add(lineAddBelow);
		addOne.add(arrayAddOneFirst);
		addOne.add(arrayAddOneLast);
		addMore.add(arrayAddMoreFirst);
		addMore.add(arrayAddMoreLast);

		contextMenuManager = new MenuManager();
		contextMenuManager.setRemoveAllWhenShown(true);
		contextMenuManager.addMenuListener(new IMenuListener() {
			@Override
			public void menuAboutToShow(final IMenuManager mgr) {
				fillContextMenu(mgr);
			}
		});
		final Menu menu = contextMenuManager.createContextMenu(treeViewer.getControl());
		tree.setMenu(menu);
	}

	/**
	 * fills the context menus according to the selected tree-object.
	 *
	 * @param mgr
	 *            - the IMenuManager
	 */
	void fillContextMenu(final IMenuManager mgr) {
		mgr.add(new GroupMarker(IWorkbenchActionConstants.MB_ADDITIONS));
		final IStructuredSelection selection = (IStructuredSelection) treeViewer.getSelection();
		final Object selectedObject = selection.getFirstElement();
		if (!(selectedObject instanceof MyTreeObject)) {
			return;
		}
		MyTreeObject myTreeObject = (MyTreeObject) selectedObject;
		if (myTreeObject.getData() instanceof Data) {
			Data data = (Data) myTreeObject.getData();
			boolean filled = false;
			if (data.isArray()) {
				contextMenuManager.add(addOne);
				// contextMenuManager.add(addMore);
				contextMenuManager.add(arrayRemoveAll);
				filled = true;
			} else {
				if (myTreeObject.getRoot() == null) {
					return;
				}
				myTreeObject = myTreeObject.getRoot();
				data = (Data) myTreeObject.getData();
				if ((null != data) && data.isArray()) {
					contextMenuManager.add(lineAdd);
					contextMenuManager.add(lineRemove);
					filled = true;
				}
			}
			mgr.setVisible(filled);
		}
	}

	/**
	 * Berechnet die optimale Höhe des Controls basierend auf der Anzahl
	 * darzustellender Objekte.
	 *
	 * Der Faktor zwei wurde experimentell ermittelt (unter Windows). Der Summand
	 * zwei berücksichtigt Kopfzeile und Platz für die Scrollbar.
	 *
	 * @return die Höhe in Pixel.
	 */
	public int calculateOptimalHeight() {
		int heightHint = ParaEditCompositeConstants.MAX_HEIGHT;
		if (null != mediatorDataObject) {
			// Der Höhenhinweis kann erst berechnet werden, nachdem showData
			// aufgerufen wurde.
			final int systemFontHeight = treeViewer.getTree().getDisplay().getSystemFont().getFontData()[0].getHeight();
			final int numNodes = mediatorDataObject.getFlatList().size();
			heightHint = Math.min(ParaEditCompositeConstants.MAX_HEIGHT, (2 + numNodes) * 2 * systemFontHeight);
			heightHint = Math.max(ParaEditCompositeConstants.MIN_HEIGHT, heightHint);
			ParaEditComposite.DEBUG.finer("numNodes=" + numNodes + ", heightHint=" + heightHint);
		}
		return heightHint;
	}

	/**
	 * Die optimale Höhe des Controls bestimmen und anwenden.
	 */
	private void resizeControl() {
		GridData gd = (GridData) tree.getLayoutData();
		final int heightHint = calculateOptimalHeight();
		gd.heightHint = heightHint;
		gd = (GridData) sectionClient.getLayoutData();
		gd.heightHint = heightHint;
		SharedScrolledComposite f = null;
		Composite currentParent = sectionClient.getParent();
		while ((currentParent != null) && !(currentParent instanceof SharedScrolledComposite)) {
			currentParent = currentParent.getParent();
		}
		if (currentParent != null) {
			f = (SharedScrolledComposite) currentParent;
			f.reflow(true);
		} else {
			sectionClient.getParent().layout();
		}
	}

	/**
	 * initializes the editors after the table got initialized has to be uneditable
	 * in the text line.
	 */
	private void initEditors() {
		if (doubleEdit != null) {
			doubleEdit.dispose();
		}
		doubleEdit = new TextCellEditor(tree);
		((Text) doubleEdit.getControl()).addVerifyListener(new UzVerifyListener(UzVerifyListener.DOUBLE_MINUS));

		if (integerEdit != null) {
			integerEdit.dispose();
		}
		integerEdit = new TextCellEditor(tree);
		((Text) integerEdit.getControl()).addVerifyListener(new UzVerifyListener(UzVerifyListener.DOUBLE_MINUS));

		if (timeEdit != null) {
			timeEdit.dispose();
		}
		timeEdit = new TextCellEditor(tree);
		((Text) timeEdit.getControl()).addVerifyListener(new UzVerifyListener(UzVerifyListener.TIME));

		if (stringEdit != null) {
			stringEdit.dispose();
		}
		stringEdit = new TextCellEditor(tree);
		if (refEdit != null) {
			refEdit.dispose();
		}
		refEdit = new TextCellEditor(tree);

		if (stateEdit != null) {
			stateEdit.dispose();
		}
		stateEdit = new PureComboBoxSelector(tree);
	}

	/**
	 * initializes double editor components.
	 *
	 * @param data
	 *            - das zu editierende Objekt
	 */
	public void setDoubleEditor(final Data data) {
		String info = "DoubleAttributeType"; //$NON-NLS-1$
		if (data.isArray()) {
			info += " - Array"; //$NON-NLS-1$
		}
		initEditors();
		editors[1] = doubleEdit;
		if (!isReadOnly()) {
			treeViewer.setCellEditors(editors);
			treeViewer.setCellModifier(textModifier);
		}
		doAdvice(info, ParaEditCompositeConstants.MESSAGE);
	}

	/**
	 * initializes integer editor components.
	 *
	 * @param attType
	 *            - den Attributtyp
	 * @param data
	 *            - das zu editierende Objekt
	 */
	public void setIntegerEditor(final IntegerAttributeType attType, final Data data) {
		initEditors();
		if (!isReadOnly()) {
			editors[1] = integerEdit;
			treeViewer.setCellEditors(editors);
			treeViewer.setCellModifier(textModifier);
		}
		doAdvice(TableTreeLabelProvider.getIntegerInfo(attType, data), ParaEditCompositeConstants.MESSAGE);
	}

	/**
	 * initializes string editor components.
	 *
	 * @param attType
	 *            - den Attributtyp
	 * @param data
	 *            - das zu editierende Objekt
	 */
	public void setStringEditor(final StringAttributeType attType, final Data data) {
		initEditors();
		if (attType.isLengthLimited()) {
			((Text) stringEdit.getControl()).setTextLimit(attType.getMaxLength());
		}
		if (!isReadOnly()) {
			editors[1] = null; // stringEdit;
			treeViewer.setCellEditors(editors);
			treeViewer.setCellModifier(textModifier);
		}
		doAdvice(TableTreeLabelProvider.getStringInfo(attType, data), ParaEditCompositeConstants.MESSAGE);
	}

	/**
	 * initializes double editor components.
	 *
	 * @param attType
	 *            - den Attributtyp
	 * @param data
	 *            - das zu editierende Objekt
	 */
	public void setTimeEditor(final TimeAttributeType attType, final Data data) {
		String info = TableTreeLabelProvider.getTimeInfo(attType, data);
		initEditors();
		info += " - Doppelklick zum Öffnen des Eingabedialogs";
		if (!isReadOnly()) {
			editors[1] = null; // timeEdit;
			treeViewer.setCellEditors(editors);
			treeViewer.setCellModifier(textModifier);
		}
		doAdvice(info, ParaEditCompositeConstants.MESSAGE);
	}

	/**
	 * initializes reference editor components.
	 *
	 * @param attType
	 *            beschreibt den Typ des zu referenzierenden Objekts.
	 * @param data
	 *            - das zu editierende Objekt
	 */
	public void setReferenceEditor(final ReferenceAttributeType attType, final Data data) {
		String info = "ReferenceAttributeType "; //$NON-NLS-1$

		if (data.isArray()) {
			info += "- Array"; //$NON-NLS-1$
		} else {
			info += "- Doppelklick zum Öffnen des Auswahldialogs"; //$NON-NLS-1$
		}

		initEditors();

		if (null == attType.getReferencedObjectType()) {
			if (!isReadOnly()) {
				editors[1] = null;
				treeViewer.setCellEditors(editors);
				treeViewer.setCellModifier(textModifier);
			}
			doAdvice(info, ParaEditCompositeConstants.MESSAGE);
			return;
		}
		final List<SystemObject> objList = new ArrayList<>();
		objList.addAll(attType.getReferencedObjectType().getElements());
		Collections.sort(objList, systemObjectComparator);

		final String[] values = new String[objList.size()];
		int i = 0;
		for (final SystemObject obj : objList) {
			values[i] = obj.toString();
			i++;
		}

		stateEdit.setItems(values);
		if (!isReadOnly()) {
			editors[1] = null;
			treeViewer.setCellEditors(editors);
			treeViewer.setCellModifier(referenceModifier);
		}
		doAdvice(info, ParaEditCompositeConstants.MESSAGE);
	}

	/**
	 * initializes state editor components.
	 *
	 * @param attType
	 *            - den Attributtyp
	 * @param data
	 *            - das zu editierende Objekt
	 */
	public void setStateEditor(final IntegerAttributeType attType, final Data data) {
		String info = "IntegerAttributeType"; //$NON-NLS-1$
		int i = 0;

		initEditors();

		final List<String> types = new ArrayList<>();
		for (final IntegerValueState state : attType.getStates()) {
			i++;
			types.add(state.getName());
		}
		if (data.isArray()) {
			info += " - Array"; //$NON-NLS-1$
		}
		info += " - mit " + i + " diskreten Zuständen (siehe Auswahlbox)"; //$NON-NLS-1$ //$NON-NLS-2$

		final String[] values = new String[types.size()];
		i = 0;
		for (final String str : types) {
			values[i] = str;
			i++;
		}
		stateEdit.setItems(values);
		if (!isReadOnly()) {
			editors[1] = stateEdit;
			treeViewer.setCellEditors(editors);
			treeViewer.setCellModifier(stateModifier);
		}
		doAdvice(info, ParaEditCompositeConstants.MESSAGE);
	}

	/**
	 * initializes the Layout.
	 */
	public void createPartControl() {

		sectionClient.setLayout(new GridLayout(1, false));

		headLine = toolkit.createLabel(sectionClient, "Benutzer:\nUrsache:\nVeranlasser:");
		final GridData headLineGridData = new GridData(SWT.LEFT, SWT.TOP, true, false);
		headLineGridData.widthHint = 1050;
		headLine.setLayoutData(headLineGridData);
		headLine.setBackground(Display.getDefault().getSystemColor(SWT.COLOR_WHITE));
		headLine.addMouseListener(new HeadLineMouseAdapter(this));
		if (additionalText != null) {
			additionalTextLabel = toolkit.createLabel(sectionClient, additionalText.getText());
			additionalTextLabel.setLayoutData(
					new GridData(GridData.HORIZONTAL_ALIGN_FILL | GridData.GRAB_HORIZONTAL | GridData.BEGINNING));
			additionalTextLabel.setText(additionalText.getText());
			additionalTextLabel.setBackground(additionalText.getBackgroundColor());
			additionalTextLabel.setForeground(additionalText.getForegroundColor());
			additionalTextLabel.setFont(additionalText.getFont());
		}

		tree = toolkit.createTree(sectionClient, SWT.BORDER | SWT.FULL_SELECTION);
		tree.setLayout(new FillLayout());
		final GridData layoutData = new GridData(SWT.FILL, SWT.TOP, true, false);
		layoutData.heightHint = calculateOptimalHeight();
		tree.setLayoutData(layoutData);
		treeViewer = new TreeViewer(tree);
		tree.setHeaderVisible(true);
		tree.setLinesVisible(true);
		tree.setVisible(true);
		treeViewer.addDoubleClickListener(new ParaEditCompositeDoubleClickListener(this));

		if (null == ParaEditComposite.alertColor) {
			ParaEditComposite.alertColor = new Color(sectionClient.getDisplay(), 250, 100, 100);
		}
		if (null == ParaEditComposite.messageColor) {
			ParaEditComposite.messageColor = new Color(sectionClient.getDisplay(), 200, 200, 230);
		}

		infoLine = toolkit.createLabel(sectionClient, "");
		infoLine.setLayoutData(new GridData(GridData.HORIZONTAL_ALIGN_FILL | GridData.GRAB_HORIZONTAL));
		doAdvice("Informationsleiste", ParaEditCompositeConstants.MESSAGE); //$NON-NLS-1$

		TreeColumn col = new TreeColumn(tree, SWT.NONE);
		col.setText(properties[0]);
		col.setWidth(150);
		col = new TreeColumn(tree, SWT.NONE);
		col.setText(properties[1]);
		col.setWidth(300);
		if (parameters.length > 1) {
			col = new TreeColumn(tree, SWT.NONE);
			col.setText(properties[2]);
			col.setWidth(300);
		}
		col = new TreeColumn(tree, SWT.NONE);
		col.setText("Wertebereich");
		col.setWidth(600);

		initEditors();
		treeViewer.setColumnProperties(properties);
		treeState = new ExpandState(treeViewer, this);

		if (!isReadOnly()) {
			createContextMenu();
		}

		currentTL = new TableSelectionListener();
		treeViewer.addSelectionChangedListener(currentTL);
	}

	/**
	 * Holt die Urlasserinformationen aus der {@link #workingCopy}, wobei nach
	 * Benutzer und Ursache jeweils ein Zeilenumbruch eingefügt wird.
	 *
	 * @param lang
	 *            Komplette Ursache 1:1 auslesen. Wenn nicht, wird der Ursachentext
	 *            auf die erste Zeile beschränkt und dann noch auf
	 *            {@value #MAX_LEN_URSACHE_TEXT} Zeichen gekürzt.
	 *
	 * @return der Text
	 */
	String getUrlasserInfosAsSingleString(final boolean lang) {
		String benutzerStr = Zeichenketten.PLUGIN_PARAM_BEZEICHNER_UNBEKANNT;
		String ursacheLang = Zeichenketten.PLUGIN_PARAM_BEZEICHNER_UNBEKANNT;
		String ursacheBearbeitet = Zeichenketten.PLUGIN_PARAM_BEZEICHNER_UNBEKANNT;
		String veranlasser = Zeichenketten.PLUGIN_PARAM_BEZEICHNER_UNBEKANNT;
		if (null != workingCopy) {
			final Data atlUrlasserItem = workingCopy.getData().getItem("Urlasser");
			final SystemObject o = atlUrlasserItem.getReferenceValue("BenutzerReferenz").getSystemObject();
			if (null == o) {
				benutzerStr = "undefiniert";
			} else {
				benutzerStr = o.toString();
			}
			veranlasser = atlUrlasserItem.getTextValue("Veranlasser").getText();
			ursacheLang = atlUrlasserItem.getTextValue("Ursache").getText();
			int firstNl = ursacheLang.indexOf('\n');
			if (-1 == firstNl) {
				firstNl = ursacheLang.indexOf('\r');
			}
			boolean needEllipsis = false;
			if (!lang && (-1 != firstNl)) {
				ursacheBearbeitet = ursacheLang.substring(0, firstNl - 1);
				needEllipsis = true;
			} else {
				ursacheBearbeitet = ursacheLang;
			}
			if (!lang && (ursacheBearbeitet.length() > ParaEditComposite.MAX_LEN_URSACHE_TEXT)) {
				ursacheBearbeitet = ursacheBearbeitet.substring(0, ParaEditComposite.MAX_LEN_URSACHE_TEXT);
				needEllipsis = true;
			}
			if (needEllipsis) {
				ursacheBearbeitet += "...";
			}
		}
		final StringBuilder b = new StringBuilder(100);
		b.append("Benutzer: ");
		b.append(benutzerStr);
		b.append("\nUrsache: ");
		b.append(ursacheBearbeitet);
		b.append("\nVeranlasser: ");
		b.append(veranlasser);
		return b.toString();
	}

	/**
	 * Aktualisiert die {@link #headLine} mit der AtlUrlasser aus der
	 * {@link #workingCopy}. Siehe Mantis#1594.
	 */
	private void updateUrlasserInfos() {
		String ursacheLang = Zeichenketten.PLUGIN_PARAM_BEZEICHNER_UNBEKANNT;
		if (null != workingCopy) {
			final Data atlUrlasserItem = workingCopy.getData().getItem("Urlasser");
			ursacheLang = atlUrlasserItem.getTextValue("Ursache").getText();
		}
		headLine.setText(getUrlasserInfosAsSingleString(false));
		headLine.setToolTipText("Ursache: " + ursacheLang);
	}

	/**
	 * displays an AttributeGroup in the Table.
	 *
	 * @param parameterToShow
	 *            the Parameter Objects whose data is to be displayed
	 * @param updateWorkingCopy
	 *            true - Die lokale Arbeitskopie des DAF-Data-Objektes
	 *            aktualisieren. Wenn wir Read-Only sind, passiert das sowieso, das
	 *            Flag hat nur eine Wirkung, wenn wir editierbar sind.
	 */
	public final void showData(final Parameter[] parameterToShow, final boolean updateWorkingCopy) {
		treeViewer.setColumnProperties(properties);
		if (!(treeViewer.getLabelProvider() instanceof TableTreeLabelProvider)) {
			treeViewer.setLabelProvider(new TableTreeLabelProvider(this));
		}
		if (!(treeViewer.getContentProvider() instanceof TableTreeContentProvider)) {
			treeViewer.setContentProvider(new TableTreeContentProvider());
		}
		if (isReadOnly() || updateWorkingCopy) {
			final Data data;
			if (parameterToShow[0].getData() == null) {
				final ClientDavInterface davIf = RahmenwerkService.getService().getObjektFactory().getDav();
				data = davIf.createData(parameters[0].getAtg());
			} else {
				data = parameterToShow[0].getData();
			}
			workingCopy = new DataWithTime(data, System.currentTimeMillis());
			mediatorDataObject = new MyDataObject(parameterToShow[0]);
			dataObject = (MyTreeObject) mediatorDataObject.getContent();
		} else {
			final DataWithTime saveData = parameterToShow[0].getDataWithTime();
			parameterToShow[0].setDataWithTime(workingCopy);
			mediatorDataObject = new MyDataObject(parameterToShow[0]);
			dataObject = (MyTreeObject) mediatorDataObject.getContent();
			parameterToShow[0].setDataWithTime(saveData);
		}
		if (updateWorkingCopy) {
			dirty = false;
		}
		int shadowLoop;
		for (shadowLoop = 1; shadowLoop < parameterToShow.length; ++shadowLoop) {
			shadowDataObjects[shadowLoop - 1] = (MyTreeObject) (new MyDataObject(parameterToShow[shadowLoop]))
					.getContent();
		}
		updateUrlasserInfos();
		treeViewer.setInput(dataObject);
		resizeControl();
	}

	/**
	 * displays infos in the info-line below the table.
	 *
	 * @param info
	 *            - the String to be displayed
	 * @param state
	 *            - either ALERT or MESSAGE
	 */
	public void doAdvice(final String info, final int state) {
		infoLine.setBackground(state == ParaEditCompositeConstants.ALERT ? ParaEditComposite.alertColor
				: ParaEditComposite.messageColor);
		infoLine.setText(" " + info);
	}

	/**
	 * choses the proper editor for the given data object.
	 *
	 * @author Thierfelder
	 *
	 */
	public class TableSelectionListener implements ISelectionChangedListener {

		/**
		 * Wird beim Klick in ein Editorfeld aufgerufen und wählt den passenden Editor
		 * aus.
		 *
		 * @param event
		 *            - Das Aufrufereignis
		 * @see org.eclipse.jface.viewers.ISelectionChangedListener#selectionChanged(org.eclipse.jface.viewers.SelectionChangedEvent)
		 */
		@Override
		public void selectionChanged(final SelectionChangedEvent event) {
			final Object elem = ((StructuredSelection) (event.getSelection())).getFirstElement();
			treeViewer.cancelEditing();

			if (elem == null) {
				return;
			}
			if (elem instanceof MyTreeObject) { // what else?
				final MyTreeObject myTreeObject = (MyTreeObject) elem;
				if (myTreeObject.getData() instanceof Data) {
					final Data data = (Data) myTreeObject.getData();
					final AttributeType attType = data.getAttributeType();
					if (attType instanceof StringAttributeType) {
						setStringEditor((StringAttributeType) attType, data);
						cleanEditors(myTreeObject);
					} else if (attType instanceof IntegerAttributeType) {
						final IntegerValueRange range = ((IntegerAttributeType) attType).getRange();
						if (range != null) {
							setIntegerEditor((IntegerAttributeType) attType, data);
							cleanEditors(myTreeObject);
						} else if (((IntegerAttributeType) attType).getStates() != null) {
							setStateEditor((IntegerAttributeType) attType, data);
							cleanEditors(myTreeObject);
						}
					} else if (attType instanceof DoubleAttributeType) {
						setDoubleEditor(data);
						cleanEditors(myTreeObject);
					} else if (attType instanceof AttributeListDefinition) {
						doAdvice("AttributeListDefinition", ParaEditCompositeConstants.MESSAGE); //$NON-NLS-1$
					} else if (attType instanceof TimeAttributeType) {
						setTimeEditor((TimeAttributeType) attType, data);
						cleanEditors(myTreeObject);
					} else if (attType instanceof ReferenceAttributeType) {
						setReferenceEditor((ReferenceAttributeType) attType, data);
						cleanEditors(myTreeObject);
					} else {
						doAdvice("Unerwarteter Fehler in TableSelectionListener(1)", ParaEditCompositeConstants.ALERT); //$NON-NLS-1$
					}
				}
			} else {
				doAdvice("???Unerwarteter Fehler in TableSelectionListener(3)", ParaEditCompositeConstants.ALERT); //$NON-NLS-1$
			}
		}

		/**
		 * Löscht alle festgesetzten Editoren für diesen Editor.
		 *
		 * @param myTreeObject
		 *            - der Inhalt des Editors
		 */
		private void cleanEditors(final MyTreeObject myTreeObject) {
			treeViewer.removeSelectionChangedListener(currentTL);
			treeViewer.editElement(myTreeObject, 1);
			currentTL = new TableSelectionListener();
			treeViewer.addSelectionChangedListener(currentTL);
		}
	}

	/**
	 * modifier concerning only with combobox-editing.
	 *
	 * @author Thierfelder
	 */
	public class StateModifier implements ICellModifier {

		/**
		 * Erfragt, ob eine bestimmtes Element modifiziert werden darf.
		 *
		 * @param object
		 *            - das Objekt
		 * @param property
		 *            - der Spaltentext in der Kopfzeile
		 * @return Element kann modifiziert werden ?
		 * @see org.eclipse.jface.viewers.ICellModifier#canModify(java.lang.Object,
		 *      java.lang.String)
		 */
		@Override
		public boolean canModify(final Object object, final String property) {
			if (property.equals(properties[0])) {
				return false;
			}

			final MyTreeObject myTreeObject = (MyTreeObject) object;
			final Data data = (Data) myTreeObject.getData();

			return data.isPlain();
		}

		/**
		 * Erfragt den Wert eines bestimmten Elements.
		 *
		 * @param element
		 *            - das Element
		 * @param property
		 *            - der Spaltentext in der Kopfzeile
		 * @return den Wert
		 * @see org.eclipse.jface.viewers.ICellModifier#getValue(java.lang.Object,
		 *      java.lang.String)
		 */
		@Override
		public Object getValue(final Object element, final String property) {
			final MyTreeObject myTreeObject = (MyTreeObject) element;
			final Data data = (Data) myTreeObject.getData();
			String valueAsText = Constants.EMPTY_STRING;

			if (data.getAttributeType() instanceof StringAttributeType) {
				final String text = data.asTextValue().getText();
				if (null != text) {
					valueAsText = text;
				}
			} else if (data.getAttributeType() instanceof IntegerAttributeType) {
				Integer valueAsInt = 0;
				if (data.asUnscaledValue().getState() != null) {
					final IntegerValueState state = data.asUnscaledValue().getState();
					final IntegerAttributeType attType = (IntegerAttributeType) (data.getAttributeType());

					int i = 0;
					for (final IntegerValueState listState : attType.getStates()) {
						if (listState.getValue() == state.getValue()) {
							valueAsInt = i;
						}
						i++;
					}
				}
				return valueAsInt;
			}
			return valueAsText;
		}

		/**
		 * Verändert den Wert eines Elements abhängig von seinem (DV-)Typ.
		 *
		 * @param element
		 *            - das Element
		 * @param property
		 *            - der Spaltentext in der Kopfzeile
		 * @param value
		 *            - der neue Wert dieses Elements (der Wert, der in das Element
		 *            eingetragen werden soll)
		 * @see org.eclipse.jface.viewers.ICellModifier#modify(java.lang.Object,
		 *      java.lang.String, java.lang.Object)
		 */
		@Override
		public void modify(final Object element, final String property, final Object value) {
			final TreeItem tti = (TreeItem) element;
			final MyTreeObject myTreeObject = (MyTreeObject) tti.getData();
			final Data data = (Data) myTreeObject.getData();
			final IntegerAttributeType attType = (IntegerAttributeType) (data.getAttributeType());
			List<IntegerValueState> list = null;

			list = attType.getStates();
			final IntegerValueState oldValue = data.asUnscaledValue().getState();
			final long oldValueLong;
			if (null == oldValue) {
				oldValueLong = 0L;
			} else {
				oldValueLong = oldValue.getValue();
			}
			final String newState = list.get(((Integer) value).intValue()).getName();
			for (final IntegerValueState ivs : attType.getStates()) {
				if ((ivs.getName()).equals(newState)) {
					data.asUnscaledValue().setState(ivs);
					if (container != null) {
						container.valueModified(ParaEditComposite.this, myTreeObject, oldValueLong, ivs.getValue(),
								data.getAttributeType());
					}
					treeState.push();
					treeViewer.refresh(myTreeObject);
					treeState.pop();
					setDirty();
					return;
				}
			}
		}
	}

	/**
	 * Modifier für Referenzen.
	 *
	 * @author hoesel
	 *
	 */
	public class ReferenceModifier implements ICellModifier {

		@Override
		public boolean canModify(final Object element, final String property) {
			if (property.equals(properties[0])) {
				return false;
			}

			final MyTreeObject myTreeObject = (MyTreeObject) element;
			final Data data = (Data) myTreeObject.getData();

			return data.isPlain();
		}

		@Override
		public Object getValue(final Object element, final String property) {
			final MyTreeObject myTreeObject = (MyTreeObject) element;
			final Data data = (Data) myTreeObject.getData();

			try {
				final ReferenceAttributeType attType = (ReferenceAttributeType) data.getAttributeType();
				final String valueText = data.asReferenceValue().getValueText();
				if ("<<null>>".equals(valueText) /* DaV2 */
						|| "undefiniert".equals(valueText) /* DaV3 */) {
					return Integer.valueOf(0);
				}
				final SystemObject obj = data.asReferenceValue().getSystemObject();
				if (null == obj) {
					// Hier sollten wir nicht landen
					return Integer.valueOf(0);
				}

				int i = 0;
				for (final SystemObject listState : attType.getReferencedObjectType().getElements()) {
					if (listState.equals(obj)) {
						return Integer.valueOf(i);
					}
					i++;
				}
			} catch (final NullPointerException e) {
				doAdvice("Unerwarteter Fehler", ParaEditCompositeConstants.ALERT); //$NON-NLS-1$
				Logger.getLogger(ParaEditComposite.class.getName()).log(Level.INFO, "", e); //$NON-NLS-1$
			}

			// Hier sollten wir nicht landen
			return Integer.valueOf(0);
		}

		@Override
		public void modify(final Object element, final String property, final Object value) {
			final TreeItem tti = (TreeItem) element;
			final MyTreeObject myTreeObject = (MyTreeObject) tti.getData();
			final Data data = (Data) myTreeObject.getData();
			final ReferenceAttributeType attType = (ReferenceAttributeType) (data.getAttributeType());
			final List<SystemObject> list = new ArrayList<>();

			list.addAll(attType.getReferencedObjectType().getElements());
			Collections.sort(list, systemObjectComparator);
			final SystemObject newValue = list.get((Integer) value);
			final SystemObject oldValue = data.asReferenceValue().getSystemObject();
			if (oldValue.equals(newValue)) {
				return;
			}
			data.asReferenceValue().setSystemObject(newValue);
			if (container != null) {
				container.valueModified(ParaEditComposite.this, myTreeObject, oldValue, newValue,
						data.getAttributeType());
			}
			treeState.push();
			treeViewer.refresh(myTreeObject);
			treeState.pop();
			setDirty();
			return;
		}

	}

	/**
	 * manages the addition and killing of a row in a n array object.
	 *
	 * @author Thierfelder
	 *
	 */
	private class AddRowAction extends TableSelectionAction {

		/**
		 * Spezifikation der Operation.
		 */
		private int op = -1;

		/**
		 * Standardkonstruktor.
		 *
		 * @param str
		 *            - Name der Aktion
		 * @param op
		 *            - Spezifikation der Operation
		 */
		public AddRowAction(final String str, final int op) {
			super(str);
			this.op = op;
		}

		/**
		 * Diese Methode wird beim Aufruf der Aktion Löschen/Einfügen aufgerufen.
		 */
		@Override
		public void run() {

			// MessageDialog outOfBoundAlert = null;
			MyTreeObject parent = null;
			Data root = null;
			if (getSelectedObject() == null) {
				doAdvice("Fehler: kein Element ausgewählt", ParaEditCompositeConstants.ALERT); //$NON-NLS-1$
				return;
			}
			final MyTreeObject myTreeObject = (MyTreeObject) getSelectedObject();
			Data data = (Data) myTreeObject.getData();
			if ((op != ParaEditCompositeConstants.ABOVE) && (op != ParaEditCompositeConstants.BELOW)) {
				ParaEditComposite.DEBUG.info("Weder ABOVE noch BELOW!");
			} else {
				parent = ((MyTreeObject) getSelectedObject()).getRoot();
				root = (Data) parent.getData();
			}

			switch (op) {
			case ParaEditCompositeConstants.ABOVE:
				data = (new ArrayObjectTransformer(root)).addLineAbove(getLocation(myTreeObject));
				newItems.add(getLocation(myTreeObject));
				if (data == null) {
					doAdvice("Fehler: Überprüfen Sie, ob Ihre Eingabe den Wertebereich verletzt.", //$NON-NLS-1$
							ParaEditCompositeConstants.ALERT);
				} else {
					myTreeObject.setData(data);
				}
				break;
			case ParaEditCompositeConstants.BELOW:
				data = (new ArrayObjectTransformer(root)).addLineBelow(getLocation(myTreeObject));
				newItems.add(getLocation(myTreeObject));
				if (data == null) {
					doAdvice("Fehler: Überprüfen Sie, ob Ihre Eingabe den Wertebereich verletzt.", //$NON-NLS-1$
							ParaEditCompositeConstants.ALERT);
				} else {
					myTreeObject.setData(data); // new
				}
				break;
			case ParaEditCompositeConstants.ONE_LAST:
				data = (new ArrayObjectTransformer(data)).addLinesEnd(1);
				if (data == null) {
					doAdvice("Fehler: Überprüfen Sie, ob Ihre Eingabe den Wertebereich verletzt.", //$NON-NLS-1$
							ParaEditCompositeConstants.ALERT);
				} else {
					myTreeObject.setData(data); // new
				}
				break;
			case ParaEditCompositeConstants.ONE_FIRST:
				data = (new ArrayObjectTransformer(data)).addLinesBegin(1);
				if (data == null) {
					doAdvice("Fehler: Überprüfen Sie, ob Ihre Eingabe den Wertebereich verletzt.", //$NON-NLS-1$
							ParaEditCompositeConstants.ALERT);
				} else {
					myTreeObject.setData(data); // new
				}
				break;
			// case MORE_LAST:
			// if ((id != null) && (id.open() == Window.OK)) {
			// size = Integer.parseInt(id.getValue());
			// data = (new
			// ArrayObjectTransformer(data)).addLinesEnd(size);
			// if (data == null) {
			// doAdvice("Fehler: Überprüfen Sie, ob Ihre Eingabe den
			// Wertebereich verletzt.", ParaEditComposite.ALERT);
			// //$NON-NLS-1$
			// } else
			// tree.setData(data); // new
			// }
			// data = (new ArrayObjectTransformer(data)).addLinesEnd(2);
			// break;
			// case MORE_FIRST:
			// if ((id != null) && (id.open() == Window.OK)) {
			// size = Integer.parseInt(id.getValue());
			// data = (new
			// ArrayObjectTransformer(data)).addLinesBegin(size);
			// if (data == null) {
			// doAdvice("Fehler: Überprüfen Sie, ob Ihre Eingabe den
			// Wertebereich verletzt.", ParaEditComposite.ALERT);
			// //$NON-NLS-1$
			// } else
			// tree.setData(data); // new
			// }
			// data = (new
			// ArrayObjectTransformer(data)).addLinesBegin(2);
			// break;
			default:
				doAdvice("Fehler: unerwarteter Fehler", ParaEditCompositeConstants.ALERT); //$NON-NLS-1$
			}
			treeState.push();
			// HINWEIS: Wir benötigen eine Update-Methode für die
			// bestehenden Trees, um den OriginalValue und damit den
			// Dirty-State der untergeordneten Elemente erhalten zu können
			showData(parameters, false);
			treeState.pop();
			setDirty();
			/*
			 * HINWEIS Den Container benachrichtigen, dass es neue Elemente gibt, damit
			 * diese in anderen ParaEditComposites ebenfalls angelegt werden können, wenn
			 * möglich.
			 */

		}
	}

	/**
	 * removes all lines from an array object.
	 *
	 * @author Thierfelder
	 *
	 */
	private class RemoveAllAction extends TableSelectionAction {

		/**
		 * Standardkonstruktor.
		 *
		 * @param alert
		 *            - der Name der Aktion
		 */
		public RemoveAllAction(final String alert) {
			super(alert);
		}

		/**
		 * Sammelt alle Kinder eines übergebenen Objektes in eine flache Liste auf.
		 *
		 * @param collectorList
		 *            die flache Liste.
		 * @param myTreeObject
		 *            Startobjekt
		 */
		private void collectChildrenRekursiv(final List<MyTreeObject> collectorList, final MyTreeObject myTreeObject) {
			collectorList.addAll(myTreeObject.getChildren());
			for (final MyTreeObject o : myTreeObject.getChildren()) {
				collectChildrenRekursiv(collectorList, o);
			}
		}

		/**
		 * Diese Methode wird aufgerufen, wenn die Aktion "Lösche alle Elemente" für
		 * eine Array-Objekt aufgerufen wird.
		 */
		@Override
		public void run() {
			if (getSelectedObject() == null) {
				doAdvice("Fehler: kein Element ausgewählt", ParaEditCompositeConstants.ALERT); //$NON-NLS-1$
				return;
			}
			final MessageBox alert = new MessageBox(treeViewer.getControl().getShell(),
					SWT.ICON_WARNING | SWT.NO | SWT.YES);
			alert.setText("Warnung"); //$NON-NLS-1$
			alert.setMessage("Möchten Sie wirklich alle Elemente dieses Arrays löschen?"); //$NON-NLS-1$
			if (SWT.NO != alert.open()) {
				final MyTreeObject myTreeObject = (MyTreeObject) getSelectedObject();
				Data data = (Data) myTreeObject.getData();

				data = (new ArrayObjectTransformer(data)).removeAllLines();
				clearMyTreeObjectsInMediatorObject(myTreeObject);
				myTreeObject.getChildren().clear();
				setDirty();
				// Die aktuelle Implementierung von ArrayObjectTransformer
				// liefert stets dasselbe Objekt zurück, insofern wäre die
				// folgende Zeile nicht nötig. Dies ist aber laut Dokumentation
				// nicht garantiert, daher die explizite Zuweisung.
				myTreeObject.setData(data);
				treeState.push();
				treeViewer.refresh();
				treeState.pop();
				resizeControl();
			}
		}

		/**
		 * Löscht alle MyTreeObjects im Mediatorobjekt, die Kinder vom soeben geclearten
		 * Array waren, sowie deren Kinder. Zuerst sammeln wir alle auf, danach löschen
		 * wir RÜCKWÄRTS, weil sonst die Indices nach dem ersten löschen nicht mehr
		 * stimmen.
		 *
		 * Das Sortieren der Indices-Liste ist eigentlich überflüssig, weil die
		 * entfernten Kinder direkt aufsteigend in der flachen Liste folgen.
		 *
		 * @param myTreeObject
		 *            der Array-Knoten, von dem alle Kinder gelöscht werden.
		 */
		private void clearMyTreeObjectsInMediatorObject(final MyTreeObject myTreeObject) {
			final List<MyTreeObject> childrenToBeRemoved = new ArrayList<>();
			collectChildrenRekursiv(childrenToBeRemoved, myTreeObject);
			final List<Integer> childrenSindices = new ArrayList<>(childrenToBeRemoved.size());
			int loop;
			for (loop = 0; loop < mediatorDataObject.getFlatList().size(); ++loop) {
				final MyTreeObject flatObject = mediatorDataObject.getFlatList().get(loop);
				boolean found = false;
				for (final MyTreeObject candidate : childrenToBeRemoved) {
					if (flatObject == candidate) {
						found = true;
						break;
					}
				}
				if (found) {
					childrenSindices.add(loop);
				}
			}
			Collections.sort(childrenSindices);
			int removerLoop;
			for (removerLoop = childrenSindices.size() - 1; removerLoop >= 0; --removerLoop) {
				final int currentlyRemovedIndex = childrenSindices.get(removerLoop);
				mediatorDataObject.getFlatList().remove(currentlyRemovedIndex);
				mediatorDataObject.getEbeneList().remove(currentlyRemovedIndex);
			}
		}
	}

	/**
	 * removes one line from an array object.
	 *
	 * @author Thierfelder
	 */
	private class LineRemoveAction extends TableSelectionAction {

		/**
		 * Standardkonstruktor.
		 *
		 * @param alert
		 *            - der Name der Aktion
		 */
		public LineRemoveAction(final String alert) {
			super(alert);
		}

		/**
		 * Wird aufgerufen, wenn ein Element eines Array-Objektes gelöscht werden soll.
		 */
		@Override
		public void run() {
			if (getSelectedObject() == null) {
				doAdvice("Fehler: kein Element ausgewählt", ParaEditCompositeConstants.ALERT); //$NON-NLS-1$
				return;
			}
			final MessageBox alert = new MessageBox(treeViewer.getControl().getShell(),
					SWT.ICON_WARNING | SWT.NO | SWT.YES);
			alert.setText("Warnung"); //$NON-NLS-1$
			alert.setMessage("Möchten Sie dieses Element wirklich löschen?"); //$NON-NLS-1$
			if (SWT.NO != alert.open()) {
				final MyTreeObject myTreeObject = (MyTreeObject) getSelectedObject();
				// Data data = (Data)tree.getData();
				final MyTreeObject myTreeObjectRoot = myTreeObject.getRoot();
				final Data parent = (Data) myTreeObjectRoot.getData();
				final int selectedChildIdx = getLocation(myTreeObject);
				new ArrayObjectTransformer(parent).removeSingleLine(selectedChildIdx);
				final int lastChildIdx = myTreeObjectRoot.getChildren().size() - 1;
				myTreeObjectRoot.getChildren().remove(lastChildIdx);
				treeState.push();
				// HINWEIS: Wir benötigen eine Update-Methode für die
				// bestehenden Trees, um den OriginalValue und damit den
				// Dirty-State der untergeordneten Elemente erhalten zu können
				showData(parameters, false);
				treeState.pop();
				setDirty();
			}
		}
	}

	/**
	 * Wir rufen einfach refreshViewer() auf. Das hat den Effekt, dass alle Werte
	 * aus #parameters[0]#getData() ausgelesen und in den Viewer gemalt werden. Man
	 * kann also mit getEditedParameter sich den DaV- Datensatz holen, einzelne
	 * Attribute verändern und dann durch Aufruf dieser Methode ein Refresh
	 * erzwingen. Dabei wird auch das Dirty-Flag auf true gesetzt.
	 *
	 * Braucht man, wenn dieses Composite nicht das einzige in der Wizardpage ist,
	 * sondern noch andere Bedienelemente neben uns liegen, die Werte im
	 * {@link Data}-Objekt direkt verändern wollen.
	 *
	 * (Siehe auch Stralsund, UFD-Parametrierung)
	 *
	 * @see org.eclipse.swt.widgets.Control#update()
	 */
	public void fireExternalModificationEvent() {
		dirty = true;
		refreshViewer();
	}

	/**
	 * Aktualisieren des gesamten Viewers über eine {@link Runnable}. Die Methode
	 * kann also von jedem Thread aus benutzt werden.
	 */
	private void refreshViewer() {
		if (!sectionClient.getDisplay().isDisposed()) {
			sectionClient.getDisplay().asyncExec(new Runnable() {

				@Override
				public void run() {
					if (!treeViewer.getControl().isDisposed()) {
						treeViewer.refresh();
					}

				}
			});
		}
	}

	/**
	 * Dispose-Methode analog zum SWT bzw. zu den FormToolkits, muss aber explizit
	 * aufgerufen werden!
	 */
	public void dispose() {
		if (additionalText != null) {
			additionalText.dispose();
		}
	}

	/**
	 * Bestimmt, ob mindestens ein Wert bereits editiert wurde.
	 *
	 * @return true - Mindestens ein Wert wurde verändert false - alles ist noch so
	 *         wie beim Erzeugen
	 */
	public boolean isDirty() {
		return dirty;
	}

	/**
	 * Markiert das Composite als verändert. Dies beinhaltet folgende Aktionen:
	 * <ul>
	 * <li>Dirty-Flag setzen</li>
	 * <li>Container benachrichtigen</li>
	 * <li>Formpart benachrichtigen</li>
	 * </ul>
	 * Falls das Composite bereits verändert war, passiert nichts.
	 */
	private void setDirty() {
		if (!dirty) {
			dirty = true;
			if (container != null) {
				container.dirtyStateChanged(this, dirty);
			}
			if (formPart != null) {
				formPart.markDirty();
			}
		}
	}

	/**
	 * Markiert das Composite als gespeichert. Dies beinhaltet folgende Aktionen:
	 * <ul>
	 * <li>Dirty-Flag zurücksetzen</li>
	 * <li>Container benachrichtigen</li>
	 * <li>Die aktuellen Werte als die Originalwerte übernehmen</li>
	 * <li>Sollparameter aktualisieren (nur bei dreispaltiger Version)</li>
	 * <li>Viewer aktualisieren</li>
	 * </ul>
	 * Falls das Composite gar nicht dirty war, passiert nichts.
	 */
	public void markSaved() {
		if (dirty) {
			dirty = false;
			if (container != null) {
				container.dirtyStateChanged(this, dirty);
			}
			dataObject.commit();
			updateSollParameter();
			updateUrlasserInfos();
			refreshViewer();
		}
	}

	/**
	 * Setzt den FormPart, in dem wir uns befinden.
	 *
	 * @param formPart
	 *            der FormPart
	 */
	public void setFormPart(final AbstractFormPart formPart) {
		this.formPart = formPart;
	}

	/**
	 * Liefert eine versendbare Kopie des gerade bearbeiteten Parameters mit dem
	 * aktuellsten Datensatz.
	 *
	 * @return der Parameter
	 */
	public Parameter getEditedParameter() {
		return new Parameter(patameterInfos[0], workingCopy);
	}

	/**
	 * Hilfsmethode für die Rekursion zum Finden eines Objektes.
	 *
	 * @param myTreeObject
	 *            Wurzelobjekt für diese Ebene der Rekursion.
	 * @param leaves
	 *            ein globales Feld, in dem alle Blätter eingetragen werden
	 */
	private void findAndAddMyTreeItemLeaves(final MyTreeObject myTreeObject, final List<MyTreeObject> leaves) {
		for (final MyTreeObject child : myTreeObject.getChildren()) {
			if (child.hasChildren()) {
				findAndAddMyTreeItemLeaves(child, leaves);
			} else {
				leaves.add(child);
			}
		}
		return;
	}

	/**
	 * Die Methode durchsucht (mit Hilfe der Hilfsfunktion
	 * findAndAddMyTreeItemLeaves(MyTreeObject, List) den gesamten Baum nach einem
	 * Objekt, welches den übergebenen Pfad hat.
	 *
	 * @param path
	 *            der Pfad
	 * @return das Wurzelobjekt, falls path null oder leer ist oder kein Objekt
	 *         gefunden werden konnte, sonst selbiges.
	 */
	public MyTreeObject getMyTreeObjectForPath(final String path) {
		MyTreeObject o = dataObject;
		if ((path != null) && (path.length() > 0)) {
			final List<MyTreeObject> leaves = new ArrayList<>();
			findAndAddMyTreeItemLeaves(dataObject, leaves);
			for (final MyTreeObject leaf : leaves) {
				if (path.equals(leaf.getFullPath())) {
					o = leaf;
					break;
				}
			}
		}
		return o;
	}

	/**
	 * Hilfsmethode für die Rekurion zum Finden korrespondierender Objekte in
	 * Vorgabe- und Sollspalte. Nur für die dreispaltige Version interessant.
	 *
	 * @param path
	 *            der gesuchte Pfad.
	 * @param myTreeObject
	 *            Wurzelobjekt für diese Ebene der Rekursion
	 * @return das gefundene Objekt oder null, in dem Fall geht die Rekursion
	 *         weiter.
	 */
	private MyTreeObject matchPath(final String path, final MyTreeObject myTreeObject) {
		MyTreeObject match = null;
		if (path.equals(myTreeObject.getFullPath())) {
			match = myTreeObject;
		} else {
			if (myTreeObject.hasChildren()) {
				for (final MyTreeObject child : myTreeObject.getChildren()) {
					match = matchPath(path, child);
					if (match != null) {
						break;
					}
				}
			}
		}
		return match;
	}

	/**
	 * Die Funktion versucht, aus einem übergebenen {@link MyTreeObject} aus der
	 * Spalte 1 (die wir gerade editieren) das korrespondierende Objekt aus einem
	 * der Schattenobjekt-Bäume (Spalten 2,3,...) herauszufinden. Dazu wird der
	 * absolute Pfad des Objekts bestimmt und dann der indizierte
	 * Schattenobjekt-Baum rekursiv mit Hilfe der Methode matchPath(String,
	 * MyTreeObject) durchsucht, ob man ein Objekt findet, welches den gleichen
	 * absoluten Pfadnamen hat.
	 *
	 * @param myTreeObject
	 *            ein Objekt im editierten Baum (Spalte "Vorgabe")
	 * @param shadowIndex
	 *            Index des zu durchsuchenden Baums. Es muss gelten: 0 <=
	 *            shadowIndex < shadowDataObjects.length
	 * @return das gefundene Objekt oder null, wenn es keins gibt. Dies passiert
	 *         regelmäßig dann, wenn ein Feld im Vorgabe-Baum mehr Elemente enthält
	 *         als im durchsuchten Baum.
	 */
	public MyTreeObject getShadowObject(final MyTreeObject myTreeObject, final int shadowIndex) {
		if ((shadowIndex < 0) || (shadowIndex >= shadowDataObjects.length)) {
			return null;
		}
		final MyTreeObject o = shadowDataObjects[shadowIndex];
		MyTreeObject[] shadowList = shadowDataObjectMap.get(myTreeObject);
		MyTreeObject match = null;
		if (shadowList != null) {
			match = shadowList[shadowIndex];
			if (match != null) {
				ParaEditComposite.DEBUG.finest("Cache Hit");
			}
		} else {
			shadowList = new MyTreeObject[shadowDataObjects.length];
			shadowDataObjectMap.put(myTreeObject, shadowList);
		}
		if (null == match) {
			match = matchPath(myTreeObject.getFullPath(), o);
			shadowList[shadowIndex] = match;
			ParaEditComposite.DEBUG.finest("Cache Miss");
		}
		return match;
	}

	/**
	 * Die dritte Spalte mit den Sollparametern aktualisieren.
	 */
	private void updateSollParameter() {
		if (patameterInfos.length > 1) {
			final int shadowIndex = 1;
			final ClientDavInterface davIf = RahmenwerkService.getService().getObjektFactory().getDav();
			final ParameterManager hierarchie = MethodenBibliothek.getParameterManager(davIf);
			try {
				parameters[shadowIndex] = hierarchie.getParameter(patameterInfos[shadowIndex])[0];
				shadowDataObjects[shadowIndex - 1] = (MyTreeObject) (new MyDataObject(parameters[shadowIndex]))
						.getContent();
				for (final MyTreeObject[] array : shadowDataObjectMap.values()) {
					if (array.length >= shadowIndex) {
						array[shadowIndex - 1] = null;
					}
				}
			} catch (final ParameterClientException e) {
				ParamPlugin.getDefault().getLogger().error(
						"Fehler bei Abfrage der gespeicherten Parameter für " + patameterInfos[shadowIndex].toString(),
						e);
			}
		}
	}

	/**
	 * Liefert true, wenn Editieren nicht möglich sein soll. Die Daten können dann
	 * nur betrachtet werden.
	 *
	 * @return read-only?
	 */
	public boolean isReadOnly() {
		return readOnly;
	}

	/**
	 * Liefert das Editierte Objekt.
	 *
	 * @return das Objekt. Kann null sein.
	 */
	public MyTreeObject getDataObject() {
		return dataObject;
	}

	/**
	 * Liefert den Treeveiwer, der den bearbeiteten Parameter darstellt.
	 *
	 * @return der TreeViewer
	 */
	public final TreeViewer getTreeViewer() {
		return treeViewer;
	}

	/**
	 * Liefert das Zwischenobjekt, an dem die flachen Listen für Exporte hängen.
	 *
	 * @return das Zwischenobjekt. Kann null sein.
	 */
	public MyDataObject getMediatorDataObject() {
		return mediatorDataObject;
	}

	/**
	 * Liefert den Container, in dem wir uns befinden. Kann null sein.
	 *
	 * @return der Container
	 */
	public IParaEditCompositeContainer getContainer() {
		return container;
	}

	/**
	 * Liefert die Parameterinfo für den bearbeiteten Parameter.
	 *
	 * @return die Info
	 */
	public ParameterInfo getVorgabeParameterInfo() {
		return patameterInfos[0];
	}

	/**
	 * den Viewer aktualisieren und dabei hinterher den ursprünglichen
	 * Aufklapp-Zustand wiederherstellen.
	 *
	 * @param myTreeObject
	 *            Wenn <code>null</code>, wird der ganze Baum aktualisiert,
	 *            ansonsten nur dieser Knoten (und alle seine Kinder)
	 */
	public void refreshPreservingExpandState(final MyTreeObject myTreeObject) {
		treeState.push();
		if (null == myTreeObject) {
			treeViewer.refresh();
		} else {
			treeViewer.refresh(myTreeObject);
		}
		treeState.pop();
		setDirty();
	}
}
