/*
 * Rahmenwerk-Plug-in "Parametrierung"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.param.editors.helper;

import java.util.List;

import org.eclipse.jface.viewers.ILabelProviderListener;
import org.eclipse.jface.viewers.ITableColorProvider;
import org.eclipse.jface.viewers.ITableLabelProvider;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.Image;

import de.bsvrz.buv.plugin.param.ParamSharedImage;
import de.bsvrz.buv.plugin.param.editors.ParameterEditorInput;
import de.bsvrz.dav.daf.main.Data;
import de.bsvrz.dav.daf.main.config.AttributeType;
import de.bsvrz.dav.daf.main.config.DoubleAttributeType;
import de.bsvrz.dav.daf.main.config.IntegerAttributeType;
import de.bsvrz.dav.daf.main.config.IntegerValueRange;
import de.bsvrz.dav.daf.main.config.IntegerValueState;
import de.bsvrz.dav.daf.main.config.ReferenceAttributeType;
import de.bsvrz.dav.daf.main.config.StringAttributeType;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.dav.daf.main.config.TimeAttributeType;

/**
 * label-provider splitting the tree content from the table content.
 *
 * @author BitCtrl Systems GmbH, Thomas Thierfelder
 */
public class TableTreeLabelProvider implements ITableLabelProvider, ITableColorProvider {

	/** das Composite, für das wir zuständig sind. */
	private final ParaEditComposite pec;

	/**
	 * Liefert Informationen über ein eine Subdatum vom Typ Double.
	 *
	 * @param attType
	 *            der Double-Attributtyp
	 * @param data
	 *            das Subdatum
	 * @return die Info
	 */
	public static final String getDoubleInfo(final DoubleAttributeType attType, final Data data) {
		String info = "DoubleAttributeType"; //$NON-NLS-1$

		if (data.isArray()) {
			info += " - Array"; //$NON-NLS-1$
		} else {
			info += "Genauigkeit: ";
			if (DoubleAttributeType.DOUBLE == attType.getAccuracy()) {
				info += " doppelt";
			} else {
				info += " einfach";
			}
		}
		return info;
	}

	/**
	 * Liefert Informationen über ein eine Subdatum vom Typ Time.
	 *
	 * @param attType
	 *            der Time-Attributtyp
	 * @param data
	 *            das Subdatum
	 * @return die Info
	 */
	public static final String getTimeInfo(final TimeAttributeType attType, final Data data) {
		String info = ""; //$NON-NLS-1$

		if (data.isArray()) {
			info += "Zeitangaben - Array"; //$NON-NLS-1$
			return info;
		}
		if (attType.isRelative()) {
			info += "relative Zeitangabe"; //$NON-NLS-1$
		} else {
			info += "absolute Zeitangabe"; //$NON-NLS-1$
		}
		if (attType.getAccuracy() == TimeAttributeType.MILLISECONDS) {
			info += ", millisekundengenau"; //$NON-NLS-1$
		} else if (attType.getAccuracy() == TimeAttributeType.SECONDS) {
			info += ", sekundengenau"; //$NON-NLS-1$
		}
		return info;
	}

	/**
	 * Liefert Informationen über ein eine Subdatum vom Typ String.
	 *
	 * @param attType
	 *            der String-Attributtyp
	 * @param data
	 *            das Subdatum
	 * @return die Info
	 */
	public static final String getStringInfo(final StringAttributeType attType, final Data data) {
		String info = "Text"; //$NON-NLS-1$

		if (data.isArray()) {
			info += " - Array"; //$NON-NLS-1$
		} else {
			info += " - Längenbegrenzung " + attType.getMaxLength() + " Zeichen"; //$NON-NLS-1$ //$NON-NLS-2$
		}
		return info;
	}

	/**
	 * Liefert Informationen über ein eine Subdatum vom Typ Referenz.
	 *
	 * @param attType
	 *            der Referenz-Attributtyp
	 * @param data
	 *            das Subdatum
	 * @return die Info
	 */
	public static final String getReferenceInfo(final ReferenceAttributeType attType, final Data data) {
		String info = "Objektreferenz "; //$NON-NLS-1$

		if (data.isArray()) {
			info += "- Array"; //$NON-NLS-1$
		} else {
			info += "auf ";
			if (null == attType.getReferencedObjectType()) {
				info += "beliebigen Typ";
			} else {
				info += "Typ ";
				info += attType.getReferencedObjectType().toString();
			}
		}
		return info;
	}

	/**
	 * Liefert Informationen über ein eine Subdatum vom Typ Integer.
	 *
	 * @param attType
	 *            der Integer-Attributtyp
	 * @param data
	 *            das Subdatum
	 * @return die Info
	 */
	public static String getIntegerInfo(final IntegerAttributeType attType, final Data data) {
		String info = attType.getNameOrPidOrId();

		if (data.isArray()) {
			info += " - Array"; //$NON-NLS-1$
			return info;
		}
		final IntegerValueRange range = attType.getRange();
		if (range != null) {
			info += " - min=" + (range.getMinimum() * range.getConversionFactor()) + " max=" //$NON-NLS-1$ //$NON-NLS-2$
					+ (range.getMaximum() * range.getConversionFactor());
		}
		info += "; Zustände: ";
		final List<IntegerValueState> states = attType.getStates();
		if ((null == states) || states.isEmpty()) {
			info += "keine.";
		} else {
			int stateLoop = 0;
			for (final IntegerValueState state : states) {
				if (stateLoop > 0) {
					info += ", ";
				}
				info += state.getValue() + " (" + state.getName() + ")";
				/*
				 * 4 ist willkürlich festgelegt, passt im allgemeinen auf eine Zeile
				 */
				if (stateLoop > 4) {
					info += " ...";
					break;
				}
				++stateLoop;
			}
		}
		return info;
	}

	/**
	 * Liefert Informationen über ein Attribut.
	 *
	 * @param data
	 *            das Subdatum
	 * @return die Info
	 */
	public static String getAttributInfo(final Data data) {
		String info = "";
		final AttributeType attrType = data.getAttributeType();
		if (attrType instanceof IntegerAttributeType) {
			info = TableTreeLabelProvider.getIntegerInfo((IntegerAttributeType) attrType, data);
		} else if (attrType instanceof ReferenceAttributeType) {
			info = TableTreeLabelProvider.getReferenceInfo((ReferenceAttributeType) attrType, data);
		} else if (attrType instanceof StringAttributeType) {
			info = TableTreeLabelProvider.getStringInfo((StringAttributeType) attrType, data);
		} else if (attrType instanceof TimeAttributeType) {
			info = TableTreeLabelProvider.getTimeInfo((TimeAttributeType) attrType, data);
		} else if (attrType instanceof DoubleAttributeType) {
			info = TableTreeLabelProvider.getDoubleInfo((DoubleAttributeType) attrType, data);
		}
		return info;
	}

	/**
	 * Labelprovider.
	 *
	 * @param pec
	 *            das Composite, für das wir zuständig sind.
	 */
	public TableTreeLabelProvider(final ParaEditComposite pec) {
		super();
		this.pec = pec;
	}

	/**
	 * Setzt ggf. dass rote Kästchen, wenn ein Element noch nicht ausgefüllt wurde,
	 * es aber ausgefüllt sein muss.
	 *
	 * @param arg0
	 *            - das Element
	 * @param arg1
	 *            - die Spalte
	 * @return das Image
	 * @see org.eclipse.jface.viewers.ITableLabelProvider#getColumnImage(java.lang.Object,
	 *      int)
	 */
	@Override
	public Image getColumnImage(final Object arg0, final int arg1) {
		final MyTreeObject myTreeObject = (MyTreeObject) arg0;
		if (!(myTreeObject.getData() instanceof Data)) {
			return null;
		}
		final Data data = (Data) myTreeObject.getData();
		final String valueString = data.valueToString();

		final int werteBereichsColumn = pec.getTreeViewer().getTree().getColumnCount() - 1;
		if ((arg1 != werteBereichsColumn)
				&& ((null == valueString) || valueString.contains("<<null>>") || valueString.contains("ndefiniert"))) {
			return ParamSharedImage.ERROR_NULL.getImage();
		}
		return null;
	}

	/**
	 * Erfragt den Text eines Editorelements in einer best. Spalte.
	 *
	 * @param arg0
	 *            - das Element
	 * @param arg1
	 *            - die Spalte
	 * @return den Text
	 * @see org.eclipse.jface.viewers.ITableLabelProvider#getColumnText(java.lang.Object,
	 *      int)
	 */
	@Override
	public String getColumnText(final Object arg0, final int arg1) {
		final MyTreeObject myTreeObject = (MyTreeObject) arg0;
		if (!(myTreeObject.getData() instanceof Data)) {
			return "Keine Daten!";
		}
		Data data = (Data) myTreeObject.getData();

		if (arg1 == 0) {
			String columnText = "";
			if (myTreeObject.isDirty()) {
				columnText = "*";
			}
			return columnText + data.getName();
		} else if (arg1 == 1) { // prepare Table content here
			data = (Data) myTreeObject.getData();
		} else if (2 == arg1) {
			final MyTreeObject myShadowObject = pec.getShadowObject(myTreeObject, arg1 - 2);
			if ((myShadowObject == null) || !(myShadowObject.getData() instanceof Data)) {
				return TableTreeLabelProvider.getAttributInfo(data);
			}
			data = (Data) myShadowObject.getData();
		} else if (3 == arg1) {
			return TableTreeLabelProvider.getAttributInfo(data);
		}
		if (data != null) {
			if (data.isArray() || data.isList()) {
				String output = data.toString();
				output = output.substring((data.getName()).length() + 1, output.length());
				return output;
			}

			if ((null == data.valueToString()) || data.valueToString().startsWith("<<null>>")) {
				return ParaEditCompositeConstants.EMPTY;
			}
			if (data.getAttributeType() instanceof ReferenceAttributeType) {
				final SystemObject o = data.asReferenceValue().getSystemObject();
				if (o != null) {
					String name = o.getName();
					if (name.isEmpty()) {
						name = "(" + o.getPidOrId() + ")";
					}
					return name + " [Doppelklick zum Ändern]";
				}
				return "Doppelklick zum Zuweisen";
			}
			return data.valueToString();
		}
		return "FEHLER: unerwarteter Datentyp"; //$NON-NLS-1$
	}

	/**
	 * nicht implementiert.
	 *
	 * @param listener
	 *            der Listener
	 * @see org.eclipse.jface.viewers.IBaseLabelProvider#addListener(org.eclipse.jface.viewers.ILabelProviderListener)
	 */
	@Override
	public void addListener(final ILabelProviderListener listener) {
		//
	}

	/**
	 * nicht implementiert.
	 *
	 * @see org.eclipse.jface.viewers.IBaseLabelProvider#dispose()
	 */
	@Override
	public void dispose() {
		//
	}

	/**
	 * Erfragt, ob die Eigenschaft gültig ist.
	 *
	 * @param obj
	 *            das Objekt
	 * @param property
	 *            die Eigenschaft
	 * @return die Eigenschaft ist gültig ?
	 * @see org.eclipse.jface.viewers.IBaseLabelProvider#isLabelProperty(java.lang.Object,
	 *      java.lang.String)
	 */
	@Override
	public boolean isLabelProperty(final Object obj, final String property) {
		return false;
	}

	/**
	 * nicht implementiert.
	 *
	 * @param listener
	 *            der Listener
	 * @see org.eclipse.jface.viewers.IBaseLabelProvider#removeListener(org.eclipse.jface.viewers.ILabelProviderListener)
	 */
	@Override
	public void removeListener(final ILabelProviderListener listener) {
		//
	}

	@Override
	public Color getBackground(final Object element, final int columnIndex) {
		Color background = null;
		final MyTreeObject myTreeObject = (MyTreeObject) element;
		if (!pec.isReadOnly() && (1 == columnIndex) && (null != pec.getContainer())
				&& (myTreeObject.getData() instanceof Data) && ((Data) myTreeObject.getData()).isPlain()) {
			final Data data = (Data) myTreeObject.getData();
			final String value = ParameterEditorInput.getPlainValueFromPlainData(data);
			background = pec.getContainer().getParameterEditorInput().getVorgabeColor(pec.getVorgabeParameterInfo(),
					myTreeObject.getFullPath(), value);
		}
		return background;
	}

	@Override
	public Color getForeground(final Object element, final int columnIndex) {
		// leer
		return null;
	}
}
