/*
 * Rahmenwerk-Plug-in "Parametrierung"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.param.editors.helper;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.List;

import org.eclipse.jface.viewers.ICellModifier;
import org.eclipse.swt.widgets.TreeItem;

import de.bsvrz.buv.plugin.param.internal.RahmenwerkService;
import de.bsvrz.dav.daf.main.ClientDavInterface;
import de.bsvrz.dav.daf.main.Data;
import de.bsvrz.dav.daf.main.config.AttributeType;
import de.bsvrz.dav.daf.main.config.DoubleAttributeType;
import de.bsvrz.dav.daf.main.config.IntegerAttributeType;
import de.bsvrz.dav.daf.main.config.IntegerValueRange;
import de.bsvrz.dav.daf.main.config.IntegerValueState;
import de.bsvrz.dav.daf.main.config.ReferenceAttributeType;
import de.bsvrz.dav.daf.main.config.StringAttributeType;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.dav.daf.main.config.SystemObjectType;
import de.bsvrz.dav.daf.main.config.TimeAttributeType;

/**
 * modifier concerning whith string-, integer- and float-editing.
 *
 * @author BitCtrl Systems GmbH, Thomas Thierfelder
 *
 *
 */
public class TextModifier implements ICellModifier {

	/** das Composite, für das wir zuständig sind. */
	private final ParaEditComposite pec;

	/**
	 * Identifier für Spalte 0, die dürfen wir nicht bearbeiten.
	 */
	private final String properties0;

	/**
	 * Konstruktor merkt sich nur Member.
	 *
	 * @param pec
	 *            das Composite, für das wir zuständig sind.
	 * @param properties0
	 *            Identifier für Spalte 0, die dürfen wir nicht bearbeiten.
	 */
	public TextModifier(final ParaEditComposite pec, final String properties0) {
		this.pec = pec;
		this.properties0 = properties0;
	}

	/**
	 * Erfragt, ob eine bestimmtes Element modifiziert werden darf.
	 *
	 * @param object
	 *            - das Objekt
	 * @param property
	 *            - der Spaltentext in der Kopfzeile
	 * @return Element kann modifiziert werden ?
	 * @see org.eclipse.jface.viewers.ICellModifier#canModify(java.lang.Object,
	 *      java.lang.String)
	 */
	@Override
	public boolean canModify(final Object object, final String property) {
		if (property.equals(properties0)) {
			return false;
		}

		final MyTreeObject myTreeObject = (MyTreeObject) object;
		final Data data = (Data) myTreeObject.getData();

		return data.isPlain();
	}

	/**
	 * Erfragt den Wert eines bestimmten Elements.
	 *
	 * @param element
	 *            - das Element
	 * @param property
	 *            - der Spaltentext in der Kopfzeile
	 * @return den Wert
	 * @see org.eclipse.jface.viewers.ICellModifier#getValue(java.lang.Object,
	 *      java.lang.String)
	 */
	@Override
	public String getValue(final Object element, final String property) {
		final MyTreeObject myTreeObject = (MyTreeObject) element;
		final Data data = (Data) myTreeObject.getData();
		final AttributeType attType = data.getAttributeType();
		String str = " "; //$NON-NLS-1$

		if (attType instanceof ReferenceAttributeType) {
			return " "; //$NON-NLS-1$
		}
		if (data.valueToString().startsWith("<<null>>")) {
			str = " "; //$NON-NLS-1$
		} else {
			str = data.valueToString();
		}
		pec.innerCheck(pec.getDataObject());
		return str;
	}

	/**
	 * Verändert den Wert eines Elements abhängig von seinem (DV-)Typ.
	 *
	 * @param element
	 *            - das Element
	 * @param property
	 *            - der Spaltentext in der Kopfzeile
	 * @param value
	 *            - der neue Wert dieses Elements (der Wert, der in das Element
	 *            eingetragen werden soll)
	 * @see org.eclipse.jface.viewers.ICellModifier#modify(java.lang.Object,
	 *      java.lang.String, java.lang.Object)
	 */
	@Override
	public void modify(final Object element, final String property, final Object value) {
		final TreeItem tto = (TreeItem) element;
		final MyTreeObject myTreeObject = (MyTreeObject) tto.getData();
		final Data data = (Data) myTreeObject.getData();
		boolean reallyModified = false;
		if (null == value) {
			return;
		}

		if (data.getAttributeType() instanceof ReferenceAttributeType) {
			final ReferenceAttributeType ref = (ReferenceAttributeType) data.getAttributeType();

			final ClientDavInterface davIf = RahmenwerkService.getService().getObjektFactory().getDav();
			final SystemObject oldValue = data.asReferenceValue().getSystemObject();
			final SystemObject newValue = davIf.getDataModel().getObject((String) value);
			if (null == newValue) {
				pec.doAdvice("Fehler: Referenz-Object konnte nicht identifiziert werden.", //$NON-NLS-1$
						ParaEditCompositeConstants.ALERT);
			} else {
				final SystemObjectType referencedType = ref.getReferencedObjectType();
				if ((referencedType == null) || newValue.isOfType(referencedType)) {
					if (!newValue.equals(oldValue)) {
						data.asReferenceValue().setSystemObject(newValue);
						reallyModified = true;
						if (pec.getContainer() != null) {
							pec.getContainer().valueModified(pec, myTreeObject, oldValue, newValue,
									data.getAttributeType());
						}
					}
				} else {
					pec.doAdvice("Fehler: Referenz-Object ist kein Element von " + referencedType.getName(),
							ParaEditCompositeConstants.ALERT);
				}
			}
		} else if ((data.getAttributeType() instanceof IntegerAttributeType) && (((String) value).length() > 0)) {
			final long oldValue = data.asUnscaledValue().longValue();
			final Double newDoubleValue = Double.valueOf(((String) value).replaceAll(",", ".")); //$NON-NLS-1$ //$NON-NLS-2$
			final long newValue = newDoubleValue.longValue();
			final IntegerAttributeType attrType = (IntegerAttributeType) data.getAttributeType();
			final IntegerValueRange range = attrType.getRange();
			boolean valueSetViaRange = false;
			if (null != range) {
				final long unscaledValue = Math.round(newDoubleValue / range.getConversionFactor());
				if ((unscaledValue >= range.getMinimum()) && (unscaledValue <= range.getMaximum())) {
					if (oldValue != unscaledValue) {
						data.asUnscaledValue().set(unscaledValue);
						reallyModified = true;
						if (pec.getContainer() != null) {
							pec.getContainer().valueModified(pec, myTreeObject, oldValue, unscaledValue,
									data.getAttributeType());
						}
						valueSetViaRange = true;
					}
				}
			}
			if (!valueSetViaRange) {
				final List<IntegerValueState> states = attrType.getStates();
				if ((null != states) && !states.isEmpty()) {
					for (final IntegerValueState state : states) {
						if (state.getValue() == newValue) {
							if (oldValue != newValue) {
								data.asUnscaledValue().set(newValue);
								reallyModified = true;
								if (pec.getContainer() != null) {
									pec.getContainer().valueModified(pec, myTreeObject, oldValue, newValue,
											data.getAttributeType());
								}
								break;
							}
						}
					}
				}
			}
		} else if (data.getAttributeType() instanceof DoubleAttributeType) {
			final double oldValue = data.asUnscaledValue().doubleValue();
			final double newValue = Double.valueOf(((String) value).replaceAll(",", ".")).doubleValue(); //$NON-NLS-1$ //$NON-NLS-2$
			if (newValue != oldValue) {
				data.asUnscaledValue().set(newValue);
				reallyModified = true;
				if (pec.getContainer() != null) {
					pec.getContainer().valueModified(pec, myTreeObject, oldValue, newValue, data.getAttributeType());
				}
			}
		} else if (data.getAttributeType() instanceof StringAttributeType) {
			final String oldValue = data.asTextValue().getText();
			final String newValue = (String) value;
			if (!oldValue.equals(newValue)) {
				data.asTextValue().setText(newValue);
				reallyModified = true;
				if (pec.getContainer() != null) {
					pec.getContainer().valueModified(pec, myTreeObject, oldValue, newValue, data.getAttributeType());
				}
			}
		} else if (data.getAttributeType() instanceof TimeAttributeType) {
			final TimeAttributeType tat = (TimeAttributeType) data.getAttributeType();
			final String v = (String) value;
			final long oldValue = data.asTimeValue().getMillis();
			long newValue;

			if (tat.isRelative()) {
				// analyse this:
				// {Tage}:{Stunden}:{Minuten}:{Sek.}:{Millisek.}
				final String[] parts = v.split(":"); //$NON-NLS-1$
				final long id = Long.parseLong(parts[0]);
				final long ih = Long.parseLong(parts[1]);
				final long imin = Long.parseLong(parts[2]);
				final long isec = Long.parseLong(parts[3]);
				long ims = Long.parseLong(parts[4]);

				ims += (isec * 1000) + (imin * 60 * 1000) + (ih * 60 * 60 * 1000) + (id * 24 * 60 * 60 * 1000);
				newValue = ims;
				if (oldValue != newValue) {
					data.asTimeValue().setMillis(newValue);
					reallyModified = true;
					if (pec.getContainer() != null) {
						pec.getContainer().valueModified(pec, myTreeObject, oldValue, newValue,
								data.getAttributeType());
					}
				}
			} else {
				// analyse this tt:mm:jjjj, hh:mm:ss:ms
				final SimpleDateFormat format = new SimpleDateFormat("dd.MM.yyyy, HH:mm:ss:SS"); //$NON-NLS-1$

				try {
					final Date date = format.parse(v);
					newValue = date.getTime();
					if (oldValue != newValue) {
						data.asTimeValue().setMillis(newValue);
						reallyModified = true;
						if (pec.getContainer() != null) {
							pec.getContainer().valueModified(pec, myTreeObject, oldValue, newValue,
									data.getAttributeType());
						}
					}
				} catch (final ParseException e) {
					pec.doAdvice("Fehler: Bitte halten Sie folgendes Eingabeformat ein: tt:mm:jjjj, hh:mm:ss:ms.", //$NON-NLS-1$
							ParaEditCompositeConstants.ALERT);
				}
			}
		} else {
			pec.doAdvice("Unbekannter Attributtyp", ParaEditCompositeConstants.ALERT);
		}

		if (reallyModified) {
			pec.refreshPreservingExpandState(null);
		}
	}
}
