/*
 * Rahmenwerk-Plug-in "Parametrierung"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.param.handler;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.core.commands.ExecutionEvent;
import org.eclipse.core.commands.ExecutionException;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Cursor;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IEditorReference;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.PlatformUI;

import de.bsvrz.buv.plugin.param.editors.ParameterEditor;
import de.bsvrz.buv.plugin.param.editors.ParameterEditorInput;
import de.bsvrz.buv.plugin.param.editors.ParameterSaveResultEnum;
import de.bsvrz.buv.plugin.param.editors.ParameterSaver;
import de.bsvrz.buv.plugin.param.editors.ParameterSaver.ParameterSaveUrsache;
import de.bsvrz.buv.plugin.param.internal.ParamPlugin;
import de.bsvrz.buv.plugin.param.jobs.ParameterEntfernenJob;
import de.bsvrz.puk.param.lib.Parameter;
import de.bsvrz.puk.param.lib.ParameterInfo;

/**
 * Handler für das Kommando zum Entfernen eines oder mehrerer {@link Parameter}.
 *
 * @author BitCtrl Systems GmbH, Albrecht Uhlmann, Uwe Peuker
 */
public class ParameterEntfernenHandler extends AbstractParameterModifizierenHandler {

	public static final String COMMAND_ID = "de.bsvrz.buv.plugin.param.actions.ParameterEntfernenAktion";

	/**
	 * Führt den Löschvorgang aus, hierbei wird ein Job benutzt, aber zur Zeit im
	 * Vordergrund (blockierende Operation, laut Kundenvorgabe).
	 *
	 * @param inputsContainingParameters
	 *            eine Liste von Editor-Input-Objekten, welche mindestens einen der
	 *            soeben gelöschten Parameter enthält. Wenn das Löschen erfolgreich
	 *            war, wird diese Liste bearbeitet, d.h. man erhält die Möglichkeit,
	 *            den neuen Wert des Parameters, den er nach dem Löschen hat, zu
	 *            laden.
	 * @param parameters
	 */
	private void doDeleteParameters(final List<ParameterEditor> inputsContainingParameters,
			final Parameter[] parameters) {
		final ParameterEntfernenJob job = new ParameterEntfernenJob(findContentProvider(), parameters);
		if (ParamPlugin.getDefault().isUsingReloadJobs()) {
			job.schedule();
		} else {
			Cursor waitCursor = null;
			final Shell shell = Display.getDefault().getActiveShell();
			if (null != shell) {
				waitCursor = new Cursor(Display.getDefault(), SWT.CURSOR_WAIT);
				shell.setCursor(waitCursor);
			}
			final IStatus status = job.run(new NullProgressMonitor());
			if (null != shell) {
				shell.setCursor(null);
			}
			if (null != waitCursor) {
				waitCursor.dispose();
			}
			if (!status.isOK()) {
				MessageDialog.openError(null, "Fehler beim Löschen von Parametern", status.getMessage());
			} else {
				handleEditorsContainingDeletedParameter(inputsContainingParameters, parameters);
			}
		}
	}

	/**
	 * Nachbearbeitung gelöschter Parameter.
	 *
	 * @param editorsContainingParameters
	 *            eine Liste von Editor-Input-Objekten, welche mindestens einen der
	 *            soeben gelöschten Parameter enthält. Wenn das Löschen erfolgreich
	 *            war, wird diese Liste bearbeitet, d.h. man erhält die Möglichkeit,
	 *            den neuen Wert des Parameters, den er nach dem Löschen hat, zu
	 *            laden.
	 *
	 *            Die Liste kann null oder leer sein.
	 * @param parameters
	 */
	private void handleEditorsContainingDeletedParameter(final List<ParameterEditor> editorsContainingParameters,
			final Parameter[] parameters) {
		if ((null == editorsContainingParameters) || editorsContainingParameters.isEmpty()) {
			return;
		}
		for (final ParameterEditor editor : editorsContainingParameters) {
			editor.selectiveReload(parameters, false);
			applyInitialDirtyState(editor.getEditorInput(), editor);
		}
	}

	/**
	 * Bestimmt aus der Menge geöffneter Parametereditoren diejenigen, die
	 * mindestens einen der zu löschenden Parameter enthalten.
	 *
	 * @param parameters
	 *
	 * @return eine Liste der in Frage kommenden Editoren. Diese kann null oder leer
	 *         sein.
	 */
	private List<ParameterEditor> getEditorsContainingDeletedParameter(final Parameter[] parameters) {
		final IEditorReference[] refs = PlatformUI.getWorkbench().getActiveWorkbenchWindow().getActivePage()
				.findEditors(null, ParameterEditor.EDITOR_ID, IWorkbenchPage.MATCH_ID);
		if ((null == refs) || (refs.length <= 0)) {
			return null;
		}
		// Mind. ein Parameter-Editor offen. Alle heraussuchen, die auch
		// nur einen der zu löschenden Parameter enthalten.
		final List<ParameterEditor> editorsContainingParameters = new ArrayList<>();
		for (final IEditorReference ref : refs) {
			final ParameterEditor editor = (ParameterEditor) ref.getEditor(false);
			final ParameterEditorInput pei = editor.getEditorInput();
			boolean editorAdded = false;
			for (final Parameter editedParameter : pei.getParameters()) {
				final ParameterInfo editedParameterInfo = editedParameter.getInfo();
				for (final Parameter deleteParameter : parameters) {
					if (editedParameterInfo.equals(deleteParameter.getInfo())) {
						editorsContainingParameters.add(editor);
						editorAdded = true;
						break;
					}
				}
				if (editorAdded) {
					break;
				}
			}
		}
		return editorsContainingParameters;
	}

	@Override
	public Object execute(final ExecutionEvent event) throws ExecutionException {

		// Zunächst Parameter mit aktuellem Wert nochmal schreiben, um das
		// Löschen zu dokumentieren.

		final Parameter[] parameters = getParameters(event);
		if ((parameters == null) || (parameters.length <= 0)) {
			return null;
		}
		final List<ParameterEditor> editorsContainingParameters = getEditorsContainingDeletedParameter(parameters);

		final ParameterSaveResultEnum saveResult = ParameterSaver.saveParameters(parameters,
				ParameterSaver.buildDefaultUrsache(parameters, parameters, ParameterSaveUrsache.GELOESCHT));
		switch (saveResult) {
		case PARAMETER_SAVE_FAILURE:
			final MessageDialog md = new MessageDialog(null, "Fehler beim Speichern von Parametern", null,
					ParameterSaver.getLastError(), MessageDialog.ERROR, new String[] { "OK" }, 0);
			md.open();
			break;
		case PARAMETER_SAVE_CANCEL:
			// Do nothing
			break;
		case PARAMETER_SAVE_SUCCESS:
			// Jetzt wirklich löschen
			doDeleteParameters(editorsContainingParameters, parameters);
			break;
		default:
			break;
		}
		return null;
	}
}
