/*
 * Rahmenwerk-Plug-in "Parametrierung"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */

package de.bsvrz.buv.plugin.param.internal;

import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;

import de.bsvrz.buv.plugin.param.ArtDesParametersatzesCache;
import de.bsvrz.buv.plugin.param.IArtDesParametersatzesProvider;
import de.bsvrz.buv.plugin.param.ParametertypenCache;
import de.bsvrz.buv.plugin.param.lib.ParameterManagerAllgemein;
import de.bsvrz.buv.rw.basislib.Rahmenwerk;
import de.bsvrz.buv.rw.basislib.berechtigung.Berechtigungen;
import de.bsvrz.buv.rw.basislib.einstellungen.Einstellungen;
import de.bsvrz.dav.daf.main.archive.ArchiveRequestManager;
import de.bsvrz.sys.funclib.bitctrl.daf.DavProvider;
import de.bsvrz.sys.funclib.bitctrl.modell.ObjektFactory;
import de.bsvrz.sys.funclib.bitctrl.modell.util.KappichModellUtil;

/**
 * Interne Komponente zur Bereitstellung der für das Plug-in erforderlichen
 * Services.
 *
 * @author BitCtrl Systems GmbH, Uwe Peuker
 *
 *
 */
public class RahmenwerkService {

	private static RahmenwerkService service;

	private Rahmenwerk rahmenWerk;
	private Berechtigungen berechtigungen;
	private Einstellungen einstellungen;
	private ObjektFactory objektFactory;

	/**
	 * Das Singleton für die Ansicht rechts-oben in Abb 5-41.
	 */
	private IArtDesParametersatzesProvider artDesParametersatzesProvider;

	/**
	 * liefert die globale Instanz der Komponente.
	 *
	 * @return die Komponente
	 */
	public static RahmenwerkService getService() {
		return RahmenwerkService.service;
	}

	/** aktiviert die Komponente. */
	protected void activate() {
		RahmenwerkService.service = this;
		artDesParametersatzesProvider = new ArtDesParametersatzesCache();
		if (objektFactory.isVerbunden()) {
			startOnline();
		} else {
			startOffline();
		}
		objektFactory.addPropertyChangeListener(new PropertyChangeListener() {
			@Override
			public void propertyChange(final PropertyChangeEvent evt) {
				if (DavProvider.PROP_VERBUNDEN.equals(evt.getPropertyName())) {
					if (Boolean.TRUE.equals(evt.getNewValue())) {
						startOnline();
					} else {
						startOffline();
					}
				}
			}
		});
	}

	/** deaktiviert die Komponente. */
	protected void deactivate() {
		RahmenwerkService.service = null;
	}

	/**
	 * bindet einen Rahmenwerk-Service an die Komponente.
	 *
	 * @param newRahmenWerk
	 *            der Service
	 */
	protected void bindRahmenwerk(final Rahmenwerk newRahmenWerk) {
		this.rahmenWerk = newRahmenWerk;
	}

	/**
	 * entfernt einen Rahmenwerk-Service von der Komponente.
	 *
	 * @param newRahmenWerk
	 *            der Service
	 */
	protected void unbindRahmenwerk(final Rahmenwerk newRahmenWerk) {
		this.rahmenWerk = null;
	}

	/**
	 * liefert den {@link Rahmenwerk}-Service.
	 *
	 * @return den Service
	 */
	public Rahmenwerk getRahmenWerk() {
		return rahmenWerk;
	}

	/**
	 * bindet einen ObjektFactory-Service an die Komponente.
	 *
	 * @param newObjektFactory
	 *            der Service
	 */
	protected void bindObjektFactory(final ObjektFactory newObjektFactory) {
		this.objektFactory = newObjektFactory;
	}

	/**
	 * entfernt einen ObjektFactory-Service von der Komponente.
	 *
	 * @param oldObjektFactory
	 *            der Service
	 */
	protected void unbindObjektFactory(final ObjektFactory oldObjektFactory) {
		this.objektFactory = null;
	}

	/**
	 * liefert den {@link ObjektFactory}-Service.
	 *
	 * @return den Service
	 */
	public ObjektFactory getObjektFactory() {
		return objektFactory;
	}

	/**
	 * bindet einen {@link Berechtigungen}-Service an die Komponente.
	 *
	 * @param newBerechtigungen
	 *            der Service
	 */
	protected void bindBerechtigungen(final Berechtigungen newBerechtigungen) {
		this.berechtigungen = newBerechtigungen;
	}

	/**
	 * entfernt einen {@link Berechtigungen}-Service von der Komponente.
	 *
	 * @param newBerechtigungen
	 *            der Service
	 */
	protected void unbindBerechtigungen(final Berechtigungen newBerechtigungen) {
		this.berechtigungen = null;
	}

	/**
	 * liefert den {@link Berechtigungen}-Service.
	 *
	 * @return den Service
	 */
	public Berechtigungen getBerechtigungen() {
		return berechtigungen;
	}

	/**
	 * bindet einen {@link Einstellungen}-Service an die Komponente.
	 *
	 * @param newEinstellungen
	 *            der Service
	 */
	protected void bindEinstellungen(final Einstellungen newEinstellungen) {
		this.einstellungen = newEinstellungen;
	}

	/**
	 * entfernt einen {@link Einstellungen}-Service von der Komponente.
	 *
	 * @param newEinstellungen
	 *            der Service
	 */
	protected void unbindEinstellungen(final Einstellungen newEinstellungen) {
		this.einstellungen = null;
	}

	/**
	 * liefert den {@link Einstellungen}-Service.
	 *
	 * @return den Service
	 */
	public Einstellungen getEinstellungen() {
		return einstellungen;
	}

	/**
	 * Das Plug-in bei vorhandener DaV-Verbindung (neu) starten.
	 */
	private void startOnline() {
		// Preload bestimmter Modellklassen erzwingen. Bei korrekter
		// Abhängigkeitenreihenfolge sollte das nicht nötig sein,
		// aber es schadet ja auch nicht.
		KappichModellUtil.getAOE(getObjektFactory());

		ParameterManagerAllgemein.getInstanz().setVerbindung(getObjektFactory().getDav());
		// Auf Verfügbarkeit des Archivs warten (SVN 23332)
		final ArchiveRequestManager archiv = getObjektFactory().getDav().getArchive();
		int waitLoop = 10;
		while (!archiv.isArchiveAvailable() && (waitLoop-- > 0)) {
			try {
				Thread.sleep(100L);
			} catch (final InterruptedException e) {
				if (ParamPlugin.getDefault() != null) {
					ParamPlugin.getDefault().getLogger()
							.warning("Unterbrechung beim Warten " + "auf Verfügbarkeit des Archivs", e);
				}
			}
		}
		if (waitLoop <= 0) {
			if (ParamPlugin.getDefault() != null) {
				ParamPlugin.getDefault().getLogger()
						.warning("Archiv ist nicht verfügbar! " + "Das Feature zum Abfragen historischer Parameter"
								+ " wird möglicherweise nicht funktionieren!");
			}
		}

		getArtDesParametersatzesProvider().setArchivAvailable(archiv.isArchiveAvailable());
		ParametertypenCache.erzeugeInstanz(getObjektFactory());
		if (ParamPlugin.getDefault() != null) {
			ParamPlugin.getDefault().getLogger()
					.info("Initialisiere die Berechtigungsverwaltung" + " des Plug-in Parametrierung");
		}
	}

	/**
	 * Das Plug-in bei fehlender DaV-Verbindung (neu) starten.
	 */
	private void startOffline() {
		ParameterManagerAllgemein.getInstanz().setVerbindung(null);
		getArtDesParametersatzesProvider().setArchivAvailable(false);
	}

	/**
	 * Zugriff auf die Eingaben in der Ansicht rechts-oben in Abb 5-41.
	 *
	 * @return das Objekt
	 */
	public IArtDesParametersatzesProvider getArtDesParametersatzesProvider() {
		return artDesParametersatzesProvider;
	}
}
