/*
 * Rahmenwerk-Plug-in "Parametrierung"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.param.jobs;

import org.eclipse.core.runtime.Assert;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.jface.viewers.ContentViewer;

import de.bsvrz.buv.plugin.param.internal.ParamPlugin;
import de.bsvrz.buv.plugin.param.provider.AbstractParamPluginContentProvider;
import de.bsvrz.buv.plugin.param.views.AbstractParamPluginView;
import de.bsvrz.puk.param.lib.ParameterClientException;

/**
 * Aufruf der Methode
 * {@link AbstractParamPluginContentProvider#reloadCurrentInput(boolean)} aus
 * einem Job heraus, damit die Bedienoberfläche nicht blockiert.
 *
 * @author BitCtrl Systems GmbH, Albrecht Uhlmann
 *
 */
public class ReloadCurrentInputJob extends Job {

	/**
	 * Contentprovider der Ansicht, die zu aktualisieren ist.
	 */
	private final AbstractParamPluginContentProvider contentProvider;

	/**
	 * Bestimmt, ob der Viewer aktualisiert werden soll oder nicht.
	 */
	private final boolean doRefreshViewer;

	/**
	 * Konstruktor stellt sicher, dass die übergebene Ansicht eine
	 * {@link AbstractParamPluginView} ist, deren
	 * {@link com.bitctrl.lib.eclipse.views.SelectionProvider} ein
	 * {@link ContentViewer} ist, dessen ContentProvider vom Typ
	 * {@link AbstractParamPluginContentProvider} ist. Der Name des Jobs wird
	 * entsprechend dynamisch ermittelt.
	 *
	 * @param paramPluginView
	 *            die Ansicht, deren Inhalt neu zu laden ist.
	 * @param doRefreshViewer
	 *            Bestimmt, ob der Viewer aktualisiert werden soll oder nicht.
	 */
	public ReloadCurrentInputJob(final AbstractParamPluginView paramPluginView, final boolean doRefreshViewer) {
		super("Ansicht '" + paramPluginView.getPartName() + "' aktualisieren");
		Assert.isTrue(paramPluginView.getSite().getSelectionProvider() instanceof ContentViewer);
		final ContentViewer viewer = (ContentViewer) paramPluginView.getSite().getSelectionProvider();
		Assert.isTrue(viewer.getContentProvider() instanceof AbstractParamPluginContentProvider);
		this.contentProvider = (AbstractParamPluginContentProvider) viewer.getContentProvider();
		this.doRefreshViewer = doRefreshViewer;
	}

	@Override
	protected IStatus run(final IProgressMonitor monitor) {
		IStatus status = Status.OK_STATUS;
		try {
			contentProvider.reloadCurrentInput(doRefreshViewer);
		} catch (final ParameterClientException e) {
			final String msg = getName() + " ist fehlgeschlagen";
			ParamPlugin.getDefault().getLogger().error(msg, e);
			status = new Status(IStatus.ERROR, ParamPlugin.PLUGIN_ID, msg, e);
		} catch (final IllegalStateException e) {
			final String msg = getName() + " ist fehlgeschlagen";
			ParamPlugin.getDefault().getLogger().error(msg, e);
			status = new Status(IStatus.ERROR, ParamPlugin.PLUGIN_ID, msg, e);
		}
		return status;
	}

}
