/*
 * Rahmenwerk-Plug-in "Parametrierung"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.param.lib;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.Set;

import de.bsvrz.dav.daf.main.ClientDavInterface;
import de.bsvrz.dav.daf.main.Data;
import de.bsvrz.dav.daf.main.config.AttributeGroup;
import de.bsvrz.dav.daf.main.config.DataModel;
import de.bsvrz.dav.daf.main.config.ObjectTimeSpecification;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.dav.daf.main.config.SystemObjectType;
import de.bsvrz.puk.param.lib.Parameter;
import de.bsvrz.puk.param.lib.ParameterInfo;
import de.bsvrz.puk.param.lib.daten.DataWithTime;
import de.bsvrz.sys.funclib.dataSerializer.Deserializer;
import de.bsvrz.sys.funclib.dataSerializer.NoSuchVersionException;
import de.bsvrz.sys.funclib.dataSerializer.SerializingFactory;
import de.bsvrz.sys.funclib.debug.Debug;

/**
 * Klasse zum Auslesen der Standardparameter von Systemobjekten der
 * Datenverteiler-Konfiguration.
 *
 * @author BitCtrl Systems GmbH, Peuker
 *
 */
class DefaultParameterProvider {

	/** Logger für Debugausgaben. */
	private static final Debug LOGGER = Debug.getLogger();

	/** die Attributgruppe, in der die Standardparameter abgelegt sind. */
	private final AttributeGroup defaultParameterAtg;

	/** das verwendete Datenmodell. */
	private final DataModel dataModel;

	/**
	 * Konstruktor, erzeugt eine Instanz der Klasse für die angegebene
	 * Datenverteilerverbindung.
	 *
	 * @param verbindung
	 *            die verwendete Datenverteilerverbindung
	 * @param preload
	 *            true - Datensätze für alle Objekte mit einem Funktionsaufruf
	 *            abgeholen und lokal zwischenspeichern
	 */
	DefaultParameterProvider(final ClientDavInterface verbindung, final boolean preload) {

		LOGGER.info("Standardparameter-Provider wird gestartet");
		dataModel = verbindung.getDataModel();

		LOGGER.info("Ermittle die Attributgruppe zur Definition der Standardparameter");
		defaultParameterAtg = dataModel.getAttributeGroup(ParaKonstanten.PID_ATG_DEFAULT_PARAMETERDATENSAETZE);

		if (preload) {
			/*
			 * Es werden die Datensätze für alle Objekte mit einem Funktionsaufruf abgeholt
			 * und lokal zwischengespeichert
			 */
			LOGGER.info("Ermittle alle gültigen Objekte");
			final Collection<SystemObject> objekte = dataModel.getObjects(null, null, ObjectTimeSpecification.valid());
			LOGGER.info("Lies die konfigurierenden Daten für alle Standardparameter");
			dataModel.getConfigurationData(objekte, defaultParameterAtg);
		}
	}

	/**
	 * liefert die Supertypen zu den in der übergeben Liste enthaltenen.
	 *
	 * @param typen
	 *            die Typen, die erweitert werden solln
	 * @return die neue Liste
	 */
	private Set<SystemObjectType> expandTypListe(final Set<SystemObjectType> typen) {
		final Set<SystemObjectType> neueListe = new HashSet<>();
		for (final SystemObjectType type : typen) {
			neueListe.addAll(type.getSuperTypes());
		}
		return neueListe;
	}

	/**
	 * liefert die Standardparameter für die übergebene Parameterspezifikation.
	 *
	 * @param info
	 *            die Spezifikation
	 * @return der Parameterdatensatz oder <i>null</i>, wenn keiner gefunden wurde
	 */
	Parameter getDefaultParameter(final ParameterInfo info) {

		LOGGER.finer("Ermittle DefaultParameter für: ", info);

		Parameter result = null;

		final Parameter[] definitionen = getDefinitionen(info.getObjekt());
		for (final Parameter parameter : definitionen) {

			Set<SystemObjectType> typen = new HashSet<>();
			typen.add(info.getTyp());

			while (typen.size() > 0) {
				if (typen.contains(parameter.getTyp())) {
					if (parameter.getAtg().equals(info.getAtg())) {
						result = parameter;
						break;
					}
				}
				typen = expandTypListe(typen);
			}
			if (result != null) {
				break;
			}
		}

		return result;
	}

	/**
	 * liefert die Definitionen für das angegebene Objekt.
	 *
	 * @param objekt
	 *            das Systemobjekt
	 * @return die Liste der ermittelten Parameter
	 */
	private Parameter[] getDefinitionen(final SystemObject objekt) {
		final Collection<Parameter> result = new ArrayList<>();

		final Data daten = objekt.getConfigurationData(defaultParameterAtg);
		if (daten != null) {
			LOGGER.finer("Standardparameterdatensätze ermittelt", objekt);

			final Data.Array array = daten.getArray("Default-Parameterdatensatz");
			for (int idx = 0; idx < array.getLength(); idx++) {
				final Data eintrag = array.getItem(idx);
				SystemObjectType typ = (SystemObjectType) eintrag.getReferenceValue("typ").getSystemObject();
				if (typ == null) {
					typ = objekt.getType();
				}
				final AttributeGroup attributGruppe = (AttributeGroup) eintrag.getReferenceValue("attributgruppe")
						.getSystemObject();
				final int serialisiererVersion = eintrag.getUnscaledValue("serialisierer").intValue();
				final byte[] datenBytes = eintrag.getUnscaledArray("datensatz").getByteArray();

				if (datenBytes != null) {
					try {
						final Deserializer deserializer = SerializingFactory.createDeserializer(serialisiererVersion,
								new ByteArrayInputStream(datenBytes));
						final Data parameterDaten = deserializer.readData(attributGruppe, dataModel);
						final Parameter parameter = new Parameter(
								new ParameterInfo(objekt, typ, attributGruppe, (short) 0));
						parameter.setDataWithTime(new DataWithTime(parameterDaten, System.currentTimeMillis()));
						result.add(parameter);
						LOGGER.finest("Daten ermittelt", parameter);
					} catch (final NoSuchVersionException e) {
						throw new RuntimeException(e);
					} catch (final IOException e) {
						throw new RuntimeException(e);
					}
				}

			}
		} else {
			LOGGER.finer("Keine Standarddaten konfiguriert", objekt);
		}

		return result.toArray(new Parameter[result.size()]);
	}
}
