/*
 * Rahmenwerk-Plug-in "Parametrierung"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.param.lib;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import de.bsvrz.buv.plugin.param.ModifiableParameterInfo;
import de.bsvrz.buv.plugin.param.ParametertypenCache;
import de.bsvrz.buv.plugin.param.Zeichenketten;
import de.bsvrz.buv.plugin.param.internal.RahmenwerkService;
import de.bsvrz.dav.daf.main.ClientDavInterface;
import de.bsvrz.dav.daf.main.Data;
import de.bsvrz.dav.daf.main.DataDescription;
import de.bsvrz.dav.daf.main.ResultData;
import de.bsvrz.dav.daf.main.archive.ArchiveData;
import de.bsvrz.dav.daf.main.archive.ArchiveDataKind;
import de.bsvrz.dav.daf.main.archive.ArchiveDataKindCombination;
import de.bsvrz.dav.daf.main.archive.ArchiveDataQueryResult;
import de.bsvrz.dav.daf.main.archive.ArchiveDataSpecification;
import de.bsvrz.dav.daf.main.archive.ArchiveDataStream;
import de.bsvrz.dav.daf.main.archive.ArchiveOrder;
import de.bsvrz.dav.daf.main.archive.ArchiveQueryPriority;
import de.bsvrz.dav.daf.main.archive.ArchiveRequestManager;
import de.bsvrz.dav.daf.main.archive.ArchiveRequestOption;
import de.bsvrz.dav.daf.main.archive.ArchiveTimeSpecification;
import de.bsvrz.dav.daf.main.archive.TimingType;
import de.bsvrz.dav.daf.main.config.Aspect;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.dav.daf.main.config.SystemObjectType;
import de.bsvrz.puk.param.lib.MethodenBibliothek;
import de.bsvrz.puk.param.lib.Parameter;
import de.bsvrz.puk.param.lib.ParameterClientException;
import de.bsvrz.puk.param.lib.ParameterInfo;
import de.bsvrz.puk.param.lib.ParameterManager;
import de.bsvrz.puk.param.lib.daten.DataWithTime;
import de.bsvrz.sys.funclib.debug.Debug;

/**
 * Methoden zur verallgemeinerten Arbeit mit Parametern, könnten später in die
 * Methodenbibliothek integriert werden.
 *
 * @author BitCtrl Systems GmbH, Albrecht Uhlmann
 */
public class ParameterManagerAllgemein {

	/**
	 * Debug-Logger für Ausgaben.
	 */
	private static final Debug DEBUG = Debug.getLogger();

	/**
	 * die Funktion füllt eine Parameterdatenfeld, falls ein Abruf zuvor ohne
	 * Datenanforderung erfolgt ist.
	 *
	 * @param parameters
	 *            das originale Array
	 * @return die Array mit den gefüllten Parametern
	 * @throws ParameterClientException
	 *             der Abruf der Daten war nicht erfolgreich
	 */
	public static Parameter[] getFilledParameterArray(final Parameter[] parameters) throws ParameterClientException {
		Parameter[] result = null;

		if (parameters != null) {
			result = new Parameter[parameters.length];
			int idx = 0;
			final List<ParameterInfo> reqList = new ArrayList<>();
			final List<Integer> idxList = new ArrayList<>();
			for (final Parameter parameter : parameters) {
				if ((parameter.getData() == null) && parameter.isDataAvailable()) {
					reqList.add(parameter.getInfo());
					idxList.add(idx);
				} else {
					result[idx] = parameter;
				}
				idx++;
			}
			final Parameter[] parameterResults = MethodenBibliothek
					.getParameterManager(RahmenwerkService.getService().getRahmenWerk().getDavVerbindung())
					.getParameter(reqList.toArray(new ParameterInfo[reqList.size()]));
			idx = 0;
			for (final Parameter parameterResult : parameterResults) {
				result[idxList.get(idx++)] = parameterResult;
			}
		}
		return result;
	}

	/**
	 * Liefert das Singleton und erzeugt es, falls es noch nicht existiert.
	 *
	 * @return der Manager
	 */
	public static ParameterManagerAllgemein getInstanz() {
		if (null == ParameterManagerAllgemein.instanz) {
			ParameterManagerAllgemein.instanz = new ParameterManagerAllgemein();
		}
		return ParameterManagerAllgemein.instanz;
	}

	/**
	 * Liefert eine textliche Repräsentation der im Parameter enthaltenen Daten. Im
	 * Prinzip p.getData().toString(), mit folgenden Verfeinerungen:
	 * <ul>
	 * <li>Die atl.urlasser wird übersprungen</li>
	 * <li>Falls p.getData() == null, wird ein entsprechender String
	 * zurückgeliefert.</li>
	 * </ul>
	 *
	 * @param p
	 *            der Parameter
	 * @return der Text. Niemals <code>null</code>
	 */
	public static String getParameterDataSafeOhneUrlasserListe(final Parameter p) {
		String text;
		final Data data = p.getData();
		if (null == data) {
			if (p.isDataAvailable()) {
				text = "Daten nicht abgerufen";
			} else {
				text = "Keine Daten oder keine Quelle";
			}
		} else {
			if (data.isPlain()) {
				text = p.getData().toString();
			} else {
				final StringBuilder b = new StringBuilder();
				final Iterator<?> iterator = data.iterator();
				int loop = 0;
				while (iterator.hasNext()) {
					final Data next = (Data) iterator.next();
					if ("Urlasser".equals(next.getName())) {
						continue;
					}
					if (loop > 0) {
						b.append(", ");
					}
					b.append(next.toString());
					++loop;
				}
				text = b.toString();
			}
		}
		return text;
	}

	/**
	 * Objekt zum Bereitstellen der Default-Parameter.
	 */
	private DefaultParameterProvider defaultParameterProvider;

	/**
	 * Das Singleton.
	 */
	private static ParameterManagerAllgemein instanz;

	/**
	 * Bestimmt, ob zwei Parameter gleich aussehen. Es wird die ParameterInfo
	 * verglichen, aber nicht die Data! <code>null</code>-Werte sind nicht zulässig.
	 *
	 * @param p1
	 *            Erster Parameter
	 * @param p2
	 *            Zweiter Parameter
	 * @return true - beide Parameter verweisen auf dieselbe Kombination aus
	 *         ATG-OBJ-TYP-SIM
	 */
	public static boolean parameterEquals(final Parameter p1, final Parameter p2) {
		boolean result = true;
		if (p1.getSim() != p2.getSim()) {
			result = false;
		} else {
			if (!p1.getAtg().equals(p2.getAtg())) {
				result = false;
			} else {
				if (!p1.getObjekt().equals(p2.getObjekt())) {
					result = false;
				} else {
					if (!p1.getTyp().equals(p2.getTyp())) {
						result = false;
					}
				}
			}
		}
		return result;
	}

	/**
	 * Verbindung zum DaV.
	 */
	private ClientDavInterface verbindung;

	/**
	 * Liefert die bei einer Parameterändrung des durch resolvedMpi beschriebenen
	 * Objektes potentiell betroffenen Objekte.
	 *
	 * @param resolvedMpi
	 *            Beschreibung eines Parameters mit genau einem Typ
	 * @return die Liste Wirft eine IllegalStateException, wenn die Parametrierung
	 *         die Werte nicht bereitstellen konnte
	 */
	public Collection<SystemObject> getAenderbareObjekteAllgemein(final ModifiableParameterInfo resolvedMpi) {
		if (null == verbindung) {
			throw new IllegalStateException(Zeichenketten.PLUGIN_PARAM_BEZEICHNER_KEINE_DAVVERBINDUNG);
		}
		final ParameterInfo info = resolvedMpi.toParameterInfos()[0];
		final ParameterManager hierarchie = MethodenBibliothek.getParameterManager(verbindung);
		ParameterManagerAllgemein.DEBUG.config("Erfrage änderbare Objekte für", info);
		final Collection<SystemObject> aenderbareObjekte = hierarchie.getAenderbareObjekte(info);
		if (null == aenderbareObjekte) {
			throw new IllegalStateException("Änderbare Objekte nicht ermittelbar");
		}
		ParameterManagerAllgemein.DEBUG.config("Ermittelte " + aenderbareObjekte.size() + " änderbare Objekte für",
				info);
		if (aenderbareObjekte.isEmpty() && !(info.getObjekt() instanceof SystemObjectType)) {
			final Collection<SystemObject> aenderbareObjektePatched = new ArrayList<>();
			aenderbareObjektePatched.add(info.getObjekt());
			ParameterManagerAllgemein.DEBUG.config("PATCHED mit self");
			return aenderbareObjektePatched;
		}
		return aenderbareObjekte;
	}

	/**
	 * Liefert die Default-Parameter für die spezifizierten Objekte. Wirft eine
	 * IllegalStateException, wenn keine Verbindung zum DaV besteht.
	 *
	 * @param infos
	 *            die Spezifikation(en)
	 * @return das Feld mit den Parametern. Es wird stets ein Feld zurückgeliefert,
	 *         welches ebenso groß wie das übergebene Infos-Feld ist und in jedem
	 *         Element ein Objekt mit einem {@link Data}-Objekt enthält. Allerdings
	 *         können einige oder gar alle der Data-Objekte unvollständig sein,
	 *         nämlich genau dann, wenn keine Defaultparameter ermittelt werden
	 *         konnten. In dem Fall wird über
	 *         {@link ClientDavInterface#createData(de.bsvrz.dav.daf.main.config.AttributeGroup)}
	 *         ein Datensatz erzeugt, aber nicht ausgefüllt.
	 */
	public Parameter[] getDefaultParameter(final ParameterInfo... infos) {
		final Parameter[] parameters = new Parameter[infos.length];
		if (null == verbindung) {
			throw new IllegalStateException(Zeichenketten.PLUGIN_PARAM_BEZEICHNER_KEINE_DAVVERBINDUNG);
		}
		if (null == defaultParameterProvider) {
			defaultParameterProvider = new DefaultParameterProvider(verbindung, false);
		}
		int loop;
		for (loop = 0; loop < infos.length; ++loop) {
			Parameter p = defaultParameterProvider.getDefaultParameter(infos[loop]);
			if (null == p) {
				p = new Parameter(infos[loop]);
				final Data data = verbindung.createData(infos[loop].getAtg());
				p.setDataWithTime(new DataWithTime(data, System.currentTimeMillis()));
			}
			parameters[loop] = p;
		}
		return parameters;
	}

	/**
	 * Liefert die Ist-Parameter für die durch infos spezifizierte Objekte. Wirft
	 * eine IllegalStateException, wenn keine Verbindung zum DaV besteht.
	 *
	 * @param infos
	 *            die Spezifikation(en)
	 * @return das Feld mit den Parametern. Es wird stets ein Feld zurückgeliefert,
	 *         welches ebenso groß wie das übergebene Infos-Feld ist und in jedem
	 *         Element ein Objekt mit einem {@link Data}-Objekt enthält. Allerdings
	 *         können einige oder gar alle der Data-Objekte unvollständig sein,
	 *         nämlich genau dann, wenn keine Isttparameter ermittelt werden
	 *         konnten. Dies kann passieren, wenn an dem fraglichen Objekt der
	 *         Ist-Parameter-Aspekt gar nicht definerit ist (z.B. bei Typen oder
	 *         Hierarchieobjekten) oder aktuell keine Daten publiziert wurden. In
	 *         dem Fall wird über
	 *         {@link ClientDavInterface#createData(de.bsvrz.dav.daf.main.config.AttributeGroup)}
	 *         ein Datensatz erzeugt, aber nicht ausgefüllt.
	 */
	public Parameter[] getIstParameter(final ParameterInfo... infos) {
		final Parameter[] parameters = new Parameter[infos.length];
		if (null == verbindung) {
			throw new IllegalStateException(Zeichenketten.PLUGIN_PARAM_BEZEICHNER_KEINE_DAVVERBINDUNG);
		}
		int loop;
		final Aspect istAspect = verbindung.getDataModel().getAspect("asp.parameterIst");
		for (loop = 0; loop < infos.length; ++loop) {
			final ParameterInfo info = infos[loop];
			final Set<SystemObjectType> realParents = ParametertypenCache.getInstanz().getRealParents(info.getAtg());
			Data istData = null;
			// Nach meinem Verständnis ist es möglich, dass eine parametrierende
			// Attributgruppe den Ist-Aspakt nicht hat, daher der erste Test zum
			// Vermeiden einer Runtime-Exception aus den DAF. Der zweite Test
			// ist nötig, weil info.getObjekt() auch Typen oder
			// Hierarchieobjekte enthalten kann, an denen die ATG und somit auch
			// der Ist-Aspekt gar nicht direkt definiert ist.
			if (info.getAtg().getAspects().contains(istAspect) && realParents.contains(info.getObjekt().getType())) {
				final DataDescription dataDescription = new DataDescription(info.getAtg(), istAspect, info.getSim());
				final ResultData result = verbindung.getData(info.getObjekt(), dataDescription, 60000L);
				istData = result.getData();
			}
			if (null == istData) {
				istData = verbindung.createData(info.getAtg());
			}
			parameters[loop] = new Parameter(info, new DataWithTime(istData, System.currentTimeMillis()));
		}
		return parameters;
	}

	/**
	 * Liefert Parameter gemäß Beschreibung, entweder Online-Parameter (dann wird
	 * die {@link MethodenBibliothek} eingesetzt) oder historische (dann wird das
	 * Archiv befragt).
	 *
	 * @param resolvedMpi
	 *            Beschreibung eines Parameters mit genau einem Typ
	 * @param aenderbareObjekte
	 *            die Liste der Objekte, für die Parameter gemäß Beschreibung zu
	 *            liefern sind
	 * @param readData
	 *            legt fest, ob Daten mit den Parameterinformationen abgerufen
	 *            werden sollen
	 * @return das Feld mit den Parametern
	 * @throws ParameterClientException
	 *             - Wenn die Parametrierung bzw. das Archiv die Werte nicht
	 *             bereitstellen konnte
	 *
	 *             IllegalStateException - keine Verbindung zum DaV, kein Archiv,
	 *             ...
	 */
	public Parameter[] getParameterAllgemein(final ModifiableParameterInfo resolvedMpi,
			final Collection<SystemObject> aenderbareObjekte, final boolean readData) throws ParameterClientException {
		if (null == verbindung) {
			throw new IllegalStateException(Zeichenketten.PLUGIN_PARAM_BEZEICHNER_KEINE_DAVVERBINDUNG);
		}
		final ParameterManager hierarchie = MethodenBibliothek.getParameterManager(verbindung);
		final Parameter[] einzelParameterIst;
		if (resolvedMpi.isHistorisch()) {
			final ArchiveRequestManager archiv = verbindung.getArchive();
			if (!archiv.isArchiveAvailable()) {
				throw new IllegalStateException("Archivsystem ist nicht verfügbar!");
			}
			final DataDescription dataDescription = new DataDescription(resolvedMpi.getAtg(),
					verbindung.getDataModel().getAspect("asp.parameterSoll"));
			final ArchiveTimeSpecification timeSpec = new ArchiveTimeSpecification(TimingType.DATA_TIME, true, 1,
					RahmenwerkService.getService().getArtDesParametersatzesProvider().getZeitpunkt());
			final ArchiveDataKindCombination dataKinds = new ArchiveDataKindCombination(ArchiveDataKind.ONLINE);
			final ArchiveRequestOption requestOption = ArchiveRequestOption.NORMAL;
			final List<ArchiveDataSpecification> specs = new ArrayList<>(aenderbareObjekte.size());
			for (final SystemObject o : aenderbareObjekte) {
				final ArchiveDataSpecification spec = new ArchiveDataSpecification(timeSpec, dataKinds,
						ArchiveOrder.BY_DATA_TIME, requestOption, dataDescription, o);
				specs.add(spec);
			}
			final ArchiveDataQueryResult antwort = archiv.request(ArchiveQueryPriority.MEDIUM, specs);
			final List<Parameter> abgerufeneParameter = new ArrayList<>(aenderbareObjekte.size());
			try {
				for (final ArchiveDataStream strom : antwort.getStreams()) {
					ArchiveData archivData = strom.take();
					final ParameterInfo einzelInfo = new ParameterInfo(strom.getDataSpecification().getObject(),
							resolvedMpi.getAtg(), resolvedMpi.getSim());
					final Parameter p = new Parameter(einzelInfo);
					abgerufeneParameter.add(p);
					/*
					 * Eigentlich sollte es reichen, den ersten Datensatz aus dem Strom zu
					 * entnehmen.
					 */
					while ((null != archivData) && (null != archivData.getData())
							&& (archivData.getDataTime() <= timeSpec.getIntervalEnd())) {
						p.setDataWithTime(new DataWithTime(archivData.getData(), archivData.getDataTime()));
						archivData = strom.take();
					}
				}
			} catch (final IllegalStateException e) {
				throw e;
			} catch (final InterruptedException e) {
				throw new IllegalStateException(e);
			} catch (final IOException e) {
				throw new IllegalStateException(e);
			}
			einzelParameterIst = abgerufeneParameter.toArray(new Parameter[abgerufeneParameter.size()]);
		} else {
			final ParameterInfo[] einzelInfos = new ParameterInfo[aenderbareObjekte.size()];
			int einzelInfosLoop = 0;
			for (final SystemObject o : aenderbareObjekte) {
				einzelInfos[einzelInfosLoop++] = new ParameterInfo(o, resolvedMpi.getAtg(), resolvedMpi.getSim());
			}
			try {
				einzelParameterIst = hierarchie.getParameter(readData, einzelInfos);
			} catch (final ParameterClientException e) {
				throw e;
			}
		}
		return einzelParameterIst;
	}

	/**
	 * Liefert die Datenverteilerverbindung (kann null sein).
	 *
	 * @return die DaV-Verbindung oder null
	 */
	public ClientDavInterface getVerbindung() {
		return verbindung;
	}

	/**
	 * Setzt die Datenverteilerverbindung (null übergeben, wenn DaV gerade nicht da
	 * ist, die Methoden werfen dann eine {@link IllegalStateException}.
	 *
	 * @param verbindung
	 *            die DaV-Verbindung oder null
	 */
	public void setVerbindung(final ClientDavInterface verbindung) {
		this.verbindung = verbindung;
	}
}
