/*
 * Rahmenwerk-Plug-in "Parametrierung"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.param.provider;

import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeSupport;

import org.eclipse.jface.viewers.ContentViewer;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.IStructuredContentProvider;
import org.eclipse.jface.viewers.Viewer;

import de.bsvrz.buv.plugin.param.views.AbstractParamPluginView;
import de.bsvrz.puk.param.lib.ParameterClientException;

/**
 * Abstrakter Contentprovider für einen Viewer im Plug-In Parametrierung.
 *
 * @author BitCtrl Systems GmbH, Albrecht Uhlmann
 */
public abstract class AbstractParamPluginContentProvider
		implements IStructuredContentProvider, ISelectionChangedListener {

	/** Name der Property, die das Flag <em>kurzinfo</em> hält. */
	public static final String PROP_KURZINFO = "kurzinfo";

	/**
	 * Der Viewer, den wir bedienen. Kann ein TableViewer oder ein TreeViewer sein.
	 */
	private ContentViewer contentViewer;

	/**
	 * der View, in dem wir uns befinden.
	 */
	private final AbstractParamPluginView paramPluginView;

	/**
	 * Kann zum feuern von {@link java.beans.PropertyChangeEvent}s verwendet werden.
	 */
	private final PropertyChangeSupport propertyChangeSupport = new PropertyChangeSupport(this);

	/**
	 * die aktuelle Kurzinfo.
	 */
	private String kurzinfo = "";

	/**
	 * Konstruktor reicht durch und merkt sich die Ansicht, in der wir uns
	 * befiinden.
	 *
	 * @param paramPluginView
	 *            der View, in dem wir uns befinden.
	 */
	public AbstractParamPluginContentProvider(final AbstractParamPluginView paramPluginView) {
		super();
		this.paramPluginView = paramPluginView;
	}

	@Override
	public void dispose() {
		if (contentViewer != null) {
			contentViewer.removeSelectionChangedListener(this);
		}
	}

	@Override
	public void inputChanged(final Viewer viewer, final Object oldInput, final Object newInput) {
		if (viewer instanceof ContentViewer) {
			if (contentViewer != viewer) {
				if (contentViewer != null) {
					contentViewer.removeSelectionChangedListener(this);
				}
				contentViewer = (ContentViewer) viewer;
				contentViewer.addSelectionChangedListener(this);
			}
		}

	}

	/**
	 * Das bestehende Input-Objekt neu laden.
	 *
	 * @param doRefreshViewer
	 *            true Nach dem Neuladen den Viewer aktualisieren
	 * @throws ParameterClientException
	 *             Fehler bei Kommunikation mit Parametrierungs-Applikation
	 */
	public abstract void reloadCurrentInput(final boolean doRefreshViewer) throws ParameterClientException;

	/**
	 * Aktualisiert den Viewer über eine SWT Runnable. Die Methode kann also von
	 * jedem Thread aus aufgerufen werden.
	 */
	protected void refreshViewer() {
		if ((contentViewer != null) && !contentViewer.getControl().isDisposed()
				&& !contentViewer.getControl().getDisplay().isDisposed()) {
			contentViewer.getControl().getDisplay().asyncExec(new Runnable() {

				@Override
				public void run() {
					if (!contentViewer.getControl().isDisposed()
							&& !contentViewer.getControl().getDisplay().isDisposed()
							&& (contentViewer.getInput() != null)) {
						/* HINWEIS Auswahl wiederherstellen, wenn möglich */
						contentViewer.refresh();
					}
				}
			});
		}
	}

	/**
	 * Hinzufügen eines Listeners für alle Properties.
	 *
	 * @param listener
	 *            der neue Listener.
	 */
	public void addPropertyChangeListener(final PropertyChangeListener listener) {
		propertyChangeSupport.addPropertyChangeListener(listener);
	}

	/**
	 * Hinzufügen eines Listeners für eine bestimmte Property.
	 *
	 * @param propertyName
	 *            Name der Property, für die angemeldet werden soll.
	 * @param listener
	 *            der neue Listener.
	 */
	public void addPropertyChangeListener(final String propertyName, final PropertyChangeListener listener) {
		propertyChangeSupport.addPropertyChangeListener(propertyName, listener);
	}

	/**
	 * Entfernen eines Listeners für alle Properties.
	 *
	 * @param listener
	 *            der Listener.
	 */
	public void removePropertyChangeListener(final PropertyChangeListener listener) {
		propertyChangeSupport.removePropertyChangeListener(listener);
	}

	/**
	 * Entfernen eines Listeners für eine bestimmte Property.
	 *
	 * @param propertyName
	 *            Name der Property, für die abgemeldet werden soll.
	 * @param listener
	 *            der Listener.
	 */
	public void removePropertyChangeListener(final String propertyName, final PropertyChangeListener listener) {
		propertyChangeSupport.removePropertyChangeListener(propertyName, listener);
	}

	/**
	 * Liefert die Kurzinfo, ein String, der den aktuellen Inhalt des
	 * Contentproviders in wenigen Zeilen zusammenfasst.
	 *
	 * @return die Kurzinfo. Niemals <code>null</code>.
	 */
	public String getKurzinfo() {
		return kurzinfo;
	}

	/*
	 * (non-Javadoc)
	 *
	 * @see java.lang.Object#toString()
	 */
	@Override
	public String toString() {
		return getKurzinfo();
	}

	/**
	 * Setzt die Kurzinfo, ein String, der den aktuellen Inhalt des Contentproviders
	 * in wenigen Zeilen zusammenfasst.
	 *
	 * @param kurzinfo
	 *            die neue Kurzinfo. Wenn <code>null</code> übergeben wird, passiert
	 *            gar nichts, ein leerer String ist aber erlaubt. Wenn die
	 *            übergebene Kurzifo ungleich der alten ist, wird ein
	 *            {@link AbstractParamPluginContentProvider#PROP_KURZINFO} Event
	 *            gefeuert.
	 */
	protected void setKurzinfo(final String kurzinfo) {
		if ((null != kurzinfo) && !kurzinfo.equals(this.kurzinfo)) {
			final String oldValue = this.kurzinfo;
			this.kurzinfo = kurzinfo;
			propertyChangeSupport.firePropertyChange(AbstractParamPluginContentProvider.PROP_KURZINFO, oldValue,
					kurzinfo);

		}
	}

	/**
	 * Liefert den Viewer, den wir bedienen. Kann ein TableViewer oder ein
	 * TreeViewer sein.
	 *
	 * @return der Viewer
	 */
	protected final ContentViewer getContentViewer() {
		return contentViewer;
	}

	/**
	 * Liefert den View, in dem wir uns befinden.
	 *
	 * @return der View
	 */
	protected final AbstractParamPluginView getParamPluginView() {
		return paramPluginView;
	}
}
