/*
 * Rahmenwerk-Plug-in "Parametrierung"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.param.provider;

import org.eclipse.jface.viewers.ITableLabelProvider;
import org.eclipse.swt.graphics.Image;

import de.bsvrz.buv.plugin.param.IArtDesParametersatzesProvider;
import de.bsvrz.buv.plugin.param.internal.RahmenwerkService;
import de.bsvrz.puk.param.lib.Parameter;

/**
 * Abstrakte Basisklasse für einen Tabellen-Labelprovider, der Objekte vom Typ
 * {@link Parameter} behandeln kann. Es gibt Default-Handler für alle möglichen
 * Member eines solchen Objekts. Ob diese in einer konkreten Tabelle verwendet
 * werden und wenn ja, an welcher Position (columnIndex) entscheidet die
 * implementierende Kindklasse, welche ein Mapping zwischen columnIndex und
 * Inhalt bereitstellen muss.
 *
 * HINWEIS Weitere Spalten BenutzerReferenz, Ursache und Urlasser einführen und
 * diese mit den entsprechenden Werten aus der atl.urlasser füllen
 *
 * HINWEIS Aus der Spalte Rohdaten die atl.urlasser entfernen
 *
 * HINWEIS Eine Spalte Zeitstempel einführen und dort den Zeitstempel des
 * Datensatzes eintragen (Falls das geht)
 *
 * @author BitCtrl Systems GmbH, Albrecht Uhlmann
 */
public abstract class AbstractParamPluginTableLabelProvider extends AbstractParamPluginLabelProvider
		implements ITableLabelProvider {

	/**
	 * Spalte 'Objekt'.
	 */
	public static final String COLUMNID_OBJECT_NAME = "Objekt";

	/**
	 * Spalte 'Objekt-PID'.
	 */
	public static final String COLUMNID_OBJECT_PID = "Objekt-PID";

	/**
	 * Spalte 'Attributgruppe'.
	 */
	public static final String COLUMNID_ATG_NAME = "Attributgruppe";

	/**
	 * Spalte 'Attributgruppen-PID'.
	 */
	public static final String COLUMNID_ATG_PID = "Attributgruppen-PID";

	/**
	 * Spalte 'Typ'.
	 */
	public static final String COLUMNID_TYP_NAME = "Typ";

	/**
	 * Spalte 'Typ-PID'.
	 */
	public static final String COLUMNID_TYP_PID = "Typ-PID";

	/**
	 * Spalte 'Simulationsvariante'.
	 */
	public static final String COLUMNID_SIM = "Simulationsvariante";

	/**
	 * Spalte 'Parameterquelle'.
	 */
	public static final String COLUMNID_QUELLE = "Parameterquelle";

	/**
	 * Spalte 'Typ der Quelle'.
	 */
	public static final String COLUMNID_QUELLTYP = "Typ der Quelle";

	/**
	 * Bevorzugte Breite der Spalte mit dem Objektnamen. Experimentell ermittelt
	 * unter Windows XP.
	 */
	private static final int PREFERRED_COLUMN_WIDTH_OBJECT_NAME = 250;

	/**
	 * Liefert einen String, der die Spalte mit Index columnIndex inhaltlich
	 * eindeutig beschreibt. Es muss einer der hier definierten Strings COLUMNID_*
	 * sein.
	 *
	 * @param columnIndex
	 *            der 0-basierte Index der Spalte
	 * @return der Identifier-String
	 */
	public abstract String getColumnId(int columnIndex);

	/**
	 * Liefert die Anzahl der Spalten, die die Kindklasse bedienen kann.
	 *
	 * @return die Spaltenzahl
	 */
	public abstract int getNumColumns();

	@Override
	public Image getColumnImage(final Object element, final int columnIndex) {
		// Keine Bilder zur Zeit
		return null;
	}

	@Override
	public String getColumnText(final Object element, final int columnIndex) {
		String text = "unbekannt";
		if (!(element instanceof Parameter)) {
			if (columnIndex == 0) {
				if (element == null) {
					text += ": kein Objekt übergeben";
				} else {
					text += "er Typ: " + element.getClass().getName();
				}
			} else {
				text = "";
			}
		} else {
			final Parameter p = (Parameter) element;
			final String columnId = getColumnId(columnIndex);
			if (AbstractParamPluginTableLabelProvider.COLUMNID_OBJECT_NAME.equals(columnId)) {
				text = p.getObjekt().toString();
			} else if (AbstractParamPluginTableLabelProvider.COLUMNID_OBJECT_PID.equals(columnId)) {
				text = p.getObjekt().getPidOrId();
			} else if (AbstractParamPluginTableLabelProvider.COLUMNID_ATG_NAME.equals(columnId)) {
				text = p.getAtg().toString();
			} else if (AbstractParamPluginTableLabelProvider.COLUMNID_ATG_PID.equals(columnId)) {
				text = p.getAtg().getPidOrId();
			} else if (AbstractParamPluginTableLabelProvider.COLUMNID_TYP_NAME.equals(columnId)) {
				text = p.getTyp().toString();
			} else if (AbstractParamPluginTableLabelProvider.COLUMNID_TYP_PID.equals(columnId)) {
				text = p.getTyp().getPidOrId();
			} else if (AbstractParamPluginTableLabelProvider.COLUMNID_SIM.equals(columnId)) {
				final IArtDesParametersatzesProvider backend = RahmenwerkService.getService()
						.getArtDesParametersatzesProvider();
				text = backend.getSimulationsVarianteString(p.getSim());
			} else if (AbstractParamPluginTableLabelProvider.COLUMNID_QUELLE.equals(columnId)) {
				text = p.getQuelle().getObjekt().toString();
			} else if (AbstractParamPluginTableLabelProvider.COLUMNID_QUELLTYP.equals(columnId)) {
				text = p.getQuelle().getTyp().toString();
			}
		}
		return text;
	}

	/**
	 * Liefert die bevorzugte Breite der Spalte.
	 *
	 * @param columnId
	 *            die Id der Spalte ({@link #COLUMNID_OBJECT_NAME} and friends.
	 *
	 * @return die Breite
	 */
	public int getPreferredColumnWidth(final String columnId) {
		int text = AbstractParamPluginTableLabelProvider.PREFERRED_COLUMN_WIDTH_OBJECT_NAME;
		if (AbstractParamPluginTableLabelProvider.COLUMNID_OBJECT_PID.equals(columnId)) {
			text = 200;
		} else if (AbstractParamPluginTableLabelProvider.COLUMNID_ATG_PID.equals(columnId)) {
			text = 150;
		} else if (AbstractParamPluginTableLabelProvider.COLUMNID_TYP_NAME.equals(columnId)) {
			text = 180;
		} else if (AbstractParamPluginTableLabelProvider.COLUMNID_TYP_PID.equals(columnId)) {
			text = 140;
		} else if (AbstractParamPluginTableLabelProvider.COLUMNID_SIM.equals(columnId)) {
			text = 150;
		} else if (AbstractParamPluginTableLabelProvider.COLUMNID_QUELLTYP.equals(columnId)) {
			text = 100;
		}
		return text;
	}
}
