/*
 * Rahmenwerk-Plug-in "Parametrierung"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.param.provider;

import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.sql.Timestamp;
import java.util.Iterator;
import java.util.List;

import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Cursor;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.PlatformUI;

import de.bsvrz.buv.plugin.param.IArtDesParametersatzesProvider;
import de.bsvrz.buv.plugin.param.internal.ParamPlugin;
import de.bsvrz.buv.plugin.param.internal.RahmenwerkService;
import de.bsvrz.buv.plugin.param.views.AbstractParamPluginView;
import de.bsvrz.dav.daf.main.DataDescription;
import de.bsvrz.dav.daf.main.archive.ArchiveAvailabilityListener;
import de.bsvrz.dav.daf.main.archive.ArchiveRequestManager;
import de.bsvrz.puk.param.lib.ParameterClientException;
import de.bsvrz.sys.funclib.bitctrl.modell.ObjektFactory;

/**
 * Contentprovider für einen Viewer vom Typ
 * {@link de.bsvrz.buv.plugin.param.viewer.ArtDesParametersatzesViewer}.
 *
 * @author BitCtrl Systems GmbH, Albrecht Uhlmann
 */
public class ArtDesParametersatzesContentProvider extends AbstractParamPluginContentProvider
		implements PropertyChangeListener, ArchiveAvailabilityListener {

	/**
	 * Aktuelles Input-Objekt.
	 */
	private IArtDesParametersatzesProvider currentInput;

	/**
	 * Konstruktor meldet sich als {@link PropertyChangeListener} am Bean
	 * {@link IArtDesParametersatzesProvider} an.
	 *
	 * @param paramPluginView
	 *            der View, in dem wir sind
	 */
	public ArtDesParametersatzesContentProvider(final AbstractParamPluginView paramPluginView) {
		super(paramPluginView);
		setKurzinfo("Hier wird bestimmt, ob aktuelle oder historische Parameter angezeigt\n"
				+ "werden sollen. Weiterhin wird gewählt, ob die Parameter des Echtsystems\n"
				+ "oder die einer Simulation angezeigt werden sollen");
	}

	@Override
	public Object[] getElements(final Object inputElement) {
		Object[] elements = null;
		if (IArtDesParametersatzesProvider.ART_SIMULATION.equals(inputElement)) {
			if (currentInput != null) {
				final List<Short> validSimulationVariants = currentInput.getValidSimlationsVarianten();
				validSimulationVariants.remove(Short.valueOf((short) 0));
				validSimulationVariants.remove(Short.valueOf(DataDescription.NO_SIMULATION_VARIANT_SET));
				elements = validSimulationVariants.toArray();
			} else {
				elements = new Object[0];
			}
		}
		return elements;
	}

	@Override
	public void inputChanged(final Viewer viewer, final Object oldInput, final Object newInput) {
		super.inputChanged(viewer, oldInput, newInput);
		final ObjektFactory factory = RahmenwerkService.getService().getObjektFactory();
		if (oldInput instanceof IArtDesParametersatzesProvider) {
			((IArtDesParametersatzesProvider) oldInput).removePropertyChangeListener(this);
		}
		if (newInput instanceof IArtDesParametersatzesProvider) {
			((IArtDesParametersatzesProvider) newInput).addPropertyChangeListener(this);
			currentInput = (IArtDesParametersatzesProvider) newInput;
			if (factory.isVerbunden()) {
				factory.getDav().getArchive().addArchiveAvailabilityListener(this);
			}
		} else {
			currentInput = null;
			if (factory.isVerbunden()) {
				factory.getDav().getArchive().addArchiveAvailabilityListener(this);
			}
		}
		refreshViewer();
	}

	@Override
	public void propertyChange(final PropertyChangeEvent evt) {
		// HINWEIS updateViewer(); aber zirkuläre Abläufe vermeiden
	}

	@Override
	public void selectionChanged(final SelectionChangedEvent event) {
		if (!event.getSelection().isEmpty() && (currentInput != null)) {
			final IStructuredSelection s = (IStructuredSelection) event.getSelection();
			final Iterator<?> iterator = s.iterator();
			final String art = (String) iterator.next();
			if (IArtDesParametersatzesProvider.ART_SIMULATION.equals(art) && iterator.hasNext()) {
				currentInput.setArtDesParametersatzes(art);
				final Short selectedSimulationsVariante = (Short) iterator.next();
				currentInput.setSelectedSimulationsVariante(selectedSimulationsVariante);
			} else if (IArtDesParametersatzesProvider.ART_HISTORISCH.equals(art) && iterator.hasNext()) {
				currentInput.setArtDesParametersatzes(art);
				currentInput.setSelectedSimulationsVariante(DataDescription.NO_SIMULATION_VARIANT_SET);
				final Timestamp timestamp = (Timestamp) iterator.next();
				currentInput.setZeitpunkt(timestamp.getTime());
			} else if (IArtDesParametersatzesProvider.ART_ORIGINAL.equals(art)) {
				currentInput.setArtDesParametersatzes(art);
				currentInput.setSelectedSimulationsVariante(DataDescription.NO_SIMULATION_VARIANT_SET);
			}
		}
	}

	@Override
	public void reloadCurrentInput(final boolean doRefreshViewer) throws ParameterClientException {
		// Default-Implementierung
		if (doRefreshViewer) {
			refreshViewer();
		}
	}

	/**
	 * Verfügbarkeitsänderung im Kontext des SWT UI Threads.
	 *
	 * @param archive
	 *            Archiv, dessen Zustand sich geändert hat.
	 * @param display
	 *            das Display, dessen UI-Thread verwendet werden soll.
	 */
	private void archiveAvailabilityChangedUI(final ArchiveRequestManager archive, final Display display) {
		final Shell shell = display.getActiveShell();
		Cursor waitCursor = null;
		try {
			if (null != currentInput) {
				if (null != shell) {
					waitCursor = new Cursor(Display.getDefault(), SWT.CURSOR_WAIT);
					shell.setCursor(waitCursor);
				}
				currentInput.setArchivAvailable(archive.isArchiveAvailable());
				reloadCurrentInput(true);
			}
		} catch (final ParameterClientException e) {
			// kann in diesem View eigentlich nicht passieren
			ParamPlugin.getDefault().getLogger().error(e.getLocalizedMessage(), e);
		} finally {
			if ((null != currentInput) && (null != shell)) {
				shell.setCursor(null);
			}
			if (null != waitCursor) {
				waitCursor.dispose();
				waitCursor = null;
			}
		}
	}

	@Override
	public void archiveAvailabilityChanged(final ArchiveRequestManager archive) {
		final Display display = PlatformUI.getWorkbench().getDisplay();
		display.asyncExec(new Runnable() {

			@Override
			public void run() {
				archiveAvailabilityChangedUI(archive, display);
			}
		});
	}
}
