/*
 * Rahmenwerk-Plug-in "Parametrierung"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.param.provider;

import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.util.Collection;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Set;

import org.eclipse.core.runtime.jobs.IJobChangeEvent;
import org.eclipse.core.runtime.jobs.JobChangeAdapter;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.ContentViewer;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Cursor;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.PlatformUI;

import de.bsvrz.buv.plugin.param.IArtDesParametersatzesProvider;
import de.bsvrz.buv.plugin.param.ModifiableParameterInfo;
import de.bsvrz.buv.plugin.param.ObjektAuswahlEnum;
import de.bsvrz.buv.plugin.param.Zeichenketten;
import de.bsvrz.buv.plugin.param.internal.ParamPlugin;
import de.bsvrz.buv.plugin.param.internal.RahmenwerkService;
import de.bsvrz.buv.plugin.param.jobs.AnsichtAktualisierenViaReloadCurrentInputJobRule;
import de.bsvrz.buv.plugin.param.jobs.ReloadCurrentInputJob;
import de.bsvrz.buv.plugin.param.lib.ParameterManagerAllgemein;
import de.bsvrz.buv.plugin.param.views.AbstractParamPluginView;
import de.bsvrz.dav.daf.main.DataDescription;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.puk.param.lib.Parameter;
import de.bsvrz.puk.param.lib.ParameterClientException;
import de.bsvrz.puk.param.lib.ParameterInfo;
import de.bsvrz.sys.funclib.debug.Debug;

/**
 * Contentprovider für die Ansicht "Betroffene Objekte".
 *
 * @author BitCtrl Systems GmbH, Albrecht Uhlmann
 */
public class BetroffeneObjekteContentProvider extends AbstractParamPluginContentProvider
		implements PropertyChangeListener {

	/**
	 * Default-Kurzinfo, falls keine Objekte geladen sind.
	 */
	private static final String DEFAULT_KURZINFO = Zeichenketten.PLUGIN_PARAM_BEZEICHNER_BETROFFENE_OBJEKTE
			+ ": Keine.";

	/**
	 * Aktueller Input, ein Feld von modifizierbaren Parameterinfos.
	 */
	private ModifiableParameterInfo[] currentInput;

	/**
	 * Die betroffenen Einzelparameter.
	 */
	private final Map<Parameter, ParameterInfo> betroffeneEinzelParameter = new LinkedHashMap<>();

	/**
	 * Die nicht änderbaren Parameter (haben bereits LOKAL_OBJEKT oder
	 * DEFAULT_OBJEKT).
	 */
	private final Map<Parameter, ParameterInfo> nichtAenderbareEinzelParameter = new LinkedHashMap<>();

	/**
	 * Flag, welches den Status des Viewmenüs 'Betroffene Objekte anzeigen' hält.
	 * Das Filtern passiert also hier im COntentprovider!
	 */
	private boolean betroffeneObjekteZeigen = true;

	/**
	 * Flag, welches den Status des Viewmenüs 'Nicht änderbare Objekte anzeigen'
	 * hält. Das Filtern passiert also hier im COntentprovider!
	 */
	private boolean nichtAenderbareObjekteZeigen = true;

	/**
	 * Debug-Logger zu Testzwecken.
	 */
	private final Debug debug = Debug.getLogger();

	/**
	 * Konstruktor reicht durch und setzt eine Default-Kurzinfo für die Ansicht.
	 *
	 * @param paramPluginView
	 *            der View, in dem wir uns befinden.
	 */
	public BetroffeneObjekteContentProvider(final AbstractParamPluginView paramPluginView) {
		super(paramPluginView);
		setKurzinfo(BetroffeneObjekteContentProvider.DEFAULT_KURZINFO);
		RahmenwerkService.getService().getArtDesParametersatzesProvider().addPropertyChangeListener(this);
		debug.config("Contentprovider initialisiert");
	}

	@Override
	public void inputChanged(final Viewer viewer, final Object oldInput, final Object newInput) {
		super.inputChanged(viewer, oldInput, newInput);
		if ((newInput instanceof ModifiableParameterInfo[]) && (newInput != oldInput)) {
			currentInput = (ModifiableParameterInfo[]) newInput;
			if (ParamPlugin.getDefault().isUsingReloadJobs()) {
				final ReloadCurrentInputJob job = new ReloadCurrentInputJob(getParamPluginView(), true);
				job.addJobChangeListener(new JobChangeAdapter() {

					@Override
					public void done(final IJobChangeEvent event) {
						if (!event.getResult().isOK()) {
							currentInput = null;
							clearDerivedContainers();
						}
					}
				});
				job.schedule();
			} else {
				final Display display = PlatformUI.getWorkbench().getDisplay();
				final Shell shell = display.getActiveShell();
				Cursor waitCursor = null;
				try {
					if (null != shell) {
						waitCursor = new Cursor(Display.getDefault(), SWT.CURSOR_WAIT);
						shell.setCursor(waitCursor);
					}
					reloadCurrentInput(true);
				} catch (final ParameterClientException e) {
					ParamPlugin.getDefault().getLogger().error("Laden des neuen Datensatzes fehlgeschlagen", e);
					currentInput = null;
					clearDerivedContainers();
					throw new IllegalStateException(e);
				} finally {
					if (null != shell) {
						shell.setCursor(null);
					}
					if (null != waitCursor) {
						waitCursor.dispose();
						waitCursor = null;
					}
				}
			}
		} else {
			currentInput = null;
			clearDerivedContainers();
		}
	}

	@Override
	public void dispose() {
		RahmenwerkService.getService().getArtDesParametersatzesProvider().removePropertyChangeListener(this);
		super.dispose();
	}

	/**
	 * Das bestehende Input-Objekt neu laden.
	 *
	 * @param doRefreshViewer
	 *            true Nach dem Neuladen den Viewer aktualisieren
	 * @throws ParameterClientException
	 *             bei Fehlern in der Methodenbibliothek
	 */
	@Override
	public void reloadCurrentInput(final boolean doRefreshViewer) throws ParameterClientException {
		if (currentInput == null) {
			if (doRefreshViewer) {
				setKurzinfo(BetroffeneObjekteContentProvider.DEFAULT_KURZINFO);
			}
			return;
		}
		try {
			clearDerivedContainers();
			updateDerivedContainers();
			updateKurzinfoFromContainers();
			if (doRefreshViewer) {
				refreshViewer();
			}
		} catch (final IllegalStateException e) {
			final Display display = PlatformUI.getWorkbench().getDisplay();
			if ((null != display) && !display.isDisposed()) {
				display.asyncExec(new Runnable() {

					@Override
					public void run() {
						final MessageDialog md = new MessageDialog(display.getActiveShell(),
								"Fehler bei Aktualisierung", null,
								"Ansicht '" + getParamPluginView().getPartName()
										+ "' konnte nicht aktualisiert werden: " + e.getLocalizedMessage(),
								MessageDialog.ERROR, new String[] { "OK" }, 0);
						md.open();
					}
				});
			}
		}
	}

	/**
	 * Aktualisiert die beiden Container mit den betroffenen und den nicht
	 * änderbaren Objetken durch eine neue Abfrage beim DaV (Parametrierung oder
	 * Archiv).
	 *
	 * @throws ParameterClientException
	 *             Fehler bei Kommunikation mit Parametrierung.
	 */
	private void updateDerivedContainers() throws ParameterClientException {
		for (final ModifiableParameterInfo mpi : currentInput) {
			final ObjektAuswahlEnum objektAuswahlArt = mpi.autoFillTypes();
			final ModifiableParameterInfo[] resolvedMpis = mpi.resolve();
			for (final ModifiableParameterInfo resolvedMpi : resolvedMpis) {
				final ParameterInfo info = resolvedMpi.toParameterInfos()[0];
				final Collection<SystemObject> aenderbareObjekte = ParameterManagerAllgemein.getInstanz()
						.getAenderbareObjekteAllgemein(resolvedMpi);
				if (aenderbareObjekte.isEmpty()) {
					continue;
				}
				final Parameter[] einzelParameterIst = ParameterManagerAllgemein.getInstanz()
						.getParameterAllgemein(resolvedMpi, aenderbareObjekte, false);
				for (final Parameter p : einzelParameterIst) {
					boolean aenderbar = true;
					if (!ObjektAuswahlEnum.INSTANZ_ECHT.equals(objektAuswahlArt)) {
						switch (p.getQuelle().getTyp()) {
						case LOKAL_OBJEKT:
							/* HINWEIS: Siehe SwEnt SWE 8.2, S. 20 */
							/* case DEFAULT_OBJEKT: */
							aenderbar = false;
							break;
						default:
							break;
						}
					}
					if (aenderbar) {
						synchronized (betroffeneEinzelParameter) {
							betroffeneEinzelParameter.put(p, info);
						}
					} else {
						synchronized (nichtAenderbareEinzelParameter) {
							nichtAenderbareEinzelParameter.put(p, info);
						}
					}
				}
			}
		}
	}

	/**
	 * Bestimmt einen neuen Kurzinfo-Text aus dem Inhalt der beiden Container mit
	 * den betroffenen bzw. nicht änderbaren Objekten.
	 */
	private void updateKurzinfoFromContainers() {
		if (currentInput == null) {
			setKurzinfo(BetroffeneObjekteContentProvider.DEFAULT_KURZINFO);
			return;
		}
		final StringBuilder b = new StringBuilder(Zeichenketten.PLUGIN_PARAM_BEZEICHNER_BETROFFENE_OBJEKTE);
		b.append(": ");
		int sum = 0;
		synchronized (betroffeneEinzelParameter) {
			sum += betroffeneEinzelParameter.size();
		}
		b.append(sum);
		b.append(" von ");
		synchronized (nichtAenderbareEinzelParameter) {
			sum += nichtAenderbareEinzelParameter.size();
		}
		b.append(sum);
		setKurzinfo(b.toString());
	}

	// public void reloadIndividualParameters(final Parameter... parameters) {
	/*
	 * HINWEIS Methode reloadIndividualParameters(Parameter parameters...)
	 * implementieren, damit nicht jedesmal die ganze Liste geladen wird, wenn ein
	 * Parametersatz gelöscht wurde
	 */
	// }

	/**
	 * Die Listen mit den ermittelten Parametern leeren.
	 */
	private void clearDerivedContainers() {
		synchronized (betroffeneEinzelParameter) {
			betroffeneEinzelParameter.clear();
		}
		synchronized (nichtAenderbareEinzelParameter) {
			nichtAenderbareEinzelParameter.clear();
		}
	}

	@Override
	public Object[] getElements(final Object inputElement) {
		final Set<Parameter> elements = new HashSet<>();
		if (isBetroffeneObjekteZeigen()) {
			synchronized (betroffeneEinzelParameter) {
				elements.addAll(betroffeneEinzelParameter.keySet());
			}
		}
		if (isNichtAenderbareObjekteZeigen()) {
			synchronized (nichtAenderbareEinzelParameter) {
				elements.addAll(nichtAenderbareEinzelParameter.keySet());
			}
		}
		return elements.toArray();
	}

	@Override
	public void selectionChanged(final SelectionChangedEvent event) {
		// leer, für Änderungen unserer eigenen Selektion interessieren wir uns
		// nicht, weil wir nichts irgendwohin zurückschreiben und niemanden
		// notifizieren müssen. Wir sind das ende der Kette aus vier Ansichten
		// Abb 5-41
	}

	/**
	 * Liefert das Flag, welches den Status des Viewmenüs 'Betroffene Objekte
	 * anzeigen' hält. Das Filtern passiert also hier im COntentprovider!
	 *
	 * @return betroffene Objekte zeigen?
	 */
	public boolean isBetroffeneObjekteZeigen() {
		return betroffeneObjekteZeigen;
	}

	/**
	 * Setzt das Flag, welches den Status des Viewmenüs 'Betroffene Objekte
	 * anzeigen' hält. Das Filtern passiert also hier im COntentprovider!
	 *
	 * @param betroffeneObjekteZeigen
	 *            true - betroffene Objekte zeigen
	 */
	public void setBetroffeneObjekteZeigen(final boolean betroffeneObjekteZeigen) {
		this.betroffeneObjekteZeigen = betroffeneObjekteZeigen;
		refreshViewer();
	}

	/**
	 * Liefert das Flag, welches den Status des Viewmenüs 'Nicht betroffene Objekte
	 * anzeigen' hält. Das Filtern passiert also hier im COntentprovider!
	 *
	 * @return nicht änderbare Objekte zeigen?
	 */
	public boolean isNichtAenderbareObjekteZeigen() {
		return nichtAenderbareObjekteZeigen;
	}

	/**
	 * Setzt das Flag, welches den Status des Viewmenüs 'Nicht betroffene Objekte
	 * anzeigen' hält. Das Filtern passiert also hier im COntentprovider!
	 *
	 * @param nichtAenderbareObjekteZeigen
	 *            true - nicht änderbare Objekte zeigen
	 */
	public void setNichtAenderbareObjekteZeigen(final boolean nichtAenderbareObjekteZeigen) {
		this.nichtAenderbareObjekteZeigen = nichtAenderbareObjekteZeigen;
		refreshViewer();
	}

	@Override
	public void propertyChange(final PropertyChangeEvent evt) {
		if (currentInput == null) {
			return;
		}
		if (evt.getOldValue().equals(evt.getNewValue())) {
			return;
		}
		boolean mustReload = true;
		final IArtDesParametersatzesProvider artProvider = RahmenwerkService.getService()
				.getArtDesParametersatzesProvider();
		if (IArtDesParametersatzesProvider.PROP_ART_DES_PARAMETERSATZES.equals(evt.getPropertyName())) {
			if (IArtDesParametersatzesProvider.ART_ORIGINAL.equals(evt.getNewValue())) {
				for (final ModifiableParameterInfo mpi : currentInput) {
					mpi.setSim(DataDescription.NO_SIMULATION_VARIANT_SET);
					mpi.setHistorisch(false);
				}
				if (DataDescription.NO_SIMULATION_VARIANT_SET != artProvider.getSelectedSimulationsVariante()) {
					// IArtDesParametersatzesProvider.PROP_SELECTEDSIMULATIONSVARIANTE
					// will follow
					mustReload = false;
				}
			} else if (IArtDesParametersatzesProvider.ART_HISTORISCH.equals(evt.getNewValue())) {
				for (final ModifiableParameterInfo mpi : currentInput) {
					mpi.setHistorisch(true);
				}
				mustReload = false;
				// IArtDesParametersatzesProvider.PROP_ZEITPUNKT will follow
			} else if (IArtDesParametersatzesProvider.ART_SIMULATION.equals(evt.getNewValue())) {
				for (final ModifiableParameterInfo mpi : currentInput) {
					mpi.setHistorisch(false);
				}
				mustReload = false;
				// IArtDesParametersatzesProvider.PROP_SELECTEDSIMULATIONSVARIANTE
				// will follow
			}
		} else if (IArtDesParametersatzesProvider.PROP_SELECTEDSIMULATIONSVARIANTE.equals(evt.getPropertyName())) {
			for (final ModifiableParameterInfo mpi : currentInput) {
				mpi.setSim(RahmenwerkService.getService().getArtDesParametersatzesProvider()
						.getSelectedSimulationsVariante());
			}
		} else if (IArtDesParametersatzesProvider.PROP_ZEITPUNKT.equals(evt.getPropertyName())) {
			for (final ModifiableParameterInfo mpi : currentInput) {
				mpi.setZeitpunkt(RahmenwerkService.getService().getArtDesParametersatzesProvider().getZeitpunkt());
			}
		}
		debug.config("PropertyChange", new Object[] { getClass().getSimpleName(), evt.getPropertyName(),
				evt.getOldValue(), evt.getNewValue(), mustReload });
		if (mustReload) {
			debug.config(RahmenwerkService.getService().getArtDesParametersatzesProvider().toString());
			if (ParamPlugin.getDefault().isUsingReloadJobs()) {
				final ReloadCurrentInputJob job = new ReloadCurrentInputJob(getParamPluginView(), true);
				job.setRule(new AnsichtAktualisierenViaReloadCurrentInputJobRule());
				job.schedule();
			} else {
				final ContentViewer viewer = (ContentViewer) getParamPluginView().getSite().getSelectionProvider();
				final AbstractParamPluginContentProvider contentProvider = (AbstractParamPluginContentProvider) viewer
						.getContentProvider();
				final Display display = PlatformUI.getWorkbench().getDisplay();
				final Shell shell = display.getActiveShell();
				Cursor waitCursor = null;
				try {
					if (null != shell) {
						waitCursor = new Cursor(Display.getDefault(), SWT.CURSOR_WAIT);
						shell.setCursor(waitCursor);
					}
					contentProvider.reloadCurrentInput(true);
				} catch (final ParameterClientException e) {
					ParamPlugin.getDefault().getLogger().error("Laden des neuen Datensatzes fehlgeschlagen", e);
					throw new IllegalStateException(e);
				} finally {
					if (null != shell) {
						shell.setCursor(null);
					}
					if (null != waitCursor) {
						waitCursor.dispose();
						waitCursor = null;
					}
				}
			}
		}
	}
}
