/*
 * Rahmenwerk-Plug-in "Parametrierung"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.param.provider;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;

import org.eclipse.jface.viewers.ContentViewer;
import org.eclipse.jface.viewers.ISelectionProvider;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.ITreeContentProvider;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.window.Window;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Cursor;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IViewPart;
import org.eclipse.ui.PlatformUI;

import com.bitctrl.lib.eclipse.wizards.ComposedWizardDialog;

import de.bsvrz.buv.plugin.param.IArtDesParametersatzesProvider;
import de.bsvrz.buv.plugin.param.ModifiableParameterInfo;
import de.bsvrz.buv.plugin.param.ParametertypenCache;
import de.bsvrz.buv.plugin.param.Zeichenketten;
import de.bsvrz.buv.plugin.param.internal.ParamPlugin;
import de.bsvrz.buv.plugin.param.internal.RahmenwerkService;
import de.bsvrz.buv.plugin.param.lib.ParameterManagerAllgemein;
import de.bsvrz.buv.plugin.param.provider.helper.MehrfachauswahlDialog;
import de.bsvrz.buv.plugin.param.views.AbstractParamPluginView;
import de.bsvrz.buv.plugin.param.views.BetroffeneObjekteView;
import de.bsvrz.buv.plugin.param.views.ObjektauswahlView;
import de.bsvrz.buv.rw.bitctrl.eclipse.MultipleSelection;
import de.bsvrz.buv.rw.bitctrl.eclipse.SystemObjectAuswahlType;
import de.bsvrz.buv.rw.bitctrl.eclipse.wizards.SystemObjectAuswahlWizardPage;
import de.bsvrz.dav.daf.main.config.AttributeGroup;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.dav.daf.main.config.SystemObjectType;
import de.bsvrz.puk.param.lib.MethodenBibliothek;
import de.bsvrz.puk.param.lib.Parameter;
import de.bsvrz.puk.param.lib.ParameterClientException;
import de.bsvrz.puk.param.lib.ParameterInfo;
import de.bsvrz.puk.param.lib.ParameterManager;

/**
 * Contentprovider für einen Treeviewer, der die Objekte für die Abb 5-41
 * links-unten darstellt. Der Contentprovider ist Selectionlistener am
 * Treeviewer in {@link de.bsvrz.buv.plugin.param.views.ParametertypenView}
 * sowie am {@link IArtDesParametersatzesProvider} aus den Plug-In.
 *
 * @author BitCtrl Systems GmbH, Albrecht Uhlmann
 */
public class ObjektauswahlContentProvider extends AbstractParamPluginContentProvider implements ITreeContentProvider {

	/**
	 * Default-Kurzinfo, falls keine Objekte geladen sind.
	 */
	private static final String DEFAULT_KURZINFO = Zeichenketten.PLUGIN_PARAM_BEZEICHNER_OBJEKTAUSWAHL
			+ ": Bitte einen Parametertypen (=Attributgruppe)\n" + "in der Ansicht 'Parameter Allgemein' auswählen";

	/**
	 * Aktuelles Input-Objekt, eine Attributgruppe.
	 */
	private AttributeGroup currentInput;

	/**
	 * Die Menge der Parameter an realen Typen.
	 */
	private final Map<SystemObjectType, Parameter> parametersOnTypes = new LinkedHashMap<>();

	/**
	 * Die Menge der Parameter an virtuellen Typen, nämlich Hierarchieobjekten.
	 */
	private final Map<SystemObject, Parameter> parametersOnVirtualInstances = new LinkedHashMap<>();

	/**
	 * Konstruktor reicht durch und setzt eine Default-Kurzinfo für die Ansicht.
	 *
	 * @param paramPluginView
	 *            der View, in dem wir uns befinden.
	 */
	public ObjektauswahlContentProvider(final AbstractParamPluginView paramPluginView) {
		super(paramPluginView);
		setKurzinfo(ObjektauswahlContentProvider.DEFAULT_KURZINFO);
	}

	@Override
	public Object[] getElements(final Object inputElement) {
		Object[] result = null;
		if (inputElement instanceof AttributeGroup) {
			result = parametersOnTypes.values().toArray();
		} else {
			result = new Object[0];
		}

		return result;
	}

	/*
	 * (non-Javadoc)
	 *
	 * @see de.bsvrz.buv.plugin.param.provider.AbstractParamPluginContentProvider
	 * #inputChanged(org.eclipse.jface.viewers.Viewer, java.lang.Object,
	 * java.lang.Object)
	 */
	@Override
	public void inputChanged(final Viewer viewer, final Object oldInput, final Object newInput) {
		super.inputChanged(viewer, oldInput, newInput);
		if (newInput instanceof AttributeGroup) {
			currentInput = (AttributeGroup) newInput;
			final Display display = PlatformUI.getWorkbench().getDisplay();
			final Shell shell = display.getActiveShell();
			Cursor waitCursor = null;
			try {
				if (null != shell) {
					waitCursor = new Cursor(Display.getDefault(), SWT.CURSOR_WAIT);
					shell.setCursor(waitCursor);
				}
				reloadCurrentInput(false);
			} catch (final ParameterClientException e) {
				ParamPlugin.getDefault().getLogger().warning("Fehler beim Lesen der Parameter", e);
			} finally {
				if (null != shell) {
					shell.setCursor(null);
				}
				if (null != waitCursor) {
					waitCursor.dispose();
					waitCursor = null;
				}
			}
		} else {
			currentInput = null;
			clearDerivedContainers();
		}
	}

	@Override
	public void reloadCurrentInput(final boolean doRefreshViewer) throws ParameterClientException {
		if (currentInput == null) {
			setKurzinfo(ObjektauswahlContentProvider.DEFAULT_KURZINFO);
			return;
		}
		final ParameterManager hierarchie = MethodenBibliothek
				.getParameterManager(RahmenwerkService.getService().getObjektFactory().getDav());
		clearDerivedContainers();
		final Set<SystemObjectType> parents = ParametertypenCache.getInstanz().getParents(currentInput);
		final ParameterInfo[] infos = new ParameterInfo[parents.size()];
		int loop = 0;
		for (final SystemObjectType t : parents) {
			infos[loop] = new ParameterInfo(t, t, currentInput,
					RahmenwerkService.getService().getArtDesParametersatzesProvider().getSelectedSimulationsVariante());
			++loop;
		}
		final Parameter[] parameters = hierarchie.getParameter(infos);
		for (final Parameter p : parameters) {
			if (p.getObjekt() instanceof SystemObjectType) {
				parametersOnTypes.put((SystemObjectType) p.getObjekt(), p);
			}
		}
		updateKurzinfoFromContainers();
		if (doRefreshViewer) {
			if ((getContentViewer() != null) && !getContentViewer().getControl().isDisposed()
					&& !getContentViewer().getControl().getDisplay().isDisposed()) {
				getContentViewer().getControl().getDisplay().asyncExec(new Runnable() {

					@Override
					public void run() {
						if (!getContentViewer().getControl().isDisposed()
								&& !getContentViewer().getControl().getDisplay().isDisposed()
								&& (getContentViewer().getInput() != null)) {
							final SelectionChangedEvent event = new SelectionChangedEvent(getContentViewer(),
									getContentViewer().getSelection());
							selectionChanged(event);
						}
					}
				});
			}
			refreshViewer();
		}
	}

	/**
	 * Bestimmt einen neuen Kurzinfo-Text aus dem Inhalt der beiden Container mit
	 * den betroffenen bzw. nicht änderbaren Objekten.
	 */
	private void updateKurzinfoFromContainers() {
		if (currentInput == null) {
			setKurzinfo(ObjektauswahlContentProvider.DEFAULT_KURZINFO);
			return;
		}
		final StringBuilder b = new StringBuilder(currentInput.getNameOrPidOrId());
		b.append(":\n");
		int numRealTypes = 0, numVirtualTypes = 0;
		final Set<SystemObjectType> directParents = ParametertypenCache.getInstanz().getRealParents(currentInput);
		for (final Entry<SystemObjectType, Parameter> entry : parametersOnTypes.entrySet()) {
			final Parameter p = entry.getValue();
			if (p.isDataAvailable()) {
				if (directParents.contains(entry.getKey())) {
					++numRealTypes;
				} else {
					++numVirtualTypes;
				}
			}
		}
		b.append(numRealTypes);
		if (numRealTypes == 1) {
			b.append(" parametrierter echter Typ, ");
		} else {
			b.append(" parametrierte echte Typen, ");
		}
		b.append(numVirtualTypes);
		if (numVirtualTypes == 1) {
			b.append(" parametrierter virtueller Typ\n");
		} else {
			b.append(" parametrierte virtuelle Typen\n");
		}
		setKurzinfo(b.toString());
	}

	/**
	 * Die Listen mit den ermittelten Parametern leeren.
	 */
	private void clearDerivedContainers() {
		parametersOnTypes.clear();
		parametersOnVirtualInstances.clear();
	}

	@Override
	public Object[] getChildren(final Object parentElement) {
		Object[] result = null;
		SystemObject instanz = null;
		if (parentElement instanceof SystemObjectType) {
			final SystemObjectType type = (SystemObjectType) parentElement;
			result = type.getElements().toArray();
		} else if (parentElement instanceof Parameter) {
			final Parameter p = (Parameter) parentElement;
			if (p.getObjekt() instanceof SystemObjectType) {
				final SystemObjectType type = (SystemObjectType) p.getObjekt();
				final List<SystemObject> vis = type.getElements();
				if (type.getAttributeGroups().contains(currentInput)) {
					final ModifiableParameterInfo mpi = new ModifiableParameterInfo();
					mpi.setObjekt(p.getObjekt());
					mpi.addTyp(p.getTyp());
					mpi.setAtg(p.getAtg());
					mpi.setSim(p.getSim());
					try {
						final Parameter[] einzelParameterIst = ParameterManagerAllgemein.getInstanz()
								.getParameterAllgemein(mpi, vis, false);
						result = einzelParameterIst;
					} catch (final ParameterClientException e) {
						ParamPlugin.getDefault().getLogger()
								.error("Fehler bei Bereitstellung der Kinder zu " + type.toString(), e);
						result = new Object[0];
					}
				} else {
					// TYP_VIRTUELL, children sind INSTANZ_VIRTUELL
					final List<Parameter> parameterListe = new ArrayList<>(vis.size());
					final List<SystemObject> missingParameters = new ArrayList<>();
					final ParameterManager hierarchie = MethodenBibliothek
							.getParameterManager(RahmenwerkService.getService().getObjektFactory().getDav());
					// Bereits vorhandene Parameter in eine Zwischen-Liste
					// einfügen,
					// die Systemobjekte fehlender Parameter merken
					for (final SystemObject o : vis) {
						final Parameter currentParameter = parametersOnVirtualInstances.get(o);
						if (currentParameter == null) {
							missingParameters.add(o);
						} else {
							parameterListe.add(currentParameter);
						}
					}
					// Fehlende Parameter abholen
					if (!missingParameters.isEmpty()) {
						try {
							int loop = 0;
							final ParameterInfo[] infos = new ParameterInfo[missingParameters.size()];
							final Set<SystemObjectType> directParents = ParametertypenCache.getInstanz()
									.getDirectParents(currentInput);
							if (directParents.isEmpty()) {
								throw new IllegalStateException(
										"Es konnte kein Typ ermittelt werden, an dem die Attributgruppe "
												+ currentInput.toString() + " definiert ist.");
							}
							SystemObjectType directParent = directParents.iterator().next();
							if (directParents.size() > 1) {
								final String[] rootTypes = new String[directParents.size()];
								int tLoop = 0;
								for (final SystemObjectType t : directParents) {
									rootTypes[tLoop] = t.getPid();
									++tLoop;
								}
								final SystemObjectAuswahlWizardPage auswahlPage = new SystemObjectAuswahlWizardPage(
										"Mehrfachvererbung!", MultipleSelection.Single,
										SystemObjectAuswahlType.OnlyTypes, new ArrayList<SystemObject>(), rootTypes);
								final String desc = "Die Attributgruppe " + currentInput.toString()
										+ " hat mehr als ein direktes Elternobjekt";
								ParamPlugin.getDefault().getLogger().warning(desc);
								auswahlPage.setDescription(desc);
								final ComposedWizardDialog wizardDialog = new ComposedWizardDialog("Mehrfachvererbung!",
										auswahlPage);
								if (Window.OK == wizardDialog.open()) {
									directParent = (SystemObjectType) auswahlPage.getAuswahl().get(0);
								} else {
									throw new IllegalStateException(
											"Mehrdeutige Auswahl: " + desc + "\nKeine Entscheidung durch den Nutzer!");
								}
							}
							for (final SystemObject o : missingParameters) {
								infos[loop] = new ParameterInfo(o, directParent, currentInput,
										RahmenwerkService.getService().getArtDesParametersatzesProvider()
												.getSelectedSimulationsVariante());
								++loop;
							}

							final Parameter[] parameters = hierarchie.getParameter(infos);
							if (parameters != null) {
								for (final Parameter parameter : parameters) {
									parametersOnVirtualInstances.put(parameter.getObjekt(), parameter);
									parameterListe.add(parameter);
								}
							} else {
								ParamPlugin.getDefault().getLogger()
										.warning("Fehler beim Auslesen der Parameter für Kindobjekte zu "
												+ type.toString() + ": Funktion liefert null");
							}
						} catch (final ParameterClientException e) {
							ParamPlugin.getDefault().getLogger().warning(0,
									"Fehler beim Auslesen der Parameter für Kindobjekte zu " + type.toString(), e);
						} catch (final IllegalStateException e) {
							ParamPlugin.getDefault().getLogger().warning(0, e.getLocalizedMessage(), e);
						}
					}
					result = parameterListe.toArray();
				}
			} else {
				instanz = p.getObjekt();
			}
		} else if (parentElement instanceof SystemObject) {
			instanz = (SystemObject) parentElement;
		}
		if (instanz != null) {
			final Set<SystemObjectType> mappedTypes = ParametertypenCache.getInstanz()
					.getMappedTypes(instanz.getType());
			mappedTypes.removeAll(ParametertypenCache.getInstanz().getDirectParents(currentInput));
			final List<Parameter> parameterListe = new ArrayList<>(mappedTypes.size());
			final ParameterManager hierarchie = MethodenBibliothek
					.getParameterManager(RahmenwerkService.getService().getObjektFactory().getDav());
			try {
				int loop = 0;
				final ParameterInfo[] infos = new ParameterInfo[mappedTypes.size()];
				final Set<SystemObjectType> directParents = ParametertypenCache.getInstanz()
						.getDirectParents(currentInput);
				if (directParents.isEmpty()) {
					throw new IllegalStateException("Es konnte kein Typ ermittelt werden, an dem die Attributgruppe "
							+ currentInput.toString() + " definiert ist.");
				}
				if (directParents.size() > 1) {
					ParamPlugin.getDefault().getLogger().warning("Die Attributgruppe " + currentInput.toString()
							+ " hat mehr als ein direktes Elternobjekt");
				}
				final SystemObjectType directParent = directParents.iterator().next();
				for (final SystemObject o : mappedTypes) {
					infos[loop] = new ParameterInfo(o, directParent, currentInput, RahmenwerkService.getService()
							.getArtDesParametersatzesProvider().getSelectedSimulationsVariante());
					++loop;
				}

				final Parameter[] parameters = hierarchie.getParameter(infos);
				if (parameters != null) {
					for (final Parameter parameter : parameters) {
						parametersOnVirtualInstances.put(parameter.getObjekt(), parameter);
						parameterListe.add(parameter);
					}
				} else {
					ParamPlugin.getDefault().getLogger()
							.warning("Fehler beim Auslesen der Parameter für Kindobjekte zu " + instanz.toString()
									+ ": Funktion liefert null");
				}
			} catch (final ParameterClientException e) {
				ParamPlugin.getDefault().getLogger().warning(0,
						"Fehler beim Auslesen der Parameter für Kindobjekte zu " + instanz.toString(), e);
			} catch (final IllegalStateException e) {
				ParamPlugin.getDefault().getLogger().warning(0, e.getLocalizedMessage(), e);
			}
			result = parameterListe.toArray();
		}
		return result;
	}

	@Override
	public Object getParent(final Object element) {
		Object parent = null;
		if (element instanceof SystemObjectType) {
			final SystemObjectType type = (SystemObjectType) element;
			parent = type.getConfigurationArea();
		} else if (element instanceof SystemObject) {
			final SystemObject object = (SystemObject) element;
			parent = object.getType();
		}
		return parent;
	}

	@Override
	public boolean hasChildren(final Object element) {
		boolean result = false;
		SystemObject o = null;
		if (element instanceof SystemObjectType) {
			final SystemObjectType type = (SystemObjectType) element;
			result = !type.getElements().isEmpty();
		} else if (element instanceof Parameter) {
			final Parameter p = (Parameter) element;
			if (p.getObjekt() instanceof SystemObjectType) {
				final SystemObjectType type = (SystemObjectType) p.getObjekt();
				result = !type.getElements().isEmpty();
			} else {
				o = p.getObjekt();
			}
		} else if (element instanceof SystemObject) {
			o = (SystemObject) element;
		}
		if (o != null) {
			final Set<SystemObjectType> mappedTypes = ParametertypenCache.getInstanz().getMappedTypes(o.getType());
			result = !mappedTypes.isEmpty();
		}
		return result;
	}

	@Override
	public void selectionChanged(final SelectionChangedEvent event) {
		if (!event.getSelection().isEmpty() && (event.getSelection() instanceof IStructuredSelection)) {
			final IStructuredSelection s = (IStructuredSelection) event.getSelection();
			if ((s.size() > 1) && MehrfachauswahlDialog.isEnabled()) {
				final MehrfachauswahlDialog mad = new MehrfachauswahlDialog(null, ObjektauswahlView.HILFE_ID);
				if (Window.OK != mad.open()) {
					return;
				}
			}
			if ((s.getFirstElement() instanceof SystemObject) || (s.getFirstElement() instanceof Parameter)) {
				final IViewPart betroffeneObjekteView = PlatformUI.getWorkbench().getActiveWorkbenchWindow()
						.getActivePage().findView(BetroffeneObjekteView.ID);
				if (betroffeneObjekteView != null) {
					final List<ModifiableParameterInfo> modifiableInfos = new ArrayList<>(2);
					final Iterator<?> selectionIterator = s.iterator();
					final IArtDesParametersatzesProvider artProvider = RahmenwerkService.getService()
							.getArtDesParametersatzesProvider();
					while (selectionIterator.hasNext()) {
						final Object next = selectionIterator.next();
						final ModifiableParameterInfo mpi = new ModifiableParameterInfo();
						mpi.setAtg(currentInput);
						if (next instanceof SystemObject) {
							final SystemObject element = (SystemObject) next;
							mpi.setObjekt(element);
						} else if (next instanceof Parameter) {
							final Parameter p = (Parameter) next;
							mpi.setObjekt(p.getObjekt());
						}
						mpi.setHistorisch(IArtDesParametersatzesProvider.ART_HISTORISCH
								.equals(artProvider.getArtDesParametersatzes()));
						mpi.setSim(artProvider.getSelectedSimulationsVariante());
						mpi.setZeitpunkt(artProvider.getZeitpunkt());
						modifiableInfos.add(mpi);
					}
					final ISelectionProvider p = betroffeneObjekteView.getSite().getSelectionProvider();
					if (p instanceof ContentViewer) {
						final ContentViewer viewer = (ContentViewer) p;
						viewer.setInput(modifiableInfos.toArray(new ModifiableParameterInfo[modifiableInfos.size()]));
					}
				}
			}
		}
	}
}
