/*
 * Rahmenwerk-Plug-in "Parametrierung"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.param.testprogramme;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.logging.Logger;

import de.bsvrz.buv.plugin.param.ParametertypenCache;
import de.bsvrz.buv.rw.bitctrl.eclipse.SystemObjectComparatorAlphabetical;
import de.bsvrz.dav.daf.main.ClientDavInterface;
import de.bsvrz.dav.daf.main.config.AttributeGroup;
import de.bsvrz.dav.daf.main.config.ConfigurationArea;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.dav.daf.main.config.SystemObjectType;
import de.bsvrz.puk.param.lib.Hierarchie;
import de.bsvrz.puk.param.lib.MethodenBibliothek;
import de.bsvrz.puk.param.lib.ParameterInfo;
import de.bsvrz.sys.funclib.application.StandardApplication;
import de.bsvrz.sys.funclib.application.StandardApplicationRunner;
import de.bsvrz.sys.funclib.bitctrl.modell.DefaultObjektFactory;
import de.bsvrz.sys.funclib.bitctrl.modell.ObjektFactory;
import de.bsvrz.sys.funclib.commandLineArgs.ArgumentList;

/**
 * Kleines Hilfsprogramm zum Testen der Klasse {@link ParametertypenCache}.
 *
 * Die Ausgaben dienen auch als Inspiration für den zu erstellenden
 * Labelprovider.
 *
 * @author BitCtrl Systems GmbH, Albrecht Uhlmann
 *
 */
public class ParametertypenFinder implements StandardApplication {

	/**
	 * Pid der ausgewählten Atributgruppe.
	 */
	private String selectedAtgPid;

	/**
	 * Pid des ausgewählten Typen.
	 */
	private String selectedTypePid;

	@Override
	public void initialize(final ClientDavInterface connection) throws Exception {
		final ObjektFactory factory = DefaultObjektFactory.getInstanz();
		factory.setDav(connection);
		ParametertypenCache.erzeugeInstanz(factory);
		SystemObjectType selectedType = null;
		AttributeGroup selectedAttributeGroup = null;
		SystemObject selectedObj = null;
		final ParametertypenCache ptypcache = ParametertypenCache.getInstanz();
		int kbLoop = 1;
		for (final ConfigurationArea kb : ptypcache.getConfigurationAreas()) {
			System.out.println(kbLoop + ": Konfigurationsbereich " + kb.getName());
			int typeLoop = 1;
			for (final SystemObjectType type : ptypcache.getSystemObjects(kb)) {
				System.out.println("   " + kbLoop + '.' + typeLoop + ": Typ " + type.getName());
				int atgLoop = 1;
				for (final AttributeGroup atg : ptypcache.getAttributeGroups(type)) {
					System.out.println(
							"      " + kbLoop + '.' + typeLoop + '.' + atgLoop + ": Attributgruppe " + atg.toString());
					if (atg.getPid().equals(selectedAtgPid)) {
						selectedAttributeGroup = atg;
					}
					++atgLoop;
				}
				if (type.getPid().equals(selectedTypePid)) {
					selectedType = type;
				}
				++typeLoop;
			}
			++kbLoop;
		}

		if (selectedAttributeGroup == null) {
			Logger.getLogger(getClass().getName())
					.warning("Gewünschte Attributgruppe " + selectedAtgPid + " nicht gefunden");
			System.exit(1);
		}
		if (selectedType == null) {
			Logger.getLogger(getClass().getName())
					.warning("Gewünschter Typ " + selectedTypePid + " nicht gefunden, suche Objekte...");
			selectedObj = connection.getDataModel().getObject(selectedTypePid);
			if (selectedObj == null) {
				Logger.getLogger(getClass().getName())
						.warning("Gewünschtes Objekt " + selectedTypePid + " nicht gefunden");
				System.exit(1);
			}
		}
		ParameterInfo info;
		if (selectedType == null) {
			info = new ParameterInfo(selectedObj, selectedAttributeGroup);
		} else {
			info = new ParameterInfo(selectedType, selectedAttributeGroup);
		}
		final Hierarchie hierarchie = MethodenBibliothek.getHierarchie(connection);
		Thread.sleep(2000L);
		final SystemObject wurzel = hierarchie.getParameterWurzel(info);
		if (wurzel == null) {
			Logger.getLogger(getClass().getName())
					.warning("Für ParameterInfo:\n" + info.toString() + "\nkonnte kein Wurzelelement ermittelt werden");
		} else {
			Logger.getLogger(getClass().getName()).info("Für ParameterInfo:\n" + info.toString()
					+ "\nwurde als Wurzelelement ermittelt:\n" + wurzel.toString());
		}
		final Collection<SystemObject> aenderbareObjekte = hierarchie.getAenderbareObjekte(info);
		if (aenderbareObjekte == null) {
			Logger.getLogger(getClass().getName()).warning("Für ParameterInfo:\n" + info.toString()
					+ "\nkonnten die betroffenen Objekte nicht ermittelt werden");
		} else {
			if (aenderbareObjekte.isEmpty()) {
				Logger.getLogger(getClass().getName())
						.info("Für ParameterInfo:\n" + info.toString() + "\nsind keine Objekte betroffen");
			} else {
				final List<SystemObject> aenderbareObjekteListe = new ArrayList<>(aenderbareObjekte.size());
				aenderbareObjekteListe.addAll(aenderbareObjekte);
				Collections.sort(aenderbareObjekteListe, new SystemObjectComparatorAlphabetical(true));
				final StringBuilder b = new StringBuilder("Für ParameterInfo:\n");
				b.append(info.toString());
				b.append("\nsind folgende ");
				b.append(aenderbareObjekteListe.size());
				b.append(" Objekte potentiell betroffen:\n");
				for (final SystemObject o : aenderbareObjekteListe) {
					b.append(o.toString());
					b.append('\n');
				}
				Logger.getLogger(getClass().getName()).info(b.toString());
				purgeAenderbareMenge(hierarchie, wurzel, info.getAtg(), aenderbareObjekteListe);
				Logger.getLogger(getClass().getName())
						.info("Zusammenfassung:\n  Potentiell betroffen: " + aenderbareObjekte.size()
								+ " Objekte.\n  Wirklich betroffen: " + aenderbareObjekteListe.size() + " Objekte.");
			}
		}
		System.exit(0);
	}

	/**
	 * Die Funktion versucht, ein mit der Methode
	 * {@link Hierarchie#getAenderbareObjekte(ParameterInfo)} ermittelte Objektmenge
	 * so zu reduzieren, dass sie nur noch diejenigen Objekte enthält, die
	 * tatsächlich modifiziert würden. Anders gesagt werden alle Objekte entfernt,
	 * die lokale Parameter haben.
	 *
	 * @param hierarchie
	 *            das zu verwendene Zugriffsobjekt für die Anfragen
	 * @param wurzel
	 *            die ursprünglich ermittelte Parameterwurzel
	 * @param atg
	 *            die betroffene Attributgruppe
	 * @param aenderbareObjekteListe
	 *            die zu reduzierende Liste
	 */
	private void purgeAenderbareMenge(final Hierarchie hierarchie, final SystemObject wurzel, final AttributeGroup atg,
			final List<SystemObject> aenderbareObjekteListe) {
		int loop;
		final ParameterInfo[] info = new ParameterInfo[aenderbareObjekteListe.size()];
		for (loop = 0; loop < aenderbareObjekteListe.size(); ++loop) {
			info[loop] = new ParameterInfo(aenderbareObjekteListe.get(loop), atg);
		}
		final SystemObject[] wurzeln = hierarchie.getParameterWurzel(info);
		if (wurzeln == null) {
			Logger.getLogger(getClass().getName()).warning(
					"Für ParameterInfo:\n" + Arrays.toString(info) + "\nkonnten keine Wurzelelemente ermittelt werden");
			return;
		}
		if (wurzel == null) {
			for (loop = 0; loop < aenderbareObjekteListe.size(); ++loop) {
				if (info[loop].getObjekt().equals(wurzeln[loop])) {
					Logger.getLogger(getClass().getName())
							.info("Objekt '" + info[loop].getObjekt().toString() + "' wird entfernt");
					aenderbareObjekteListe.remove(loop);
				}
			}
		} else {
			for (loop = 0; loop < aenderbareObjekteListe.size(); ++loop) {
				if (!wurzeln[loop].equals(wurzel)) {
					Logger.getLogger(getClass().getName())
							.info("Objekt '" + info[loop].getObjekt().toString() + "' wird entfernt");
					aenderbareObjekteListe.remove(loop);
				}
			}
		}
	}

	@Override
	public void parseArguments(final ArgumentList argumentList) throws Exception {
		selectedAtgPid = argumentList.fetchArgument("-selectedAtgPid").asString();
		selectedTypePid = argumentList.fetchArgument("-selectedTypePid").asString();
	}

	/**
	 * Startpunkt.
	 *
	 * @param args
	 *            die Argumentliste enthält nur die allgemeinen DaV-App-Argumente
	 */
	public static void main(final String[] args) {
		StandardApplicationRunner.run(new ParametertypenFinder(), "typ.applikation", args);
	}

}
