/*
 * Rahmenwerk-Plug-in "Parametrierung"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.param.testprogramme;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Scanner;
import java.util.Set;
import java.util.logging.Logger;

import de.bsvrz.buv.plugin.param.ObjektAuswahlEnum;
import de.bsvrz.buv.plugin.param.ParametertypenCache;
import de.bsvrz.buv.rw.bitctrl.eclipse.SystemObjectComparatorAlphabetical;
import de.bsvrz.dav.daf.main.ClientDavInterface;
import de.bsvrz.dav.daf.main.Data;
import de.bsvrz.dav.daf.main.config.AttributeGroup;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.dav.daf.main.config.SystemObjectType;
import de.bsvrz.puk.param.lib.MethodenBibliothek;
import de.bsvrz.puk.param.lib.Parameter;
import de.bsvrz.puk.param.lib.ParameterInfo;
import de.bsvrz.puk.param.lib.ParameterManager;
import de.bsvrz.puk.param.lib.daten.DataWithTime;
import de.bsvrz.sys.funclib.application.StandardApplication;
import de.bsvrz.sys.funclib.application.StandardApplicationRunner;
import de.bsvrz.sys.funclib.bitctrl.modell.DefaultObjektFactory;
import de.bsvrz.sys.funclib.bitctrl.modell.ObjektFactory;
import de.bsvrz.sys.funclib.commandLineArgs.ArgumentList;

/**
 * Kleines Hilfsprogramm zum Testen der Klasse {@link ParametertypenCache}.
 *
 * Die Ausgaben dienen auch als Inspiration für den zu erstellenden
 * Labelprovider.
 *
 * @author BitCtrl Systems GmbH, Albrecht Uhlmann
 *
 */
public class VerkehrslageVerfahren1Parametrierer implements StandardApplication {

	/**
	 * Pid der ausgewählten Attributgruppe.
	 */
	private final String selectedAtgPid = "atg.verkehrsLageVerfahren1";

	/**
	 * Kann sowohl ein Objekt als auch ein Typ sein.
	 */
	private String selectedObjectPid;

	/**
	 * Attribut v1.
	 */
	private int v1;

	/**
	 * Attribut v2.
	 */
	private int v2;

	/**
	 * Attribut k1.
	 */
	private int k1;

	/**
	 * Attribut k2.
	 */
	private int k2;

	/**
	 * Attribut Ursache aus atl.urlasser.
	 */
	private String ursache;

	/**
	 * true - Parameter löschen.
	 */
	private boolean delete;

	/**
	 * true - nicht änderbare Objekte löschen.
	 */
	private boolean deleteNichtAenderbare;

	/**
	 * true - Parameter schreiben.
	 */
	private boolean setzen;

	@Override
	public void initialize(final ClientDavInterface connection) throws Exception {
		final ObjektFactory factory = DefaultObjektFactory.getInstanz();
		factory.setDav(connection);
		ParametertypenCache.erzeugeInstanz(factory);
		final AttributeGroup atgVerkehrsLageVerfahren1 = connection.getDataModel().getAttributeGroup(selectedAtgPid);
		if (atgVerkehrsLageVerfahren1 == null) {
			final String s = "Gewünschte Attributgruppe " + selectedAtgPid + " nicht gefunden";
			Logger.getLogger(getClass().getName()).warning(s);
			throw new IllegalArgumentException(s);
		}
		final SystemObjectType typMessQuerschnitt = connection.getDataModel().getType("typ.messQuerschnitt");
		assert typMessQuerschnitt != null;
		final SystemObjectType typFahrStreifen = connection.getDataModel().getType("typ.fahrStreifen");
		assert typFahrStreifen != null;
		// selectedObj ist die Auswahl in der Ansicht "Objektauswahl"
		final SystemObject selectedObj = connection.getDataModel().getObject(selectedObjectPid);
		if (selectedObj == null) {
			final String s = "Gewünschtes Objekt " + selectedObjectPid + " nicht gefunden";
			Logger.getLogger(getClass().getName()).warning(s);
			throw new IllegalArgumentException(s);
		}
		final ParametertypenCache ptypcache = ParametertypenCache.getInstanz();
		final Set<SystemObjectType> parents = ptypcache.getParents(atgVerkehrsLageVerfahren1);
		Logger.getLogger(getClass().getName()).info(
				"Für Attributgruppe:\n" + selectedAtgPid + "\nwurden als Eltern ermittelt:\n" + parents.toString());
		SystemObjectType selectedType;
		ObjektAuswahlEnum objektAuswahlArt;
		if (selectedObj instanceof SystemObjectType) {
			selectedType = (SystemObjectType) selectedObj;

			// Die folgende Abfrage ist in der GUI überflüssig
			if (!parents.contains(selectedType)) {
				final String s = "Selektierter Typ '" + selectedType.toString() + "' erreicht die selektierte ATG '"
						+ atgVerkehrsLageVerfahren1 + "' nicht";
				Logger.getLogger(getClass().getName()).warning(s);
				throw new IllegalArgumentException(s);
			}

			if (selectedType.getAttributeGroups().contains(atgVerkehrsLageVerfahren1)) {
				// Fall 1: In der Ansicht "Objektauswahl" wurde ein DaV-Typ
				// selektiert,
				// der die ATG tatsächlich enthält
				objektAuswahlArt = ObjektAuswahlEnum.TYP_ECHT;
			} else {
				// Fall 2: In der Ansicht "Objektauswahl" wurde ein DaV-Typ
				// selektiert,
				// der die ATG nur indirekt über Hierarchien referenziert
				objektAuswahlArt = ObjektAuswahlEnum.TYP_VIRTUELL;
			}
		} else {
			selectedType = selectedObj.getType();
			if (selectedType.getAttributeGroups().contains(atgVerkehrsLageVerfahren1)) {
				// Fall 3: In der Ansicht "Objektauswahl" wurde ein DaV-Objekt
				// selektiert,
				// das die ATG tatsächlich enthält
				objektAuswahlArt = ObjektAuswahlEnum.INSTANZ_ECHT;
			} else {
				// Fall 4: In der Ansicht "Objektauswahl" wurde ein DaV-Objekt
				// selektiert,
				// das die ATG nur indirekt über Hierarchien referenziert
				objektAuswahlArt = ObjektAuswahlEnum.INSTANZ_VIRTUELL;
			}
		}
		final ParameterInfo info;
		switch (objektAuswahlArt) {
		case TYP_ECHT:
			info = new ParameterInfo(selectedObj, selectedType, atgVerkehrsLageVerfahren1);
			break;
		case INSTANZ_ECHT:
			info = new ParameterInfo(selectedObj, atgVerkehrsLageVerfahren1);
			break;
		case TYP_VIRTUELL:
		case INSTANZ_VIRTUELL:
			final Set<SystemObjectType> possibleTypes = new HashSet<>();
			possibleTypes.addAll(parents);
			possibleTypes.retainAll(ptypcache.getDirectParents(atgVerkehrsLageVerfahren1));
			SystemObjectType t = null;
			if (possibleTypes.isEmpty()) {
				final String s = "Keine echten Elternobjekte für Objekt " + selectedObj.toString();
				Logger.getLogger(getClass().getName()).warning(s);
				throw new IllegalArgumentException(s);
			}
			if (possibleTypes.size() > 1) {
				System.out.print("Provide PID of type, possible values: ");
				System.out.print(possibleTypes);
				System.out.print(": ");
				final Scanner scanner = new Scanner(System.in);
				String pid = "";
				do {
					pid = scanner.nextLine();
					t = connection.getDataModel().getType(pid);
				} while (t == null || !possibleTypes.contains(t));
			} else {
				t = possibleTypes.iterator().next();
			}
			info = new ParameterInfo(selectedObj, t, atgVerkehrsLageVerfahren1);
			break;
		default:
			final String s = "Unbekannte Auswahlart " + objektAuswahlArt;
			Logger.getLogger(getClass().getName()).warning(s);
			throw new IllegalArgumentException(s);
		}
		final ParameterManager hierarchie = MethodenBibliothek.getParameterManager(connection);
		Thread.sleep(2000L);
		// SystemObject wurzel = hierarchie.getParameterWurzel(info);
		// if (wurzel == null) {
		// Logger.getLogger(getClass().getName()).warning(
		// "Für ParameterInfo:\n" + info.toString()
		// + "\nkonnte kein Wurzelelement ermittelt werden");
		// } else {
		// Logger.getLogger(getClass().getName()).info(
		// "Für ParameterInfo:\n" + info.toString()
		// + "\nwurde als Wurzelelement ermittelt:\n"
		// + wurzel.toString());
		// }
		final Collection<SystemObject> aenderbareObjekte = hierarchie.getAenderbareObjekte(info);
		if (aenderbareObjekte == null) {
			Logger.getLogger(getClass().getName()).warning("Für ParameterInfo:\n" + info.toString()
					+ "\nkonnten die betroffenen Objekte nicht ermittelt werden");
		} else {
			aenderbareObjekte.add(selectedObj);
			if (aenderbareObjekte.isEmpty()) {
				Logger.getLogger(getClass().getName())
						.info("Für ParameterInfo:\n" + info.toString() + "\nsind keine Objekte betroffen");
			} else {
				final List<SystemObject> aenderbareObjekteListe = new ArrayList<>(aenderbareObjekte.size());
				final List<SystemObject> nichtAenderbareObjekteListe = new ArrayList<>(aenderbareObjekte.size());
				aenderbareObjekteListe.addAll(aenderbareObjekte);
				Collections.sort(aenderbareObjekteListe, new SystemObjectComparatorAlphabetical(true));
				final StringBuilder b = new StringBuilder("Für ParameterInfo:\n");
				b.append(info.toString());
				b.append("\nsind folgende ");
				b.append(aenderbareObjekteListe.size());
				b.append(" Objekte potentiell betroffen:\n");
				Logger.getLogger(getClass().getName())
						.info("Bestimme Parameterquellen für " + aenderbareObjekte.size() + " Objekte...");
				final ParameterInfo[] einzelInfos = new ParameterInfo[aenderbareObjekte.size()];
				int einzelInfosLoop = 0;
				for (final SystemObject o : aenderbareObjekte) {
					einzelInfos[einzelInfosLoop++] = new ParameterInfo(o, atgVerkehrsLageVerfahren1);
				}
				final Parameter[] einzelParameterIst = hierarchie.getParameter(einzelInfos);
				for (final Parameter p : einzelParameterIst) {
					final SystemObject o = p.getObjekt();
					b.append(o.toString());
					b.append(" --> Quelle: ");
					b.append(p.getQuelle().getTyp());
					if (!ObjektAuswahlEnum.INSTANZ_ECHT.equals(objektAuswahlArt) && !selectedObj.equals(o)) {
						switch (p.getQuelle().getTyp()) {
						case LOKAL_OBJEKT:
							aenderbareObjekteListe.remove(o);
							nichtAenderbareObjekteListe.add(o);
							break;
						default:
							break;
						}
					}
					b.append('\n');
				}
				Logger.getLogger(getClass().getName()).info(b.toString());

				// purgeAenderbareMenge(hierarchie, wurzel, info.getAtg(),
				// aenderbareObjekteListe);
				Logger.getLogger(getClass().getName())
						.info("Zusammenfassung:\n  Potentiell betroffen: " + aenderbareObjekte.size()
								+ " Objekte.\n  Wirklich betroffen: " + aenderbareObjekteListe.size() + " Objekte.");
				if (setzen) {
					final Data data = connection.createData(atgVerkehrsLageVerfahren1);
					data.getUnscaledValue("v1").set(v1);
					data.getUnscaledValue("v2").set(v2);
					data.getUnscaledValue("k1").set(k1);
					data.getUnscaledValue("k2").set(k2);
					data.getItem("Urlasser").getReferenceValue("BenutzerReferenz")
							.setSystemObject(connection.getLocalUser());
					data.getItem("Urlasser").getTextValue("Ursache").setText(ursache);
					data.getItem("Urlasser").getTextValue("Veranlasser").setText("");
					final Parameter parameterSoll = new Parameter(info,
							new DataWithTime(data, System.currentTimeMillis()));
					hierarchie.setParameter(parameterSoll);
				}
				if (deleteNichtAenderbare) {
					final Parameter[] einzelParameterSoll = new Parameter[nichtAenderbareObjekteListe.size()];
					int loop;
					for (loop = 0; loop < einzelParameterSoll.length; ++loop) {
						einzelParameterSoll[loop] = new Parameter(einzelInfos[loop]);
					}
					hierarchie.setParameter(einzelParameterSoll);
				}
			}
		}
		final Parameter[] parameters = hierarchie.getParameter(info);
		if (parameters != null) {
			for (final Parameter p : parameters) {
				System.out.println(p.getQuelle());
			}
		}
		if (delete && parameters != null) {
			for (final Parameter p : parameters) {
				p.setDataWithTime(null);
				hierarchie.setParameter(p);
			}
		}
		System.exit(0);
	}

	@Override
	public void parseArguments(final ArgumentList argumentList) throws Exception {
		selectedObjectPid = argumentList.fetchArgument("-selectedObjectPid").asString();
		v1 = argumentList.fetchArgument("-v1=41").intValueBetween(-3, 254);
		v2 = argumentList.fetchArgument("-v2=85").intValueBetween(-3, 254);
		k1 = argumentList.fetchArgument("-k1=12").intValueBetween(-3, 10000);
		k2 = argumentList.fetchArgument("-k2=43").intValueBetween(-3, 10000);
		delete = argumentList.fetchArgument("-delete=false").booleanValue();
		deleteNichtAenderbare = argumentList.fetchArgument("-deleteNichtAenderbare=false").booleanValue();
		setzen = argumentList.fetchArgument("-setzen=false").booleanValue();
		ursache = argumentList.fetchArgument("-ursache=Test des Param-Plug-In (AU)").asNonEmptyString();
	}

	/**
	 * Startpunkt.
	 *
	 * @param args
	 *            die Argumentliste enthält nur die allgemeinen DaV-App-Argumente
	 */
	public static void main(final String[] args) {
		StandardApplicationRunner.run(new VerkehrslageVerfahren1Parametrierer(), "typ.applikation", args);
	}

}
