/*
 * Rahmenwerk-Plug-in "Parametrierung"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.param.viewer;

import org.eclipse.core.runtime.Assert;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.ContentViewer;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Cursor;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IViewReference;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PlatformUI;

import de.bsvrz.buv.plugin.param.editors.ParameterSaveResultEnum;
import de.bsvrz.buv.plugin.param.editors.ParameterSaver;
import de.bsvrz.buv.plugin.param.editors.ParameterSaver.ParameterSaveUrsache;
import de.bsvrz.buv.plugin.param.internal.ParamPlugin;
import de.bsvrz.buv.plugin.param.jobs.AnsichtAktualisierenViaReloadCurrentInputJobRule;
import de.bsvrz.buv.plugin.param.jobs.ReloadCurrentInputJob;
import de.bsvrz.buv.plugin.param.provider.AbstractParamPluginContentProvider;
import de.bsvrz.buv.plugin.param.views.AbstractParamPluginView;
import de.bsvrz.buv.plugin.param.views.AbstractParameterKopierenView;
import de.bsvrz.buv.plugin.param.views.BetroffeneObjekteView;
import de.bsvrz.buv.plugin.param.views.ObjektauswahlView;
import de.bsvrz.puk.param.lib.Parameter;
import de.bsvrz.puk.param.lib.ParameterClientException;
import de.bsvrz.sys.funclib.debug.Debug;

/**
 * Abstrakte Basisklasse für SelectionAdapter am 'Kopieren'-Button.
 *
 * @author BitCtrl Systems GmbH, Albrecht Uhlmann
 */
abstract class AbstractParameterKopierenButtonSelectionAdapter extends SelectionAdapter {

	/** Debug-Logger. */
	private static final Debug DEBUG = Debug.getLogger();

	/** Der Viewer, zu dem wir gehören. */
	private final AbstractParameterKopierenViewer viewer;

	/**
	 * Konstruktor merkt sich den viewer und stellt sicher, dass dieser nicht
	 * <code>null</code> ist.
	 *
	 * @param viewer
	 *            der Viewer, an dessen "Kopieren"- Button wir angeschlossen sind.
	 */
	public AbstractParameterKopierenButtonSelectionAdapter(final AbstractParameterKopierenViewer viewer) {
		this.viewer = viewer;
		Assert.isNotNull(viewer);
	}

	/**
	 * Wird aufgerufen, wenn
	 * {@link #widgetSelected(org.eclipse.swt.events.SelectionEvent)} mit einem für
	 * das Kopieren gültigen Selektion aufgerufen wurde.
	 *
	 * @param selection
	 *            Eine nicht-leere Auswahl
	 * @param input
	 *            Ein Feld von Parametern mit mindestens einem Element
	 * @param selectedSimulationsVariante
	 *            die im Viewer ausgewählte Simulationsvariante
	 */
	protected abstract void handleWidgetSelectedSafe(ISelection selection, Parameter[] input,
			short selectedSimulationsVariante);

	/**
	 * Liefert die Ursache für den Speichervorgang (Kopieren oder Tauschen).
	 *
	 * @return die Ursache
	 */
	protected abstract ParameterSaveUrsache getSaveUrsache();

	@Override
	public void widgetSelected(final SelectionEvent event) {
		if (viewer.getSelection().isEmpty() || !(viewer.getSelection() instanceof IStructuredSelection)
				|| (null == viewer.getInput())) {
			return;
		}
		handleWidgetSelectedSafe(viewer.getSelection(), (Parameter[]) viewer.getInput(),
				viewer.getSelectedSimulationsVariante());
	}

	/**
	 * Die eigentliche Speichern-Methode.
	 *
	 * @param inputParameters
	 *            das Feld der Quellparameter. Kann <code>null</code> sein.
	 * @param editedParameters
	 *            das Feld der zu speichernden Parameter
	 */
	protected void doSaveEditedParameters(final Parameter[] inputParameters, final Parameter[] editedParameters) {
		final ParameterSaveResultEnum saveResult = ParameterSaver.saveParameters(editedParameters,
				ParameterSaver.buildDefaultUrsache(inputParameters, editedParameters, getSaveUrsache()));
		switch (saveResult) {
		case PARAMETER_SAVE_FAILURE:
			final MessageDialog md = new MessageDialog(null, "Fehler beim Speichern von Parametern", null,
					ParameterSaver.getLastError(), MessageDialog.ERROR, new String[] { "OK" }, 0);
			md.open();
			break;
		case PARAMETER_SAVE_CANCEL:
			// Do nothing
			break;
		case PARAMETER_SAVE_SUCCESS:
			final IWorkbenchWindow window = PlatformUI.getWorkbench().getActiveWorkbenchWindow();
			if (window != null) {
				final String[] partIds = new String[] { ObjektauswahlView.ID, BetroffeneObjekteView.ID };
				for (final String id : partIds) {
					final IWorkbenchPart part = window.getActivePage().findView(id);
					if (part instanceof AbstractParamPluginView) {
						final AbstractParamPluginView paramPluginView = (AbstractParamPluginView) part;
						if (ParamPlugin.getDefault().isUsingReloadJobs()) {
							final ReloadCurrentInputJob job = new ReloadCurrentInputJob(paramPluginView, true);
							job.setRule(new AnsichtAktualisierenViaReloadCurrentInputJobRule());
							job.schedule();
						} else {
							final ContentViewer viewViewer = (ContentViewer) paramPluginView.getSite()
									.getSelectionProvider();
							final AbstractParamPluginContentProvider contentProvider = (AbstractParamPluginContentProvider) viewViewer
									.getContentProvider();
							final Display display = PlatformUI.getWorkbench().getDisplay();
							final Shell shell = display.getActiveShell();
							Cursor waitCursor = null;
							try {
								if (null != shell) {
									waitCursor = new Cursor(Display.getDefault(), SWT.CURSOR_WAIT);
									shell.setCursor(waitCursor);
								}
								contentProvider.reloadCurrentInput(true);
							} catch (final ParameterClientException e) {
								ParamPlugin.getDefault().getLogger().error("Laden des neuen Datensatzes fehlgeschlagen",
										e);
								throw new IllegalStateException(e);
							} finally {
								if (null != shell) {
									shell.setCursor(null);
								}
								if (null != waitCursor) {
									waitCursor.dispose();
								}
							}
						}
					}
				}
				if (null != viewer.getDialogId()) {
					final String secondaryId = AbstractParameterKopierenView.buildSecondaryId(inputParameters);
					final IViewReference part = window.getActivePage().findViewReference(viewer.getDialogId(),
							secondaryId);
					if (null != part) {
						final String partName = part.getPartName() + '-' + secondaryId;
						part.getPage().hideView(part);
						AbstractParameterKopierenButtonSelectionAdapter.DEBUG.fine("Ansicht geschlossen", partName);
					}
				}
			}
			break;
		default:
			break;
		}
	}

	/**
	 * Liefert den Viewer, zu dem wir gehören.
	 *
	 * @return der Viewer
	 */
	protected final AbstractParameterKopierenViewer getViewer() {
		return viewer;
	}

}
