/*
 * Rahmenwerk-Plug-in "Parametrierung"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.param.viewer;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.jface.viewers.ContentViewer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.ScrolledComposite;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Group;

import de.bsvrz.buv.plugin.param.IArtDesParametersatzesProvider;
import de.bsvrz.dav.daf.main.DataDescription;
import de.bsvrz.puk.param.lib.Parameter;

/**
 * Abstrakte Superklasse für die ContentViewer der Ansichten
 * {@link de.bsvrz.buv.plugin.param.views.ParameterKopierenView} und
 * {@link de.bsvrz.buv.plugin.param.views.MehrereParameterKopierenView}.
 *
 * @author BitCtrl Systems GmbH, Albrecht Uhlmann
 */
public abstract class AbstractParameterKopierenViewer extends ContentViewer {

	/**
	 * Die Grupe mit den Controls für die Zielobjekte der Operation.
	 */
	private final Group nachGroup;

	/**
	 * Die Schaltfläche 'Kopieren'.
	 */
	private Button kopierenButton;

	/**
	 * Auswalbox der Simulationsvarianten.
	 */
	private Combo simulationsVarianteCombo;

	/**
	 * Die tatsächlich ausgewählte Simulationsvariante.
	 */
	private short selectedSimulationsVariante = DataDescription.NO_SIMULATION_VARIANT_SET;

	/**
	 * Das Hauptcontrol, in dem sich Von- und Nachgroup befinden.
	 */
	private final Composite mainControl;

	/**
	 * Gruppe für die Controls mit den Quellobjekt der Operation.
	 */
	private final Group vonGroup;

	/**
	 * ID des enthaltenden Dialogs/Views. Kann im {@link #kopierenButton} Selection
	 * Listener benutzt werden, um bei Erfolgreichem Kopieren/Tauschen den
	 * Dialog/View zu schließen
	 */
	private final String dialogId;

	/**
	 * Hilfekontext-Id des enthaltenden Dialogs/Views. Wird dem
	 * Mehrfachauswahl-Dialog übergeben.
	 */
	private final String hilfeId;

	/**
	 * Konstruktor übernimmt Member und erzeugt alle Controls, teilweise unter
	 * Zuhilfenahme abstrakter MEthoden, die dann in den Kindklassen
	 * unterschiedliche Dinge tun.
	 *
	 * @param parent
	 *            Elterncontrol.
	 * @param dialogId
	 *            Dialog- oder View-Id, in der wir uns befinden.
	 * @param hilfeId
	 *            Id für den F1 Hilfecontext.
	 */
	public AbstractParameterKopierenViewer(final Composite parent, final String dialogId, final String hilfeId) {
		super();
		this.dialogId = dialogId;
		this.hilfeId = hilfeId;

		final ScrolledComposite scolledComposite = new ScrolledComposite(parent, SWT.H_SCROLL | SWT.V_SCROLL);
		final GridLayout slayout = new GridLayout(1, false);
		scolledComposite.setLayout(slayout);
		//
		// 3-spaltiges Mutter-Composite
		//
		mainControl = new Composite(scolledComposite, SWT.NONE);
		final GridLayout layout = new GridLayout(3, false);
		mainControl.setLayout(layout);

		scolledComposite.setContent(mainControl);
		scolledComposite.setExpandHorizontal(true);
		scolledComposite.setExpandVertical(true);
		scolledComposite.setMinWidth(800);
		scolledComposite.setMinHeight(300);

		GridData gd;
		vonGroup = new Group(mainControl, SWT.NONE);
		final GridLayout vonGroupLayout = new GridLayout(2, false);
		vonGroup.setLayout(vonGroupLayout);
		vonGroup.setText(getVonGroupLabel());
		gd = new GridData(SWT.FILL, SWT.FILL, false, false);
		vonGroup.setLayoutData(gd);

		fillVonGroup();

		createKopierenButton();

		// Die Gruppe "Nach", in dieser befindet sich ein Treeviewer, welcher in
		// etwa dasselbe anzeigt wie der Objektauswahl-Viewer, sowie die
		// Combobox zur Auswahl der Simulationsvariante
		nachGroup = new Group(mainControl, SWT.NONE);
		final GridLayout nachGroupLayout = new GridLayout(2, false);
		nachGroup.setLayout(nachGroupLayout);
		nachGroup.setText(getNachGroupLabel());
		gd = new GridData(SWT.FILL, SWT.FILL, true, false);
		nachGroup.setLayoutData(gd);
		fillNachGroup();
	}

	/**
	 * Einen neuen {@link SelectionListener} für den Kopieren-Button
	 * {@link #kopierenButton} bereitstellen.
	 *
	 * @return der Listener
	 */
	protected abstract SelectionListener getKopierenButtonSelectionListener();

	/**
	 * Die linke Seite des Viewers ausfüllen.
	 */
	protected abstract void fillVonGroup();

	/**
	 * Liefert die Beschriftung für die linke Gruppe.
	 *
	 * @return der Text
	 */
	protected abstract String getVonGroupLabel();

	/**
	 * Liefert die Beschriftung für die rechte Gruppe.
	 *
	 * @return der Text
	 */
	protected abstract String getNachGroupLabel();

	/**
	 * Liefert den Text für den {@link #kopierenButton}.
	 *
	 * @return der Text
	 */
	protected abstract String getKopierenButtonText();

	/**
	 * Liefert das Image für den {@link #kopierenButton}.
	 *
	 * @return das Image
	 */
	protected abstract Image getKopierenButtonImage();

	/**
	 * die rechte Seite des Viewers ausfüllen.
	 */
	protected abstract void fillNachGroup();

	/**
	 * Erzeugt die Schaltfläche 'Kopieren', die Eigenschaften werden von den
	 * implementierenden Klassen erfragt.
	 */
	protected void createKopierenButton() {
		GridData gd;
		kopierenButton = new Button(mainControl, SWT.PUSH | SWT.LEFT);
		kopierenButton.setText(getKopierenButtonText());
		gd = new GridData(SWT.CENTER, SWT.CENTER, false, false);
		gd.widthHint = 110;
		kopierenButton.setLayoutData(gd);
		kopierenButton.setImage(getKopierenButtonImage());
		kopierenButton.setEnabled(false);
		kopierenButton.addSelectionListener(getKopierenButtonSelectionListener());
	}

	/**
	 * Erzeugt die Auswalbox der Simulationsvarianten und registriert Listener, die
	 * {@link #selectedSimulationsVariante} aktualisieren.
	 */
	protected void createSimulationsVarianteCombo() {
		GridData gd;
		simulationsVarianteCombo = new Combo(nachGroup, SWT.DROP_DOWN | SWT.READ_ONLY);
		gd = new GridData(SWT.FILL, SWT.CENTER, true, false);
		gd.horizontalSpan = 1;
		simulationsVarianteCombo.setLayoutData(gd);
		simulationsVarianteCombo.addSelectionListener(new SelectionListener() {

			@Override
			public void widgetSelected(final SelectionEvent e) {
				final Short s = (Short) simulationsVarianteCombo.getData(simulationsVarianteCombo.getText());
				if (s != null) {
					selectedSimulationsVariante = s;
				}
			}

			@Override
			public void widgetDefaultSelected(final SelectionEvent e) {
				// leer
			}
		});
	}

	@Override
	public Control getControl() {
		return mainControl;
	}

	@Override
	protected void inputChanged(final Object input, final Object oldInput) {
		refresh();
	}

	/**
	 * Liefert die tatsächlich ausgewählte Simulationsvariante.
	 *
	 * @return die Simulationsvariante
	 */
	public short getSelectedSimulationsVariante() {
		return selectedSimulationsVariante;
	}

	/**
	 * Aktualisiert die Combobox mit den aktuell gültigen Simulationsvarianten.
	 *
	 * @param p
	 *            ein Parameter oder <code>null</code>. Wenn ungleich
	 *            <code>null</code>, so wird die Simulationsvariante dieses
	 *            Parameters, sofern sie gültig ist, vorselektiert.
	 * @param backend
	 *            Providerobjekt für die gültigen Simulationen
	 */
	protected void updateSimulationsVarianteCombo(final Parameter p, final IArtDesParametersatzesProvider backend) {
		simulationsVarianteCombo.removeAll();
		final List<String> nameListe = new ArrayList<>();
		String preSetText = null;
		short preSetSim = DataDescription.NO_SIMULATION_VARIANT_SET;
		for (final Short s : backend.getValidSimlationsVarianten()) {
			final String key = backend.getSimulationsVarianteString(s);
			if ((null != p) && (p.getSim() == s)) {
				preSetText = key;
				preSetSim = s;
			}
			nameListe.add(key);
			simulationsVarianteCombo.setData(key, s);
		}
		simulationsVarianteCombo.setItems(nameListe.toArray(new String[nameListe.size()]));
		if (null != preSetText) {
			simulationsVarianteCombo.setText(preSetText);
			selectedSimulationsVariante = preSetSim;
		}
	}

	/**
	 * Liefert die ID des enthaltenden Dialogs/Views. Kann im kopierenButton
	 * Selection Listener benutzt werden, um bei Erfolgreichem Kopieren/Tauschen den
	 * Dialog/View zu schließen
	 *
	 * @return die ID oder <code>null</code>
	 */
	public String getDialogId() {
		return dialogId;
	}

	/**
	 * Liefert die Hilfekontext-Id des enthaltenden Dialogs/Views. Wird dem
	 * Mehrfachauswahl-Dialog übergeben.
	 *
	 * @return die Id
	 */
	public String getHilfeId() {
		return hilfeId;
	}

	/**
	 * Bestimmt, ob mehr als ein Zielobjekt für die Aktion ausgewählt werden darf.
	 * Hat nichts mit dem
	 * {@link de.bsvrz.buv.plugin.param.provider.helper.MehrfachauswahlDialog} zu
	 * tun, es geht um die prinzipielle Machbarkeit! Die Methode kann zur Bestimmung
	 * des Flags SWT.MULTI an Trees oder Tables benutzt werden
	 *
	 * @return true - Ja, mehrere Zielobjekte erlaubt.
	 */
	protected boolean isMultipleTargetsAllowed() {
		return true;
	}

	/**
	 * Liefert die Grupe mit den Controls für die Zielobjekte der Operation.
	 *
	 * @return die Grupe
	 */
	protected final Group getNachGroup() {
		return nachGroup;
	}

	/**
	 * Liefert die Schaltfläche 'Kopieren'.
	 *
	 * @return die Schaltfläche
	 */
	protected final Button getKopierenButton() {
		return kopierenButton;
	}

	/**
	 * Liefert die Gruppe für die Controls mit den Quellobjekt der Operation.
	 *
	 * @return die Gruppe
	 */
	protected final Group getVonGroup() {
		return vonGroup;
	}
}
