/*
 * Rahmenwerk-Plug-in "Parametrierung"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.param.viewer;

import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.Date;
import java.util.Iterator;
import java.util.List;

import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.ContentViewer;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.ScrolledComposite;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;

import de.bsvrz.buv.plugin.param.IArtDesParametersatzesProvider;
import de.bsvrz.buv.plugin.param.ParamSharedImage;
import de.bsvrz.buv.plugin.param.Zeichenketten;
import de.bsvrz.buv.plugin.param.internal.RahmenwerkService;
import de.bsvrz.buv.plugin.param.provider.ArtDesParametersatzesContentProvider;
import de.bsvrz.buv.rw.basislib.kalender.DatumZeit;

/**
 * Viewer für die Teilansicht rechts-oben in Abb 5-41.
 *
 * @author BitCtrl Systems GmbH, Albrecht Uhlmann
 */
public class ArtDesParametersatzesViewer extends ContentViewer {

	/**
	 * Realisiert den Listener für die Klicks auf die Radiobuttons.
	 *
	 * @author BitCtrl Systems GmbH, Albrecht Uhlmann
	 */
	private final class RadioButtonSelectionListener implements SelectionListener {

		/**
		 * Aus irgendeinem Grund wird für den
		 * {@link ArtDesParametersatzesViewer#originalRadioButton} diese Methode einmal
		 * aufgerufen, nachdem die Radio-Group fertig gebaut ist - auch ohne User-Klick,
		 * setInput o.ä. Das überschreibt Werte im backend.
		 */
		private boolean first = true;

		@Override
		public void widgetSelected(final SelectionEvent e) {
			if ((getInput() instanceof IArtDesParametersatzesProvider) && (e.widget instanceof Button)
					&& ((Button) e.widget).getSelection() && (!first || (e.widget != originalRadioButton))) {
				enableOrDisableControls((Button) e.widget);
				buildAndPublishCurrentSelection();
			}
			first = false;
		}

		@Override
		public void widgetDefaultSelected(final SelectionEvent e) {
			widgetSelected(e);
		}
	}

	/**
	 * Hauptcomposite für alle Controls.
	 */
	private final ScrolledComposite mainControl;

	/**
	 * Der Radio-Knopf 'Online'.
	 */
	private final Button originalRadioButton;

	/**
	 * Der Radio-Knopf 'Historisch'.
	 */
	private final Button historischRadioButton;

	/**
	 * Der Radio-Knopf 'Simulation'.
	 */
	private final Button simulationRadioButton;

	/**
	 * Eine Group zur Zusammenfassung der drei Radio-Knöpfe.
	 */
	private final Group mainGroup;

	/**
	 * Auswahlbox für die möglichen Simulationsvarianten.
	 */
	private final Combo simulationsVarianteCombo;

	/**
	 * Wenn wir in {@link #refresh()} die Items löschen und neu über
	 * {@link #getContentProvider()}.getElements() auslesen, werden massenweise
	 * Modify-Events erzeugt, die wiederum auf das Backend durchschreiben würden.
	 * das muss vermieden werden.
	 */
	private boolean allowComboSelectionchanged = true;

	/**
	 * Rahmenwerk-Control zur asuwahl des Zeitpunktes, für den historische Parameter
	 * abgefragt werden sollen.
	 */
	private final DatumZeit datumZeit;

	/**
	 * Starte mit leerer Auswahl.
	 */
	private ISelection currentSelection = new StructuredSelection();

	/**
	 * Die Schaltfläche 'Aktualisieren'. Aktiv nur, wenn der Radio-Knopf
	 * 'Historisch' gewählt wurd UND das Archiv verfügbar ist UND ein Zeitpunkt aus
	 * der Vergangenheit gewählt wurde. In den beiden letzteren Fällen wird auch der
	 * Text auf der Schaltfläche entsprechend geändert.
	 */
	private final Button aktualisierenButton;

	/**
	 * Konstruktor erzeugt die Widgets und registriert Listener an selbigen.
	 *
	 * @param parent
	 *            Eltern-Composite
	 */
	public ArtDesParametersatzesViewer(final Composite parent) {
		//
		// 1-spaltiges Mutter-Composite
		//
		mainControl = new ScrolledComposite(parent, SWT.H_SCROLL | SWT.V_SCROLL);
		final GridLayout layout = new GridLayout(1, false);
		mainControl.setLayout(layout);

		GridData gd;
		mainGroup = new Group(mainControl, SWT.NONE);
		final GridLayout mainGroupLayout = new GridLayout(3, false);
		mainGroup.setLayout(mainGroupLayout);
		mainGroup.setText(Zeichenketten.PLUGIN_PARAM_BEZEICHNER_ART_DES_PARAMETERSATZES);
		mainGroup.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, false));
		mainControl.setContent(mainGroup);
		mainControl.setExpandHorizontal(true);
		mainControl.setExpandVertical(true);
		mainControl.setMinWidth(350);
		mainControl.setMinHeight(100);
		// Könnte man machen, sieht aber nicht besonders gut aus.
		// mainControl.setAlwaysShowScrollBars(true);

		originalRadioButton = new Button(mainGroup, SWT.RADIO);
		gd = new GridData(SWT.LEFT, SWT.TOP, false, false);
		gd.horizontalSpan = 3;
		gd.verticalAlignment = SWT.CENTER;
		gd.horizontalAlignment = SWT.LEFT;
		originalRadioButton.setLayoutData(gd);
		originalRadioButton.setText(IArtDesParametersatzesProvider.ART_ORIGINAL);
		originalRadioButton.addSelectionListener(new RadioButtonSelectionListener());
		simulationRadioButton = new Button(mainGroup, SWT.RADIO);
		gd = new GridData(SWT.LEFT, SWT.TOP, false, false);
		gd.horizontalSpan = 1;
		gd.horizontalAlignment = SWT.LEFT;
		gd.verticalAlignment = SWT.CENTER;
		simulationRadioButton.setLayoutData(gd);
		simulationRadioButton.setText(IArtDesParametersatzesProvider.ART_SIMULATION);
		simulationRadioButton.addSelectionListener(new RadioButtonSelectionListener());
		final Label l = new Label(mainGroup, SWT.NONE);
		gd = new GridData(SWT.LEFT, SWT.TOP, true, false);
		gd.horizontalSpan = 1;
		gd.horizontalAlignment = SWT.RIGHT;
		gd.verticalAlignment = SWT.CENTER;
		l.setLayoutData(gd);
		l.setText("Variante:");
		simulationsVarianteCombo = new Combo(mainGroup, SWT.DROP_DOWN | SWT.READ_ONLY);
		gd = new GridData(SWT.LEFT, SWT.TOP, true, false);
		gd.horizontalSpan = 1;
		gd.minimumWidth = 100;
		gd.horizontalAlignment = SWT.FILL;
		gd.verticalAlignment = SWT.CENTER;
		simulationsVarianteCombo.setLayoutData(gd);
		simulationsVarianteCombo.addModifyListener(new ModifyListener() {

			@Override
			public void modifyText(final ModifyEvent e) {
				if (allowComboSelectionchanged) {
					buildAndPublishCurrentSelection();
				}
			}
		});

		historischRadioButton = new Button(mainGroup, SWT.RADIO);
		gd = new GridData(SWT.LEFT, SWT.TOP, false, false);
		gd.horizontalSpan = 1;
		gd.horizontalAlignment = SWT.LEFT;
		gd.verticalAlignment = SWT.CENTER;
		historischRadioButton.setLayoutData(gd);
		historischRadioButton.setText(IArtDesParametersatzesProvider.ART_HISTORISCH + " vom:");
		historischRadioButton.addSelectionListener(new RadioButtonSelectionListener());

		final DatumZeit.Eingabetyp eingabeTyp = DatumZeit.Eingabetyp.datumuhrMS;
		datumZeit = new DatumZeit(mainGroup, SWT.NONE, eingabeTyp, false, false);
		datumZeit.setToolTipText("Datum und Uhrzeit für die Archivabfrage");
		datumZeit.setLayout(new GridLayout());
		long initialTime = System.currentTimeMillis() - 3600000L;
		initialTime /= 1000L;
		initialTime *= 1000;
		datumZeit.setDatum(new Date(initialTime));
		gd = new GridData(SWT.LEFT, SWT.TOP, true, false);
		gd.horizontalSpan = 1;
		gd.horizontalAlignment = SWT.FILL;
		gd.verticalAlignment = SWT.CENTER;
		gd.minimumWidth = 130;
		datumZeit.setLayoutData(gd);
		datumZeit.hinzufuegenSelektionsZuhoerer(new SelectionListener() {

			@Override
			public void widgetSelected(final SelectionEvent e) {
				final long now = RahmenwerkService.getService().getObjektFactory().getDav().getTime();
				final long newTime = datumZeit.getDatum().getTime();
				if (newTime > now) {
					aktualisierenButton.setText("Zukunft!");
					aktualisierenButton.setImage(null);
					aktualisierenButton.setEnabled(false);
				} else {
					aktualisierenButton.setEnabled(true);
					aktualisierenButton.setText(Zeichenketten.PLUGIN_PARAM_BUTTON_AKTUALISIEREN);
					aktualisierenButton.setImage(ParamSharedImage.AKTUALISIEREN.getImage());
				}
			}

			@Override
			public void widgetDefaultSelected(final SelectionEvent e) {
				widgetSelected(e);
			}
		});

		aktualisierenButton = new Button(mainGroup, SWT.PUSH);
		aktualisierenButton.setText(Zeichenketten.PLUGIN_PARAM_BUTTON_AKTUALISIEREN);
		aktualisierenButton.setImage(ParamSharedImage.AKTUALISIEREN.getImage());
		aktualisierenButton.addSelectionListener(new SelectionListener() {

			@Override
			public void widgetSelected(final SelectionEvent e) {
				if ((getInput() != null) && (getInput() instanceof IArtDesParametersatzesProvider)) {
					final long now = RahmenwerkService.getService().getObjektFactory().getDav().getTime();
					final long newTime = datumZeit.getDatum().getTime();
					if (newTime > now) {
						// Hierher dürfte man eigentlich gar nicht kommen, da
						// die Schaltfläche 'Aktualisieren' bereits disabled
						// ist, aber...
						MessageDialog.openError(null, "Fehler",
								"Bitte wählen Sie einen Zeitpunkt aus der Vergangenheit!");
					} else {
						buildAndPublishCurrentSelection();
					}
				}

			}

			@Override
			public void widgetDefaultSelected(final SelectionEvent e) {
				widgetSelected(e);
			}
		});
		gd = new GridData(SWT.LEFT, SWT.TOP, true, false);
		gd.horizontalSpan = 1;
		gd.horizontalAlignment = SWT.FILL;
		gd.verticalAlignment = SWT.CENTER;
		aktualisierenButton.setLayoutData(gd);
	}

	@Override
	public Control getControl() {
		return mainControl;
	}

	@Override
	public ISelection getSelection() {
		return currentSelection;
	}

	@Override
	public void refresh() {
		if ((getInput() instanceof IArtDesParametersatzesProvider)
				&& (getContentProvider() instanceof ArtDesParametersatzesContentProvider)) {
			final IArtDesParametersatzesProvider backend = (IArtDesParametersatzesProvider) getInput();
			final ArtDesParametersatzesContentProvider provider = (ArtDesParametersatzesContentProvider) getContentProvider();
			allowComboSelectionchanged = false;
			simulationsVarianteCombo.removeAll();
			final List<String> nameListe = new ArrayList<>();
			for (final Object o : provider.getElements(IArtDesParametersatzesProvider.ART_SIMULATION)) {
				if (o instanceof Short) {
					final short s = (Short) o;
					final String key = backend.getSimulationsVarianteString(s);
					nameListe.add(key);
					simulationsVarianteCombo.setData(key, s);
				}
			}
			simulationsVarianteCombo.setItems(nameListe.toArray(new String[nameListe.size()]));

			Button selectedRadio = null;
			if (IArtDesParametersatzesProvider.ART_ORIGINAL.equals(backend.getArtDesParametersatzes())) {
				originalRadioButton.setSelection(true);
				selectedRadio = originalRadioButton;
				simulationRadioButton.setSelection(false);

				simulationsVarianteCombo.setEnabled(false);
				historischRadioButton.setSelection(false);
				datumZeit.setEnabled(false);
			} else if (IArtDesParametersatzesProvider.ART_SIMULATION.equals(backend.getArtDesParametersatzes())) {
				originalRadioButton.setSelection(false);
				simulationRadioButton.setSelection(true);
				selectedRadio = simulationRadioButton;
				simulationsVarianteCombo.setEnabled(true);
				historischRadioButton.setSelection(false);
				datumZeit.setEnabled(false);
				datumZeit.setDatum(new Date(backend.getZeitpunkt()));
			} else if (IArtDesParametersatzesProvider.ART_HISTORISCH.equals(backend.getArtDesParametersatzes())) {
				originalRadioButton.setSelection(false);
				simulationRadioButton.setSelection(false);
				simulationsVarianteCombo.setEnabled(false);
				historischRadioButton.setSelection(true);
				selectedRadio = historischRadioButton;
				datumZeit.setEnabled(true);
				datumZeit.setDatum(new Date(backend.getZeitpunkt()));
			}
			final String text = backend.getSimulationsVarianteString(backend.getSelectedSimulationsVariante());
			if (nameListe.contains(text)) {
				simulationsVarianteCombo.setText(text);
			}
			allowComboSelectionchanged = true;
			enableOrDisableControls(selectedRadio);
			getControl().redraw();
		}
	}

	@Override
	public void setSelection(final ISelection selection, final boolean reveal) {
		originalRadioButton.setSelection(false);
		simulationRadioButton.setSelection(false);
		historischRadioButton.setSelection(false);
		if ((getInput() instanceof IArtDesParametersatzesProvider) && !selection.isEmpty()
				&& (selection instanceof IStructuredSelection)
				&& (((IStructuredSelection) selection).getFirstElement() instanceof String)) {
			final Iterator<?> iterator = ((IStructuredSelection) selection).iterator();
			final String selectedRadio = (String) iterator.next();
			if (IArtDesParametersatzesProvider.ART_ORIGINAL.equals(selectedRadio)) {
				originalRadioButton.setSelection(true);
			} else if (IArtDesParametersatzesProvider.ART_SIMULATION.equals(selectedRadio) && iterator.hasNext()) {
				simulationRadioButton.setSelection(true);
				final Short s = (Short) iterator.next();
				final IArtDesParametersatzesProvider backend = (IArtDesParametersatzesProvider) getInput();
				final String text = backend.getSimulationsVarianteString(s);
				simulationsVarianteCombo.setText(text);
			} else if (IArtDesParametersatzesProvider.ART_HISTORISCH.equals(selectedRadio)) {
				historischRadioButton.setSelection(true);
				final IArtDesParametersatzesProvider backend = (IArtDesParametersatzesProvider) getInput();
				datumZeit.setDatum(new Date(backend.getZeitpunkt()));
			} else {
				throw new IllegalArgumentException("Ungültige Selektion: " + selectedRadio);
			}
			this.currentSelection = selection;
		}
	}

	/**
	 * eine StructuredSelection zusammenbauen, mit den aktuellen Werten füllen:
	 * <ul>
	 * <li>Original: Nur ein Element, der String
	 * {@link IArtDesParametersatzesProvider#ART_ORIGINAL}</li>
	 * <li>Simulation: Erstes Element ist der String
	 * {@link IArtDesParametersatzesProvider#ART_SIMULATION}, zweites Element ist
	 * die Simulationsvariante als short.</li>
	 * <li>Historisch: Erstes Element ist der String
	 * {@link IArtDesParametersatzesProvider#ART_HISTORISCH}, zweites Element ist
	 * der gewählte Zeitpunkt als {@link java.sql.Timestamp}.</li>
	 * </ul>
	 *
	 * und versenden.
	 */
	protected void buildAndPublishCurrentSelection() {
		ISelection selection;
		if (originalRadioButton.getSelection()) {
			selection = new StructuredSelection(IArtDesParametersatzesProvider.ART_ORIGINAL);
		} else if (simulationRadioButton.getSelection()) {
			Short selectedSimulationsVariante = 1000;
			final Short s = (Short) simulationsVarianteCombo.getData(simulationsVarianteCombo.getText());
			if (s != null) {
				selectedSimulationsVariante = s;
			}
			if (selectedSimulationsVariante < 1000) {
				selection = new StructuredSelection(
						new Object[] { IArtDesParametersatzesProvider.ART_SIMULATION, selectedSimulationsVariante });
			} else {
				selection = currentSelection;
			}
		} else if (historischRadioButton.getSelection()) {
			final Timestamp timestamp = new Timestamp(datumZeit.getDatum().getTime());
			selection = new StructuredSelection(
					new Object[] { IArtDesParametersatzesProvider.ART_HISTORISCH, timestamp });
		} else {
			selection = new StructuredSelection();
		}
		currentSelection = selection;
		fireSelectionChanged(new SelectionChangedEvent(this, currentSelection));
	}

	/**
	 * Die verschiedenen Steuerelemente in Abhängigkeit von der Stellung der
	 * Radio-Knöpfe aktivieren oder deaktivieren.
	 *
	 * @param selectedRadio
	 *            der aktive Radio-Knopf.
	 */
	private void enableOrDisableControls(final Button selectedRadio) {
		if (selectedRadio == originalRadioButton) {
			simulationsVarianteCombo.setEnabled(false);
			datumZeit.setEnabled(false);
			aktualisierenButton.setEnabled(false);
		} else if (selectedRadio == simulationRadioButton) {
			simulationsVarianteCombo.setEnabled(true);
			datumZeit.setEnabled(false);
			aktualisierenButton.setEnabled(false);
		} else if (selectedRadio == historischRadioButton) {
			simulationsVarianteCombo.setEnabled(false);
			boolean historischEnabled = false;
			if (getInput() instanceof IArtDesParametersatzesProvider) {
				final IArtDesParametersatzesProvider backend = (IArtDesParametersatzesProvider) getInput();
				historischEnabled = backend.isArchivAvailable();
			}
			datumZeit.setEnabled(historischEnabled);
			if (historischEnabled) {
				aktualisierenButton.setEnabled(historischEnabled);
				aktualisierenButton.setText(Zeichenketten.PLUGIN_PARAM_BUTTON_AKTUALISIEREN);
			} else {
				aktualisierenButton.setEnabled(true);
				aktualisierenButton.setText("Kein Archiv!");
				aktualisierenButton.setEnabled(false);
			}
		}
	}

}
