/*
 * Rahmenwerk-Plug-in "Parametrierung"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.param.viewer;

import org.eclipse.core.runtime.Assert;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.jface.viewers.ViewerComparator;
import org.eclipse.jface.window.Window;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Tree;

import de.bsvrz.buv.plugin.param.IArtDesParametersatzesProvider;
import de.bsvrz.buv.plugin.param.ParamSharedImage;
import de.bsvrz.buv.plugin.param.Zeichenketten;
import de.bsvrz.buv.plugin.param.internal.RahmenwerkService;
import de.bsvrz.buv.plugin.param.provider.helper.MehrfachauswahlDialog;
import de.bsvrz.puk.param.lib.Parameter;

/**
 * ContentViewer für die Ansicht
 * {@link de.bsvrz.buv.plugin.param.views.ParameterKopierenView}. Als Input
 * dient ein Objekt vom Typ {@link Parameter}, die Selection ist die Menge der
 * in dem NACH Tree ausgewählten Kopierziele (können Typen und Instanzen sein,
 * auch gemischt).
 *
 * Als Contentprovider wird ein
 * {@link de.bsvrz.buv.plugin.param.provider.ParameterKopierenContentProvider}
 * verwendet, dieser erfüllt allerdings für uns direkt keine Funktion, vielmehr
 * wird dasselbe Objekt auch dem intern verwendeten {@link TreeViewer}
 * übergeben, der die Objektauswahl für die Kopierziele realisiert. Es handelt
 * sich um eine abgespeckte Variante des
 * {@link de.bsvrz.buv.plugin.param.provider.ObjektauswahlContentProvider}.
 *
 * @author BitCtrl Systems GmbH, Albrecht Uhlmann
 */
public class ParameterKopierenViewer extends AbstractParameterKopierenViewer {

	/**
	 * Die einzelnen Labels auf der Von-Seite.
	 */
	private static final Label[] VON_LABELS_VALUES = new Label[5];

	/**
	 * Die Texte auf den einzelnen Labels.
	 */
	private static final String[] VON_LABELS_KEYS = new String[] { "Objekt:", "Typ:", "Attributgruppe:", "Sim.Var.:",
			"Quelltyp:" };
	// HINWEIS später noch dazu: Zeitstempel, Online/Historisch/Datei

	/**
	 * Treeviewer mit den möglichen Zielobjekten.
	 */
	private TreeViewer nachTreeViewer;

	/**
	 * Konstruktor reicht nur durch.
	 *
	 * @param parent
	 *            Elterncontrol.
	 * @param dialogId
	 *            Dialog- oder View-Id, in der wir uns befinden.
	 * @param hilfeId
	 *            Id für den F1 Hilfecontext.
	 */
	public ParameterKopierenViewer(final Composite parent, final String dialogId, final String hilfeId) {
		super(parent, dialogId, hilfeId);
		Assert.isTrue(
				ParameterKopierenViewer.VON_LABELS_KEYS.length == ParameterKopierenViewer.VON_LABELS_VALUES.length);
	}

	@Override
	protected void fillNachGroup() {
		GridData gd;
		Label l;
		int style = SWT.BORDER;
		if (isMultipleTargetsAllowed()) {
			style |= SWT.MULTI;
		} else {
			style |= SWT.SINGLE;
		}
		final Tree tree = new Tree(getNachGroup(), style);
		gd = new GridData(SWT.FILL, SWT.FILL, true, true);
		gd.horizontalSpan = 2;
		gd.heightHint = 220;
		tree.setLayoutData(gd);
		nachTreeViewer = new TreeViewer(tree);
		nachTreeViewer.addSelectionChangedListener(new ISelectionChangedListener() {

			@Override
			public void selectionChanged(final SelectionChangedEvent event) {
				boolean enabled = !event.getSelection().isEmpty();
				if ((((IStructuredSelection) event.getSelection()).size() > 1) && MehrfachauswahlDialog.isEnabled()) {
					final MehrfachauswahlDialog mad = new MehrfachauswahlDialog(null, getHilfeId());
					final int userResponse = mad.open();
					enabled = Window.OK == userResponse;
				}
				getKopierenButton().setEnabled(enabled);
			}
		});
		nachTreeViewer.setComparator(new ViewerComparator());
		l = new Label(getNachGroup(), SWT.NONE);
		l.setText(ParameterKopierenViewer.VON_LABELS_KEYS[3]);
		gd = new GridData(SWT.FILL, SWT.CENTER, false, false);
		gd.horizontalSpan = 1;
		l.setLayoutData(gd);

		createSimulationsVarianteCombo();
	}

	@Override
	protected void fillVonGroup() {
		GridData gd;
		Label l;
		int loop;
		for (loop = 0; loop < ParameterKopierenViewer.VON_LABELS_KEYS.length; ++loop) {
			l = new Label(getVonGroup(), SWT.NONE);
			l.setText(ParameterKopierenViewer.VON_LABELS_KEYS[loop]);
			gd = new GridData(SWT.FILL, SWT.FILL, false, false);
			l.setLayoutData(gd);
			l = new Label(getVonGroup(), SWT.NONE);
			l.setText(Zeichenketten.PARAM_PLUGIN_HINWEIS_KEINE_QUELLE);
			gd = new GridData(SWT.FILL, SWT.FILL, true, false);
			gd.widthHint = 310;
			l.setLayoutData(gd);
			ParameterKopierenViewer.VON_LABELS_VALUES[loop] = l;
		}
	}

	@Override
	protected SelectionListener getKopierenButtonSelectionListener() {
		return new ParameterKopierenButtonSelectionAdapter(this);
	}

	@Override
	public ISelection getSelection() {
		return nachTreeViewer.getSelection();
	}

	@Override
	public void refresh() {
		if (getInput() instanceof Parameter[]) {
			final Parameter[] parameters = (Parameter[]) getInput();
			final Parameter p = parameters[0];
			final IArtDesParametersatzesProvider backend = RahmenwerkService.getService()
					.getArtDesParametersatzesProvider();
			ParameterKopierenViewer.VON_LABELS_VALUES[0].setText(p.getObjekt().getName());
			ParameterKopierenViewer.VON_LABELS_VALUES[1].setText(p.getTyp().getName());
			ParameterKopierenViewer.VON_LABELS_VALUES[2].setText(p.getAtg().getName());
			ParameterKopierenViewer.VON_LABELS_VALUES[3].setText(backend.getSimulationsVarianteString(p.getSim()));
			ParameterKopierenViewer.VON_LABELS_VALUES[4].setText(p.getQuelle().getTyp().toString());
			nachTreeViewer.setContentProvider(getContentProvider());
			nachTreeViewer.setLabelProvider(getLabelProvider());
			nachTreeViewer.setInput(parameters);
			updateSimulationsVarianteCombo(p, backend);
		} else {
			int loop;
			for (loop = 0; loop < ParameterKopierenViewer.VON_LABELS_VALUES.length; ++loop) {
				ParameterKopierenViewer.VON_LABELS_VALUES[loop]
						.setText(Zeichenketten.PARAM_PLUGIN_HINWEIS_KEINE_QUELLE);
			}
		}

	}

	@Override
	public void setSelection(final ISelection selection, final boolean reveal) {
		nachTreeViewer.setSelection(selection, reveal);
	}

	@Override
	protected String getNachGroupLabel() {
		return "Nach";
	}

	@Override
	protected String getVonGroupLabel() {
		return "Von";
	}

	@Override
	protected String getKopierenButtonText() {
		return Zeichenketten.PLUGIN_PARAM_BUTTON_KOPIEREN;
	}

	@Override
	protected Image getKopierenButtonImage() {
		return ParamSharedImage.ARROW_RIGHT.getImage();
	}
}
