/*
 * Rahmenwerk-Plug-in "Parametrierung"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.param.views;

import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;

import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.Platform;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.action.Separator;
import org.eclipse.jface.viewers.ContentViewer;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ISelectionProvider;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.ui.ISelectionListener;
import org.eclipse.ui.IWorkbenchActionConstants;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.part.ViewPart;

import de.bsvrz.buv.plugin.param.Zeichenketten;
import de.bsvrz.buv.plugin.param.internal.RahmenwerkService;
import de.bsvrz.buv.plugin.param.provider.AbstractParamPluginContentProvider;
import de.bsvrz.buv.plugin.param.provider.AbstractParamPluginLabelProvider;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.sys.funclib.bitctrl.daf.DavProvider;
import de.bsvrz.sys.funclib.bitctrl.modell.util.KappichModellUtil;

/**
 * Abstrakte Superklasse für alle Ansichten des Plug-in "Parametrierung".
 *
 * @author BitCtrl Systems GmbH, Albrecht Uhlmann
 */
public abstract class AbstractParamPluginView extends ViewPart implements PropertyChangeListener, ISelectionListener {

	/**
	 * Der Container zur Darstellung der Widgets der Ansicht.
	 */
	private Composite panel;

	/**
	 * der Viewer mit eigenem Modell für die Darstellung des Inhalts des View.
	 */
	private ContentViewer contentViewer;

	/**
	 * Bestimmt, ob die Ansicht Selektionen anderer Workbench-Parts automatisch
	 * übernimmt oder nicht.
	 */
	private boolean selektionEmpfangen;

	/**
	 * Keine besonderen Aktionen im Konstruktor.
	 */
	public AbstractParamPluginView() {
		super();
	}

	/**
	 * Liefert einen Contentprovider für den in dieser Ansicht enthaltenen
	 * {@link #contentViewer}.
	 *
	 * @return der Contentprovider. Darf nicht null sein.
	 */
	protected abstract AbstractParamPluginContentProvider getContentProvider();

	/**
	 * Liefert einen Labelprovider für den in dieser Ansicht enthaltenen
	 * {@link #contentViewer}.
	 *
	 * @return der Labelprovider. Darf nicht null sein. Kann auch Color- und
	 *         Fontprovider sein.
	 */
	protected abstract AbstractParamPluginLabelProvider getLabelProvider();

	@Override
	public void setFocus() {
		if (contentViewer != null) {
			contentViewer.getControl().setFocus();
		}
	}

	/**
	 * Den Zustand der Steuerelemente bestimmen anhand verschiedener Kriterien.
	 *
	 * @param enabled
	 *            true - Steuerelemente wenn möglich einschalten
	 */
	protected void setControlsState(final boolean enabled) {
		if ((panel != null) && !panel.isDisposed()) {
			panel.getDisplay().asyncExec(new Runnable() {

				@Override
				public void run() {
					if (enabled) {
						if (getContentProvider() != null) {
							setContentDescription(getContentProvider().getKurzinfo());
						} else {
							setContentDescription("");
						}
					} else {
						setContentDescription(Zeichenketten.PLUGIN_PARAM_BEZEICHNER_KEINE_DAVVERBINDUNG);
					}
					if ((contentViewer != null) && (contentViewer.getControl() != null)
							&& !contentViewer.getControl().isDisposed()) {
						contentViewer.getControl().setEnabled(enabled);
					}
				}
			});
		}
	}

	/*
	 * (non-Javadoc)
	 *
	 * @seejava.beans.PropertyChangeListener#propertyChange(java.beans.
	 * PropertyChangeEvent)
	 */
	@Override
	public void propertyChange(final PropertyChangeEvent evt) {
		if (DavProvider.PROP_VERBUNDEN.equals(evt.getPropertyName()) && (evt.getNewValue() instanceof Boolean)) {
			if (Boolean.TRUE.equals(evt.getNewValue())) {
				verbindungAufgebaut();
			} else {
				verbindungAbgebaut();
			}
		} else if (AbstractParamPluginContentProvider.PROP_KURZINFO.equals(evt.getPropertyName())) {
			if ((panel != null) && !panel.isDisposed()) {
				final String neueKurzinfo = (String) evt.getNewValue();
				panel.getDisplay().asyncExec(new Runnable() {

					@Override
					public void run() {
						setContentDescription(neueKurzinfo);
					}
				});
			}
		}
	}

	/**
	 * Behandlung, wenn die Verbindung zum DaV (wieder) abgebaut wurde.
	 */
	protected void verbindungAbgebaut() {
		setControlsState(false);
		if ((panel != null) && !panel.isDisposed()) {
			panel.getDisplay().asyncExec(new Runnable() {

				@Override
				public void run() {
					if ((contentViewer != null) && (!contentViewer.getControl().isDisposed())) {
						contentViewer.setInput(null);
					}
				}
			});
		}
	}

	/**
	 * Behandlung, wenn die Verbindung zum DaV (wieder) aufgebaut wurde.
	 */
	protected void verbindungAufgebaut() {
		setControlsState(true);
		if ((panel != null) && (panel.getDisplay() != null) && !panel.getDisplay().isDisposed()) {
			panel.getDisplay().asyncExec(new Runnable() {

				@Override
				public void run() {
					if (contentViewer != null) {
						contentViewer
								.setInput(KappichModellUtil.getAOE(RahmenwerkService.getService().getObjektFactory()));

					}
				}
			});
		}
	}

	/**
	 * Erzeugt ein Kontextmenü für diese Ansicht.
	 */
	protected final void createContextMenu() {
		final MenuManager menuManager = new MenuManager();
		menuManager.add(new Separator(IWorkbenchActionConstants.MB_ADDITIONS));
		contentViewer.getControl().setMenu(menuManager.createContextMenu(contentViewer.getControl()));
		getSite().registerContextMenu(menuManager, contentViewer);
	}

	/*
	 * (non-Javadoc)
	 *
	 * @see org.eclipse.ui.part.WorkbenchPart#dispose()
	 */
	@Override
	public void dispose() {
		RahmenwerkService.getService().getObjektFactory().removePropertyChangeListener(DavProvider.PROP_VERBUNDEN,
				this);
		if (getContentProvider() != null) {
			getContentProvider().removePropertyChangeListener(AbstractParamPluginContentProvider.PROP_KURZINFO, this);
		}
		super.dispose();
	}

	@Override
	public void selectionChanged(final IWorkbenchPart part, final ISelection selection) {
		final ISelectionProvider selectionProvider = getViewSite().getSelectionProvider();
		if (null == selectionProvider) {
			throw new UnsupportedOperationException(
					"Die Ansicht '" + getPartName() + "' unterstützt die Übernahme von Selektionen nicht,"
							+ " weil eine solche Übernahme nicht sinnvoll zu definieren ist.");
		}
		if (!(selection instanceof IStructuredSelection)) {
			throw new UnsupportedOperationException("Die Ansicht '" + getPartName()
					+ "' erwartet eine strukturierte Selektion (IStructuredSelection).");
		}
		if (selection.isEmpty()) {
			selectionProvider.setSelection(selection);
			return;
		}
		final IStructuredSelection s = (IStructuredSelection) selection;
		final Set<SystemObject> selectedSystemObjects = new HashSet<>();
		final Iterator<?> selectedObjectsIterator = s.iterator();
		while (selectedObjectsIterator.hasNext()) {
			final Object nextSelectedObject = selectedObjectsIterator.next();
			if (nextSelectedObject instanceof SystemObject) {
				selectedSystemObjects.add((SystemObject) nextSelectedObject);
			} else {
				final Object adapted;
				if (nextSelectedObject instanceof IAdaptable) {
					adapted = ((IAdaptable) nextSelectedObject).getAdapter(SystemObject.class);
				} else {
					adapted = Platform.getAdapterManager().getAdapter(nextSelectedObject, SystemObject.class);
				}
				if (adapted instanceof SystemObject) {
					selectedSystemObjects.add((SystemObject) adapted);
				}
			}
		}
		final StructuredSelection systemObjectSelection = new StructuredSelection(selectedSystemObjects.toArray());
		if (selectionProvider instanceof Viewer) {
			((Viewer) selectionProvider).setSelection(systemObjectSelection, true);
		} else {
			selectionProvider.setSelection(systemObjectSelection);
		}
	}

	/**
	 * Liefert den Container zur Darstellung der Widgets der Ansicht.
	 *
	 * @return der Container
	 */
	protected final Composite getPanel() {
		return panel;
	}

	/**
	 * Setzt den Container zur Darstellung der Widgets der Ansicht.
	 *
	 * @param panel
	 *            der Container
	 */
	protected final void setPanel(final Composite panel) {
		this.panel = panel;
	}

	/**
	 * Liefert den Viewer mit eigenem Modell für die Darstellung des Inhalts des
	 * View.
	 *
	 * @return der Viewer
	 */
	protected final ContentViewer getContentViewer() {
		return contentViewer;
	}

	/**
	 * Setzt den Viewer mit eigenem Modell für die Darstellung des Inhalts des View.
	 *
	 * @param contentViewer
	 *            der Viewer
	 */
	protected final void setContentViewer(final ContentViewer contentViewer) {
		this.contentViewer = contentViewer;
	}

	/**
	 * Bestimmt, ob die Ansicht Selektionen anderer Workbench-Parts automatisch
	 * übernimmt oder nicht.
	 *
	 * @return true, wenn aktuell die Selektionen übernommen werden.
	 */
	public boolean isSelektionEmpfangen() {
		return selektionEmpfangen;
	}

	/**
	 * Setzt, ob die Ansicht Selektionen anderer Workbench-Parts automatisch
	 * übernimmt oder nicht.
	 *
	 * @param selektionEmpfangen
	 *            true, wenn aktuell die Selektionen übernommen werden sollen.
	 */
	public void setSelektionEmpfangen(final boolean selektionEmpfangen) {
		this.selektionEmpfangen = selektionEmpfangen;
	}
}
