/*
 * Rahmenwerk-Plug-in "Parametrierung"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */

package de.bsvrz.buv.plugin.param.views;

import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Cursor;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.PlatformUI;

import de.bsvrz.buv.plugin.param.internal.ParamPlugin;
import de.bsvrz.buv.plugin.param.internal.RahmenwerkService;
import de.bsvrz.buv.plugin.param.provider.AbstractParamPluginContentProvider;
import de.bsvrz.buv.plugin.param.provider.AbstractParamPluginLabelProvider;
import de.bsvrz.buv.plugin.param.provider.StandardParamPluginLabelProvider;
import de.bsvrz.buv.plugin.param.viewer.AbstractParameterKopierenViewer;
import de.bsvrz.puk.param.lib.Parameter;
import de.bsvrz.puk.param.lib.ParameterClientException;
import de.bsvrz.sys.funclib.bitctrl.daf.DavProvider;

/**
 * Abstrakte Basisklasse für {@link ParameterKopierenView} und
 * {@link MehrereParameterKopierenView}.
 *
 * @author BitCtrl Systems GmbH, Albrecht Uhlmann
 */
public abstract class AbstractParameterKopierenView extends AbstractParamPluginView {

	/**
	 * Erstellt eine secondary Id aus dem Parameter. Sie reiht die Pids von Objekt,
	 * Typ und Atg aneinander, getrennt durch Bindestrich, dann noch die
	 * Simulationsvariante.
	 *
	 * @param parameters
	 *            der Parameter, der später auch über {@link #setInput(Parameter[])}
	 *            übergeben wird
	 * @return die secondary Id
	 */
	public static String buildSecondaryId(final Parameter[] parameters) {
		int loop;
		final StringBuilder b = new StringBuilder();
		for (loop = 0; loop < parameters.length; ++loop) {
			final Parameter parameter = parameters[loop];
			if (loop > 0) {
				b.append(',');
			}
			b.append(parameter.getObjekt().getPid());
			b.append('-');
			b.append(parameter.getTyp().getPid());
			b.append('-');
			b.append(parameter.getAtg().getPid());
			b.append('-');
			b.append(parameter.getSim());
		}
		return b.toString();
	}

	/**
	 * Erstellt einen Viewer, der im View zur Anzeige gebracht wird.
	 *
	 * @param parent
	 *            Eltern-Composite
	 * @return der Viewer
	 */
	protected abstract AbstractParameterKopierenViewer createParameterKopierenViewer(final Composite parent);

	@Override
	public void createPartControl(final Composite parent) {
		setPanel(parent);
		setContentViewer(createParameterKopierenViewer(parent));
		getContentViewer().setContentProvider(getContentProvider());
		// input wissen wir hier noch nicht

		setControlsState(RahmenwerkService.getService().getObjektFactory().isVerbunden());
		RahmenwerkService.getService().getObjektFactory().addPropertyChangeListener(DavProvider.PROP_VERBUNDEN, this);
		getContentProvider().addPropertyChangeListener(AbstractParamPluginContentProvider.PROP_KURZINFO, this);

		getSite().setSelectionProvider(getContentViewer());

		// Kontextmenü ist momentan nicht vorgesehen
	}

	@Override
	protected AbstractParamPluginLabelProvider getLabelProvider() {
		AbstractParamPluginLabelProvider provider = null;
		if ((getContentViewer() == null) || (getContentViewer().getLabelProvider() == null)
				|| !(getContentViewer().getLabelProvider() instanceof AbstractParamPluginLabelProvider)) {
			provider = new StandardParamPluginLabelProvider();
		} else {
			provider = (AbstractParamPluginLabelProvider) getContentViewer().getLabelProvider();
		}
		return provider;
	}

	@Override
	public void verbindungAufgebaut() {
		setControlsState(true);
		if ((getPanel() != null) && (getPanel().getDisplay() != null) && !getPanel().getDisplay().isDisposed()) {
			getPanel().getDisplay().asyncExec(new Runnable() {

				@Override
				public void run() {
					if ((getContentViewer() != null) && (getContentViewer()
							.getContentProvider() instanceof AbstractParamPluginContentProvider)) {
						final Display display = PlatformUI.getWorkbench().getDisplay();
						final Shell shell = display.getActiveShell();
						Cursor waitCursor = null;
						try {
							if (null != shell) {
								waitCursor = new Cursor(Display.getDefault(), SWT.CURSOR_WAIT);
								shell.setCursor(waitCursor);
							}
							((AbstractParamPluginContentProvider) getContentViewer().getContentProvider())
									.reloadCurrentInput(true);
						} catch (final ParameterClientException e) {
							ParamPlugin.getDefault().getLogger().warning(e.getLocalizedMessage(), e);
						} finally {
							if (null != shell) {
								shell.setCursor(null);
							}
							if (null != waitCursor) {
								waitCursor.dispose();
								waitCursor = null;
							}
						}
					}
				}
			});
		}
	}

	/**
	 * Setzt das Input-Objekt für den in uns enthaltenen
	 * {@link de.bsvrz.buv.plugin.param.viewer.ParameterKopierenViewer}.
	 *
	 * @param parameters
	 *            das Input-Objekt.
	 */
	public void setInput(final Parameter[] parameters) {
		if (null == getContentViewer()) {
			throw new IllegalStateException(
					"ContentViewer muss gesetzt sein, " + "bevor Input zugewiesen werden kann!");
		}
		if ((getPanel() != null) && !getPanel().isDisposed()) {
			getPanel().getDisplay().asyncExec(new Runnable() {

				@Override
				public void run() {
					getContentViewer().setInput(parameters);
				}
			});
		}
	}

}
