/*
 * Rahmenwerk-Plug-in "Parametrierung"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */

package de.bsvrz.buv.plugin.param.views;

import org.eclipse.jface.viewers.ViewerComparator;
import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.dialogs.FilteredTree;
import org.eclipse.ui.forms.widgets.Form;
import org.eclipse.ui.forms.widgets.FormToolkit;

import de.bsvrz.buv.plugin.param.ParametertypenCache;
import de.bsvrz.buv.plugin.param.internal.ParamPlugin;
import de.bsvrz.buv.plugin.param.internal.RahmenwerkService;
import de.bsvrz.buv.plugin.param.provider.AbstractParamPluginContentProvider;
import de.bsvrz.buv.plugin.param.provider.AbstractParamPluginLabelProvider;
import de.bsvrz.buv.plugin.param.provider.ParametertypenContentProvider;
import de.bsvrz.buv.plugin.param.provider.StandardParamPluginLabelProvider;
import de.bsvrz.buv.rw.bitctrl.eclipse.Messages;
import de.bsvrz.buv.rw.bitctrl.eclipse.PatternOrSubstringFilter;
import de.bsvrz.sys.funclib.bitctrl.daf.DavProvider;

/**
 * Ansicht zur Darstellung der Parametertypen (=Attributgruppen), linke obere
 * Ansicht in Abb 5-41.
 *
 * @author BitCtrl Systems GmbH, Albrecht Uhlmann
 *
 */
public class ParametertypenView extends AbstractParamPluginView {

	/** die Kontext-Hilfe-ID. */
	private static final String HILFE_ID = ParamPlugin.PLUGIN_ID + "." + ParametertypenView.class.getSimpleName();

	/** die ID der Ansicht. */
	public static final String ID = ParametertypenView.class.getName();

	/** alphabetischer Sortierer für die dargestellten Elemente. */
	private final ViewerComparator alphabetischerSortierer = new ViewerComparator();

	private FormToolkit toolkit;

	private Form form;

	private FilteredTree filteredTree;

	/** {@inheritDoc} */
	@Override
	public void createPartControl(final Composite parent) {
		toolkit = new FormToolkit(parent.getDisplay());
		form = toolkit.createForm(parent);
		toolkit.decorateFormHeading(form);
		final Label headLabel = toolkit.createLabel(form.getHead(), Messages.SystemObjectAuswahl_Description, SWT.WRAP);
		headLabel.setBackground(null);
		form.setHeadClient(headLabel);
		final Composite formBody = form.getBody();
		formBody.setLayout(new GridLayout());

		setPanel(formBody);

		filteredTree = new FilteredTree(formBody, SWT.BORDER | SWT.SINGLE, new PatternOrSubstringFilter(), true);
		toolkit.adapt(filteredTree);
		setContentViewer(filteredTree.getViewer());

		// columnViewer.addDoubleClickListener(new SimDoubleClickListener());

		filteredTree.getViewer().setContentProvider(getContentProvider());
		filteredTree.getViewer().setLabelProvider(getLabelProvider());
		filteredTree.getViewer().setComparator(alphabetischerSortierer);
		setKonfigurationsBereicheAnzeigen(true);
		setSystemObjektTypenAnzeigen(true);
		if (RahmenwerkService.getService().getObjektFactory().isVerbunden()) {
			filteredTree.getViewer().setInput(ParametertypenCache.getInstanz());
		}
		setControlsState(RahmenwerkService.getService().getObjektFactory().isVerbunden());
		RahmenwerkService.getService().getObjektFactory().addPropertyChangeListener(DavProvider.PROP_VERBUNDEN, this);

		getSite().setSelectionProvider(filteredTree.getViewer());

		createContextMenu();

		// Online-Hilfe hinzufügen
		PlatformUI.getWorkbench().getHelpSystem().setHelp(filteredTree.getViewer().getControl(),
				ParametertypenView.HILFE_ID);
	}

	/**
	 * ermittelt, ob die Konfigurationsbereiche der dargestellten Objekte angezeigt
	 * werden sollen.
	 *
	 * @return <code>true</code>, wenn die Konfigurationsbereiche angezeigt werden
	 *         sollen
	 */
	boolean isKonfigurationsBereicheAnzeigen() {
		boolean result = false;
		final AbstractParamPluginLabelProvider provider = getLabelProvider();
		if (provider != null) {
			result = provider.isKonfigurationsBereicheAnzeigen();
		}
		return result;
	}

	/**
	 * markiert, ob die Konfigurationsbereiche der dargestellten Objekte angezeigt
	 * werden sollen.
	 *
	 * @param status
	 *            <code>true</code>, wenn die Konfigurationsbereiche angezeigt
	 *            werden sollen
	 */
	public void setKonfigurationsBereicheAnzeigen(final boolean status) {
		final AbstractParamPluginLabelProvider provider = getLabelProvider();
		if ((provider != null) && (status != provider.isKonfigurationsBereicheAnzeigen())) {
			provider.setKonfigurationsBereicheAnzeigen(status);
			getContentViewer().refresh();
		}
	}

	/**
	 * ermittelt, ob die Systemobjekttypen der dargestellten Parametertypen
	 * (=Attributgruppen) angezeigt werden sollen.
	 *
	 * @return <code>true</code>, wenn die Systemobjekttypen angezeigt werden sollen
	 */
	boolean isSystemObjektTypenAnzeigen() {
		boolean result = false;
		final AbstractParamPluginLabelProvider provider = getLabelProvider();
		if (provider != null) {
			result = provider.isSystemObjektTypenAnzeigen();
		}
		return result;
	}

	/**
	 * bestimmt, ob die Systemobjekttypen der dargestellten Parametertypen
	 * (=Attributgruppen) angezeigt werden sollen.
	 *
	 * @param status
	 *            <code>true</code>, wenn die Systemobjekttypen angezeigt werden
	 *            sollen
	 */
	public void setSystemObjektTypenAnzeigen(final boolean status) {
		final AbstractParamPluginLabelProvider provider = getLabelProvider();
		if ((provider != null) && (status != provider.isSystemObjektTypenAnzeigen())) {
			provider.setSystemObjektTypenAnzeigen(status);
			getContentViewer().refresh();
		}
	}

	@Override
	protected AbstractParamPluginContentProvider getContentProvider() {
		AbstractParamPluginContentProvider provider = null;
		if ((getContentViewer() == null) || (getContentViewer().getContentProvider() == null)) {
			provider = new ParametertypenContentProvider(this);
		} else {
			provider = (AbstractParamPluginContentProvider) getContentViewer().getContentProvider();
		}
		return provider;
	}

	@Override
	protected AbstractParamPluginLabelProvider getLabelProvider() {
		AbstractParamPluginLabelProvider provider = null;
		if ((getContentViewer() == null) || (getContentViewer().getLabelProvider() == null)
				|| !(getContentViewer().getLabelProvider() instanceof AbstractParamPluginLabelProvider)) {
			provider = new StandardParamPluginLabelProvider();
		} else {
			provider = (AbstractParamPluginLabelProvider) getContentViewer().getLabelProvider();
		}
		return provider;
	}

}
