/*
 * Rahmenwerk-Plugin fuer den Zugriff auf die Umfassende Datenanalyse (UDA)
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */

package de.bsvrz.buv.plugin.uda.builder;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Map;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IMarker;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IResourceDelta;
import org.eclipse.core.resources.IncrementalProjectBuilder;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;

import de.bsvrz.buv.plugin.uda.UdaToolsPlugin;
import de.bsvrz.dav.daf.main.MissingParameterException;
import de.bsvrz.ibv.uda.client.LokalesSkript;
import de.bsvrz.ibv.uda.client.SkriptLaufDaten;
import de.bsvrz.ibv.uda.interpreter.UdaInterpreter;
import de.bsvrz.ibv.uda.interpreter.UdaInterpreterException;
import de.bsvrz.ibv.uda.interpreter.UdaParserFehler;

/**
 * Schnittstelle zum UDA-Interpreter.
 *
 * @author BitCtrl Systems GmbH, Uwe Peuker
 *
 *
 */
public class UdaBuilder extends IncrementalProjectBuilder {

	/**
	 * ID.
	 */
	public static final String BUILDER_ID = UdaBuilder.class.getName();

	/**
	 * Standardkonstruktor.
	 */
	public UdaBuilder() {
		//
	}

	@Override
	protected IProject[] build(final int kind, @SuppressWarnings("rawtypes") final Map args,
			final IProgressMonitor monitor) throws CoreException {

		for (final IResourceDelta delta : getDelta(getProject()).getAffectedChildren()) {
			final Collection<IFile> files = getFiles(delta, "uda");
			for (final IFile file : files) {
				compileSkript(file, monitor);
			}
		}
		return null;
	}

	/**
	 * Ruft den UDA-Interpreter fuer ein Skript auf.
	 *
	 * @param udaFile
	 *            das UDA-Skript.
	 * @param monitor
	 *            Der Zustandsmonitor.
	 * @throws CoreException
	 *             wenn ein interner Fehler aufgetreten ist.
	 */
	@SuppressWarnings("unused")
	public void compileSkript(final IFile udaFile, final IProgressMonitor monitor) throws CoreException {

		Collection<UdaParserFehler> fehlerListe = null;
		final IMarker[] problems = udaFile.findMarkers(IMarker.PROBLEM, true, 1);
		for (final IMarker marker : problems) {
			marker.delete();
		}

		try {
			// TODO lokalen SkriptLauf
			final SkriptLaufDaten lauf = new SkriptLaufDaten(null, "CompilerLauf",
					new LokalesSkript(udaFile.getLocation().toFile().getAbsolutePath()));
			new UdaInterpreter(UdaToolsPlugin.getDefault().getConnector().getVerbindung(), lauf);
		} catch (final UdaInterpreterException e) {
			fehlerListe = e.getFehlerListe();
		} catch (final MissingParameterException e) {
			e.printStackTrace();
		}

		if (fehlerListe != null) {
			for (final UdaParserFehler fehler : fehlerListe) {
				final IMarker marker = udaFile.createMarker(IMarker.PROBLEM);
				marker.setAttribute(IMarker.SEVERITY, IMarker.SEVERITY_ERROR);
				marker.setAttribute(IMarker.MESSAGE, fehler.getMessage());
				marker.setAttribute(IMarker.LOCATION, "Zeile " + fehler.getLine());
				marker.setAttribute(IMarker.LINE_NUMBER, fehler.getLine());
			}
		}
	}

	/**
	 * Erfragt die Menge der UDA-Skripte, die interpretiert werden sollen.
	 *
	 * @param item
	 *            alle Resourcen die veraendert wurden.
	 * @param string
	 *            Fileextension.
	 * @return die Menge der UDA-Skripte, die interpretiert werden sollen.
	 */
	private Collection<IFile> getFiles(final IResourceDelta item, final String string) {
		final Collection<IFile> result = new ArrayList<>();
		final IResource resource = item.getResource();
		if (resource instanceof IFile) {
			final IFile file = (IFile) resource;
			if (string.equals(file.getFileExtension())) {
				result.add(file);
			}
		} else {
			final IResourceDelta[] delta = item.getAffectedChildren();
			for (final IResourceDelta deltaItem : delta) {
				result.addAll(getFiles(deltaItem, string));
			}
		}
		return result;
	}
}
