/*
 * Rahmenwerk-Plugin fuer den Zugriff auf die Umfassende Datenanalyse (UDA)
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */

package de.bsvrz.buv.plugin.uda.editor;

import org.eclipse.jface.action.IAction;
import org.eclipse.jface.text.TextAttribute;
import org.eclipse.jface.text.rules.IToken;
import org.eclipse.jface.text.rules.Token;
import org.eclipse.jface.text.source.ISourceViewer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.layout.RowLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.texteditor.AbstractDecoratedTextEditor;
import org.eclipse.ui.texteditor.ITextEditorActionDefinitionIds;
import org.eclipse.ui.texteditor.TextOperationAction;

import de.bsvrz.buv.plugin.uda.UdaToolsPlugin;

/**
 * Editor für ein UDA-File.
 *
 * @author BitCtrl Systems GmbH, Uwe Peuker
 *
 *
 */
public class UdaEditor extends AbstractDecoratedTextEditor {

	/**
	 * das Anzeigemuster für Schlüsselworte.
	 */
	public static final String EDITOR_ID = UdaEditor.class.getName();

	/**
	 * das Anzeigemuster für Schlüsselworte.
	 */
	public static final IToken KEYWORD_TOKEN = new Token(
			new TextAttribute(Display.getDefault().getSystemColor(SWT.COLOR_DARK_MAGENTA), null, SWT.BOLD));

	/**
	 * das Anzeigemuster für Zeichenketten.
	 */
	public static final IToken STRING_TOKEN = new Token(
			new TextAttribute(Display.getDefault().getSystemColor(SWT.COLOR_BLUE), null, SWT.ITALIC));

	/**
	 * das Anzeigemuster für Kommentare.
	 */
	public static final IToken COMMENT_TOKEN = new Token(
			new TextAttribute(Display.getDefault().getSystemColor(SWT.COLOR_GRAY), null, SWT.ITALIC));

	/**
	 * die Context-Hilfe-ID.
	 */
	private static final String HILFE_ID = UdaToolsPlugin.PLUGIN_ID + "." + UdaEditor.class.getSimpleName();

	/**
	 * Basis-Composite.
	 */
	private Composite basisComposite;

	/**
	 * Style-Provider.
	 */
	private final UdaStyleProvider styleProvider = new UdaStyleProvider();

	/**
	 * Konstruktor.
	 */
	public UdaEditor() {
		setDocumentProvider(new UdaDocumentProvider());
		setSourceViewerConfiguration(new UdaEditorSourceViewerConfiguration(styleProvider));
	}

	/**
	 * Wird beim Druecken der Schaltflaeche <code>Hilfe</code> aufgerufen.
	 */
	protected void aktionHilfe() {
		PlatformUI.getWorkbench().getHelpSystem().displayHelp(UdaEditor.HILFE_ID);
	}

	/**
	 * Wird beim Druecken der Schaltflaeche <code>Ok</code> aufgerufen.
	 */
	protected void aktionOk() {
		// mache nichts
	}

	/**
	 * Wird beim Druecken der Schaltflaeche <code>Schliessen</code> aufgerufen.
	 */
	protected void aktionSchliessen() {
		close(true);
	}

	@Override
	protected void createActions() {
		super.createActions();

		final IAction action = new TextOperationAction(UdaToolsPlugin.getDefault().getResourceBundle(),
				"ContentAssistProposal.", //$NON-NLS-1$
				this, ISourceViewer.CONTENTASSIST_PROPOSALS);
		action.setActionDefinitionId(ITextEditorActionDefinitionIds.CONTENT_ASSIST_PROPOSALS);
		setAction("ContentAssist.", action); //$NON-NLS-1$
		markAsStateDependentAction("ContentAssist.", true); //$NON-NLS-1$
	}

	@Override
	public void createPartControl(final Composite parent) {

		final Composite innerComposite = new Composite(parent, SWT.NONE);
		innerComposite.setLayout(new FillLayout(SWT.HORIZONTAL));

		final GridLayout gridLayout = new GridLayout();
		gridLayout.numColumns = 1;
		parent.setLayout(gridLayout);

		final GridData gridData = new GridData();
		gridData.grabExcessHorizontalSpace = true;
		gridData.verticalAlignment = GridData.FILL;
		gridData.grabExcessVerticalSpace = true;
		gridData.horizontalAlignment = GridData.FILL;
		innerComposite.setLayoutData(gridData);

		super.createPartControl(innerComposite);

		createStandardSchaltflaechen(parent);

		basisComposite = parent;

		PlatformUI.getWorkbench().getHelpSystem().setHelp(parent, UdaEditor.HILFE_ID);
	}

	/**
	 * Initialisiert die drei Schaltflaechen <code>OK</code>,
	 * <code>Schliessen</code> und <code>Hilfe</code>.
	 *
	 * @param parent
	 *            uebergeordnetes Composite.
	 */
	protected void createStandardSchaltflaechen(final Composite parent) {
		final RowLayout rowLayout = new RowLayout();
		rowLayout.pack = false;
		rowLayout.justify = false;
		rowLayout.fill = false;
		rowLayout.wrap = false;
		final GridData gridData = new GridData();
		gridData.grabExcessHorizontalSpace = false;
		gridData.verticalAlignment = GridData.CENTER;
		gridData.horizontalAlignment = GridData.END;
		final Composite composite = new Composite(parent, SWT.NONE);
		composite.setLayoutData(gridData);
		composite.setLayout(rowLayout);

		final Button buttonOk = new Button(composite, SWT.NONE);
		buttonOk.setText("OK");
		buttonOk.addMouseListener(new org.eclipse.swt.events.MouseAdapter() {
			@Override
			public void mouseDown(final org.eclipse.swt.events.MouseEvent e) {
				aktionOk();
			}
		});

		final Button buttonSchliessen = new Button(composite, SWT.NONE);
		buttonSchliessen.setText("Schließen");
		buttonSchliessen.addMouseListener(new org.eclipse.swt.events.MouseAdapter() {
			@Override
			public void mouseDown(final org.eclipse.swt.events.MouseEvent e) {
				aktionSchliessen();
			}
		});

		final Button buttonHilfe = new Button(composite, SWT.NONE);
		buttonHilfe.setText("Hilfe");
		buttonHilfe.addMouseListener(new org.eclipse.swt.events.MouseAdapter() {
			@Override
			public void mouseDown(final org.eclipse.swt.events.MouseEvent e) {
				aktionHilfe();
			}
		});
	}

	@Override
	public void setFocus() {
		if (basisComposite != null) {
			basisComposite.setFocus();
		}
	}

}
