/*
 * Rahmenwerk-Plugin fuer den Zugriff auf die Umfassende Datenanalyse (UDA)
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */

package de.bsvrz.buv.plugin.uda.navigator;

import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.action.Separator;
import org.eclipse.jface.viewers.DoubleClickEvent;
import org.eclipse.jface.viewers.IDoubleClickListener;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.jface.viewers.ViewerComparator;
import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.ui.IWorkbenchActionConstants;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.part.ViewPart;

import de.bsvrz.buv.plugin.uda.UdaToolsPlugin;
import de.bsvrz.buv.plugin.uda.handler.EditSkriptTextHandler;
import de.bsvrz.buv.plugin.uda.handler.SkriptLaufStartenHandler;
import de.bsvrz.buv.plugin.uda.handler.SkriptLaufViewHandler;
import de.bsvrz.dav.daf.main.ClientDavInterface;
import de.bsvrz.dav.daf.main.MissingParameterException;
import de.bsvrz.ibv.uda.client.ClientModul;
import de.bsvrz.ibv.uda.client.ClientSkript;
import de.bsvrz.ibv.uda.client.ClientSkriptLauf;
import de.bsvrz.ibv.uda.client.ModulListener;
import de.bsvrz.ibv.uda.client.SkriptLaufListener;
import de.bsvrz.ibv.uda.client.SkriptListener;
import de.bsvrz.ibv.uda.client.UdaClient;
import de.bsvrz.ibv.uda.client.UdaConnectionListener;

/**
 * Navigatoransicht zur Anzeige von UDA-Modulen und deren Skripten und
 * Skriptläufen.
 *
 * @author BitCtrl Systems GmbH, Uwe Peuker
 *
 *
 */
public class UdaModulNavigator extends ViewPart
		implements UdaConnectionListener, ModulListener, SkriptListener, SkriptLaufListener, IDoubleClickListener {

	/** die ID der Ansicht. */
	public static final String ID = UdaModulNavigator.class.getName();

	/**
	 * die Context-Hilfe-ID.
	 */
	private static final String HILFE_ID = UdaToolsPlugin.PLUGIN_ID + "." + UdaModulNavigator.class.getSimpleName();

	/** der Viewer zur Anzeige der Elemente. */
	private TreeViewer navigatorTree;

	/** der Client zum Austausch mit der Umfassenden Datenanalyse. */
	private UdaClient connector;

	/**
	 * meldet die Ansicht als Listener für Änderungen innerhalb der Umfassenden
	 * Datenanalyse ab.
	 *
	 * @param client
	 *            der verwendete Client.
	 */
	private void abmelden(final ClientModul client) {
		client.removeClientListener(this);
		client.removeSkriptListener(this);
		client.removeLaufListener(this);
	}

	@Override
	public void connected() {
		initialisiereDaten();
		refreshTree();
	}

	@Override
	public void connectionClosed(final ClientDavInterface connection) {
		initialisiereDaten();
		refreshTree();
	}

	/**
	 * Erzeugt ein Kontextmenü für dieses Ansicht.
	 */
	private void createContextMenu() {
		final MenuManager menuManager = new MenuManager();
		menuManager.add(new Separator(IWorkbenchActionConstants.MB_ADDITIONS));
		navigatorTree.getControl().setMenu(menuManager.createContextMenu(navigatorTree.getControl()));
		getSite().registerContextMenu(UdaModulNavigator.ID + ".contextMenu", menuManager, navigatorTree);
	}

	@Override
	public void createPartControl(final Composite parent1) {
		final GridLayout gridLayout = new GridLayout();
		gridLayout.numColumns = 1;
		final Composite basisComposite = new Composite(parent1, SWT.NONE);
		basisComposite.setLayout(gridLayout);

		try {
			connector = UdaToolsPlugin.getDefault().getConnector();
		} catch (final MissingParameterException e) {
			e.printStackTrace();
		}
		connector.addConnectionListener(this);
		initialisiereDaten();

		navigatorTree = new TreeViewer(basisComposite, SWT.NONE);

		final GridData gridData1 = new GridData();
		gridData1.grabExcessHorizontalSpace = true;
		gridData1.verticalAlignment = GridData.FILL;
		gridData1.grabExcessVerticalSpace = true;
		gridData1.horizontalAlignment = GridData.FILL;
		navigatorTree.getTree().setLayoutData(gridData1);

		navigatorTree.setContentProvider(new UdaModulNavigatorContentProvider(this));
		navigatorTree.setLabelProvider(new UdaModulNavigatorLabelProvider());
		navigatorTree.setComparator(new ViewerComparator());
		navigatorTree.addDoubleClickListener(this);

		navigatorTree.setInput(this);

		createContextMenu();
		getSite().setSelectionProvider(navigatorTree);

		PlatformUI.getWorkbench().getHelpSystem().setHelp(basisComposite, UdaModulNavigator.HILFE_ID);
	}

	@Override
	public void dispose() {
		for (final ClientModul client : connector.getModule()) {
			abmelden(client);
		}
		connector.removeConnectionListener(this);
		super.dispose();
	}

	@Override
	public void doubleClick(final DoubleClickEvent event) {
		final IStructuredSelection selektion = (IStructuredSelection) event.getSelection();
		if (selektion != null) {
			final Object selektiertesObjekt = selektion.getFirstElement();
			if (selektiertesObjekt == null) {
				return;
			}

			if (selektiertesObjekt instanceof ClientSkript) {
				new EditSkriptTextHandler().edit((ClientSkript) selektiertesObjekt);
			} else if (selektiertesObjekt instanceof ClientSkriptLauf) {
				new SkriptLaufViewHandler().openView((ClientSkriptLauf) selektiertesObjekt);
				new SkriptLaufStartenHandler().startSkriptLauf((ClientSkriptLauf) selektiertesObjekt);
			}
		}
	}

	@Override
	public void ergebnisAktualisiert(final ClientSkriptLauf lauf) {
		// die Ergebnisse von Skriptläufen werden hier nicht visualisiert.
	}

	/**
	 * liefert den verwendeten UDA-Client.
	 *
	 * @return connector den Client.
	 */
	public UdaClient getConnector() {
		return connector;
	}

	/**
	 * initialisiert die Verbindung zur Umfassenden Datenanalyse.
	 */
	private void initialisiereDaten() {
		if (connector.isConnected()) {
			for (final ClientModul modul : connector.getModule()) {
				modul.addClientListener(this);
				modul.addSkriptListener(this);
				modul.addLaufListener(this);
			}
			Display.getDefault().asyncExec(new Runnable() {
				@Override
				public void run() {
					refreshTree();
				}
			});
		}
	}

	@Override
	public void meldungsListeAktualisiert(final ClientSkriptLauf skript) {
		// die Ausgaben von Skriptläufen werden in dieser Ansicht nicht
		// visualisiert
	}

	@Override
	public void neuerModulStatus(final ClientModul modul, final boolean valid) {
		refreshTree();
	}

	/**
	 * Aktualisiert den Inhalt der Ansicht.
	 */
	void refreshTree() {
		Display.getDefault().asyncExec(new Runnable() {
			@Override
			public void run() {
				navigatorTree.refresh(true);
			}
		});
	}

	@Override
	public void skriptAktualisiert(final ClientSkript skript) {
		refreshTree();
	}

	@Override
	public void skriptAktualisiert(final ClientSkriptLauf skript) {
		refreshTree();
	}

	@Override
	public void skriptAngelegt(final ClientModul client, final ClientSkript skript) {
		refreshTree();
	}

	@Override
	public void skriptEntfernt(final ClientModul modul, final ClientSkript skript) {
		refreshTree();
	}

	@Override
	public void skriptLaufAngelegt(final ClientModul modul, final ClientSkriptLauf lauf) {
		refreshTree();
	}

	@Override
	public void skriptLaufEntfernt(final ClientModul modul, final ClientSkriptLauf lauf) {
		refreshTree();
	}

	@Override
	public void skriptZustandAktualisiert(final ClientSkriptLauf skript) {
		refreshTree();
	}

	@Override
	public void setFocus() {
		if ((navigatorTree != null) && (!navigatorTree.getControl().isDisposed())) {
			navigatorTree.getControl().setFocus();
		}
	}
}
