/*
 * Rahmenwerk-Plugin fuer den Zugriff auf die Umfassende Datenanalyse (UDA)
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */

package de.bsvrz.buv.plugin.uda.views;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.jface.viewers.ListViewer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.part.ViewPart;

import de.bsvrz.buv.plugin.uda.UdaToolsPlugin;
import de.bsvrz.dav.daf.main.ClientDavInterface;
import de.bsvrz.dav.daf.main.MissingParameterException;
import de.bsvrz.ibv.uda.client.ClientModul;
import de.bsvrz.ibv.uda.client.ClientSkriptLauf;
import de.bsvrz.ibv.uda.client.SkriptLaufListener;
import de.bsvrz.ibv.uda.client.UdaConnectionListener;

/**
 * Ansicht zur Anzeige der Ausgaben eines Skriptlaufs, die über den
 * Datenverteiler publiziert werden.
 *
 * @author BitCtrl Systems GmbH, Uwe Peuker
 *
 *
 */
public class LaufLogSicht extends ViewPart implements UdaConnectionListener, SkriptLaufListener {

	/** die ID der Ansicht. */
	public static final String ID = LaufLogSicht.class.getName();

	/**
	 * die Context-Hilfe-ID.
	 */
	private static final String HILFE_ID = UdaToolsPlugin.PLUGIN_ID + "." + LaufLogSicht.class.getSimpleName();

	/** die PID des verwendeten UDA-Moduls. */
	private String modulPid;

	/** der Skriptlauf, dessen Ausgaben angezeigt werden sollen. */
	private ClientSkriptLauf loggedLauf;

	/** der Viewer zur Anzeige der Daten. */
	private ListViewer viewer;

	/** die Liste der empfangenen Ausgabetexte. */
	private final List<String> meldungen = new ArrayList<>();

	/**
	 * Konstruktor.
	 */
	public LaufLogSicht() {
		try {
			UdaToolsPlugin.getDefault().getConnector().addConnectionListener(this);
		} catch (final MissingParameterException e) {
			e.printStackTrace();
		}
	}

	@Override
	public void connected() {
		try {
			if ((loggedLauf != null) && (modulPid != null)) {
				for (final ClientModul modul : UdaToolsPlugin.getDefault().getConnector().getModule()) {
					if (modul.getModulObjekt().getPid().equals(modulPid)) {
						for (final ClientSkriptLauf skriptLauf : modul.getSkriptLaeufe()) {
							if (skriptLauf.getDavObjekt().getPid().equals(loggedLauf.getDavObjekt().getPid())) {
								setLauf(skriptLauf);
								break;
							}
						}
						break;
					}
				}
			}
		} catch (final MissingParameterException e) {
			e.printStackTrace();
		}
	}

	@Override
	public void connectionClosed(final ClientDavInterface connection) {
		Display.getDefault().asyncExec(new Runnable() {
			@Override
			public void run() {
				viewer.getList().removeAll();
			}
		});
	}

	@Override
	public void createPartControl(final Composite parent) {
		final GridLayout gridLayout = new GridLayout();
		gridLayout.numColumns = 1;
		final Composite composite = new Composite(parent, SWT.NONE);
		composite.setLayout(gridLayout);

		viewer = new ListViewer(composite, SWT.BORDER | SWT.V_SCROLL | SWT.H_SCROLL);

		final GridData gridData1 = new GridData();
		gridData1.grabExcessHorizontalSpace = true;
		gridData1.verticalAlignment = GridData.FILL;
		gridData1.grabExcessVerticalSpace = true;
		gridData1.horizontalAlignment = GridData.FILL;
		viewer.getList().setLayoutData(gridData1);

		PlatformUI.getWorkbench().getHelpSystem().setHelp(parent, LaufLogSicht.HILFE_ID);
	}

	@Override
	public void dispose() {
		if (loggedLauf != null) {
			loggedLauf.getModul().removeLaufListener(this);
		}
		super.dispose();
	}

	@Override
	public void ergebnisAktualisiert(final ClientSkriptLauf lauf) {
		// das Eregnis eines Skriptlaufs wird hier nicht visualisiert.
	}

	@Override
	public void meldungsListeAktualisiert(final ClientSkriptLauf skript) {
		if (this.loggedLauf.equals(skript)) {
			meldungen.clear();
			for (final String line : loggedLauf.getMeldungen()) {
				meldungen.add(0, line);
			}
			Display.getDefault().asyncExec(new Runnable() {

				@Override
				public void run() {
					updateViewer();
				}
			});
		}
	}

	/**
	 * Setzt den Skriptlauf.
	 *
	 * @param lauf
	 *            den Skriptlauf.
	 */
	public void setLauf(final ClientSkriptLauf lauf) {
		if (lauf != null) {
			setPartName(lauf.getName());
			lauf.getModul().addLaufListener(this);
			modulPid = lauf.getModul().getModulObjekt().getPid();
		} else {
			modulPid = null;
		}
		this.loggedLauf = lauf;
	}

	@Override
	public void skriptAktualisiert(final ClientSkriptLauf skript) {
		// der Skriptzustand wird in dieser Zustand nicht aktualisiert
	}

	@Override
	public void skriptZustandAktualisiert(final ClientSkriptLauf skript) {
		// der Skriptlaufzustand wird in dieser Ansicht nicht aktualisiert.
	}

	/**
	 * aktualisiert den Inhalt der Ansicht.
	 */
	void updateViewer() {
		viewer.getList().removeAll();
		for (final String line : meldungen) {
			viewer.getList().add(line);
		}
	}

	@Override
	public void setFocus() {
		if ((viewer != null) && (!viewer.getControl().isDisposed())) {
			viewer.getControl().setFocus();
		}
	}

}
