/*
 * Rahmenwerk-Plugin fuer den Zugriff auf die Umfassende Datenanalyse (UDA)
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */

package de.bsvrz.buv.plugin.uda.views;

import java.util.ArrayList;
import java.util.Collection;

import org.eclipse.jface.viewers.IStructuredContentProvider;
import org.eclipse.jface.viewers.ITableColorProvider;
import org.eclipse.jface.viewers.ITableLabelProvider;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.widgets.Display;

import com.bitctrl.Constants;

import de.bsvrz.ibv.uda.client.ClientModul;
import de.bsvrz.ibv.uda.client.ClientSkriptLauf;
import de.bsvrz.ibv.uda.client.UdaClient;
import de.bsvrz.ibv.uda.uda.data.Modul;
import de.bsvrz.ibv.uda.uda.data.SkriptLaufStatus;
import de.bsvrz.ibv.uda.uda.data.SkriptZustand;

/**
 * Ansicht mit einer Übersicht aller Skriptläufe.
 *
 * @author BitCtrl Systems GmbH, Uwe Peuker
 */
public class SkriptLaufOverViewProvider extends LabelProvider
		implements IStructuredContentProvider, ITableLabelProvider, ITableColorProvider {

	/** Farbe WEISS. */
	private static final Color WHITE = Display.getDefault().getSystemColor(SWT.COLOR_WHITE);

	/** Farbe ROT. */
	private static final Color RED = Display.getDefault().getSystemColor(SWT.COLOR_RED);

	/** Farbe BLAU. */
	private static final Color BLUE = Display.getDefault().getSystemColor(SWT.COLOR_BLUE);

	/** Farbe GRAU. */
	private static final Color GRAY = Display.getDefault().getSystemColor(SWT.COLOR_GRAY);

	/** Farbe Schwarz. */
	private static final Color BLACK = Display.getDefault().getSystemColor(SWT.COLOR_BLACK);

	/** Farbe Dunkel-Grün. */
	private static final Color DARK_GREEN = Display.getDefault().getSystemColor(SWT.COLOR_DARK_GREEN);

	/** die Spaltenüberschriften der Übersichtstabelle. */
	private static final String[] COLUMN_NAMES = { "Modul", "Skriptlauf", "Skript", "Status", "Zeit", "Ergebnis" };

	@Override
	public void dispose() {
		//
	}

	@Override
	public Color getBackground(final Object element, final int columnIndex) {
		return SkriptLaufOverViewProvider.WHITE;
	}

	@Override
	public Image getColumnImage(final Object element, final int columnIndex) {
		return null;
	}

	/**
	 * liefert die Namen der anzuzeigenden Spalten.
	 *
	 * @return die Liste der Namen
	 */
	String[] getColumnNames() {
		return SkriptLaufOverViewProvider.COLUMN_NAMES;
	}

	@Override
	public String getColumnText(final Object element, final int columnIndex) {
		String result = null;
		if (element instanceof ClientSkriptLauf) {
			final ClientSkriptLauf lauf = (ClientSkriptLauf) element;
			switch (columnIndex) {
			case 0: // "Modul"
				if (lauf.getModul() != null) {
					result = lauf.getModul().getModulObjekt().getNameOrPidOrId();
				}
				break;
			case 1: // "Skriplauf"
				result = lauf.getName();
				break;
			case 2: // "Skript"
				if (lauf.getSkript() != null) {
					result = lauf.getSkript().getName();
				}
				break;
			case 3: // "Status"
				result = lauf.getStatus().toString();
				break;
			case 4: // "Zeit"
				if (lauf.getLaufErgebnis() != null) {
					result = lauf.getLaufErgebnis().getZeitString();
				}
				break;
			case 5: // "Ergebnis"
				if (lauf.getLaufErgebnis() != null) {
					result = lauf.getLaufErgebnis().getText(0);
				}
				break;
			default:
				result = Constants.EMPTY_STRING;
			}
		}
		if (result == null) {
			result = Constants.EMPTY_STRING;
		}
		return result;
	}

	@Override
	public Object[] getElements(final Object element) {
		final Collection<Object> elements = new ArrayList<>();
		if (element instanceof UdaClient) {
			for (final Modul modul : ((UdaClient) element).getModule()) {
				if (modul instanceof ClientModul) {
					elements.addAll(((ClientModul) modul).getSkriptLaeufe());
				}
			}
		}
		return elements.toArray();
	}

	@Override
	public Color getForeground(final Object element, final int columnIndex) {
		Color result = SkriptLaufOverViewProvider.BLACK;
		if (element instanceof ClientSkriptLauf) {
			final ClientSkriptLauf lauf = (ClientSkriptLauf) element;
			if ((lauf.getSkript() == null) || (!SkriptZustand.FREIGEGEBEN.equals(lauf.getSkript().getZustand()))) {
				result = SkriptLaufOverViewProvider.GRAY;
			} else if (SkriptLaufStatus.AKTIV.equals(lauf.getStatus())) {
				result = SkriptLaufOverViewProvider.DARK_GREEN;
			} else if ((lauf.getLaufErgebnis() != null) && (!lauf.getLaufErgebnis().isOk())) {
				result = SkriptLaufOverViewProvider.RED;
			} else {
				result = SkriptLaufOverViewProvider.BLUE;
			}
		}
		return result;
	}

	@Override
	public void inputChanged(final Viewer viewer, final Object oldInput, final Object newInput) {
		//
	}
}
