/*
 * Rahmenwerk-Plug-in "RDS-Detailanzeige"
 * Spezial-Plug-in VRZ3 - Baden-Württemberg
 *
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */

package de.bsvrz.buv.plugin.verkehrsmeldung.detail.internal;

import org.eclipse.core.runtime.Assert;

import de.bsvrz.buv.rw.basislib.Rahmenwerk;
import de.bsvrz.buv.rw.basislib.berechtigung.Berechtigungen;
import de.bsvrz.buv.rw.basislib.einstellungen.Einstellungen;
import de.bsvrz.sys.funclib.bitctrl.modell.ObjektFactory;
import de.bsvrz.sys.funclib.bitctrl.modell.SystemObjekt;
import de.bsvrz.sys.funclib.bitctrl.modell.tmtmcverkehrsmeldung.objekte.TmcLandesMeldeStelle;
import de.bsvrz.sys.funclib.commandLineArgs.ArgumentList;
import de.bsvrz.sys.funclib.debug.Debug;

/**
 * Interne Komponente zur Bereitstellung der für das Plug-in erforderlichen
 * Services.
 *
 * @author BitCtrl Systems GmbH, Uwe Peuker
 *
 *
 */
public class RahmenwerkService {

	private static final String LANDESMELDESTELLE_OBJEKT_PARAMETER = "-lms";

	private static RahmenwerkService service;

	/**
	 * liefert die globale Instanz der Komponente.
	 *
	 * @return die Komponente
	 */
	public static RahmenwerkService getService() {
		return RahmenwerkService.service;
	}

	private Rahmenwerk rahmenWerk;
	private Berechtigungen berechtigungen;

	private Einstellungen einstellungen;

	private ObjektFactory objektFactory;

	/**
	 * Flag, welches angibt, ob dieses Plug-in ausf&uuml;hrbar ist. Das Plug-in ist
	 * nicht ausf&uuml;hrbar, wenn keine g&uuml;ltige Landesmeldestelle als
	 * Aufrufparameter &uuml;bergeben wurde.
	 */
	private boolean ausfuehrbar;

	/**
	 * Die Landesmeldestelle.
	 */
	private TmcLandesMeldeStelle landesMeldeStelle;

	private String defaultLmsPid;

	/** aktiviert die Komponente. */
	protected void activate() {
		RahmenwerkService.service = this;

		// Defaultnetz der Applikation vorbereiten
		final ArgumentList startParameter = getRahmenWerk().getStartParameter();

		// Default-LMS der Applikation vorbereiten
		if (startParameter.hasArgument(RahmenwerkService.LANDESMELDESTELLE_OBJEKT_PARAMETER)) {
			defaultLmsPid = startParameter.fetchArgument(RahmenwerkService.LANDESMELDESTELLE_OBJEKT_PARAMETER)
					.getValue();
			ausfuehrbar = true;
		}
		if (null == defaultLmsPid) {
			Debug.getLogger().warning(
					"Es wurde kein Parameter " + RahmenwerkService.LANDESMELDESTELLE_OBJEKT_PARAMETER + " angegeben!");
		}
	}

	/**
	 * bindet einen {@link Berechtigungen}-Service an die Komponente.
	 *
	 * @param newBerechtigungen der Service
	 */
	protected void bindBerechtigungen(final Berechtigungen newBerechtigungen) {
		this.berechtigungen = newBerechtigungen;
	}

	/**
	 * bindet einen {@link Einstellungen}-Service an die Komponente.
	 *
	 * @param newEinstellungen der Service
	 */
	protected void bindEinstellungen(final Einstellungen newEinstellungen) {
		this.einstellungen = newEinstellungen;
	}

	protected void bindObjektFactory(final ObjektFactory newObjektFactory) {
		objektFactory = newObjektFactory;
		// objektFactory.addPropertyChangeListener(this);
	}

	/**
	 * bindet einen Rahmenwerk-Service an die Komponente.
	 *
	 * @param newRahmenWerk der Service
	 */
	protected void bindRahmenwerk(final Rahmenwerk newRahmenWerk) {
		this.rahmenWerk = newRahmenWerk;
	}

	/** deaktiviert die Komponente. */
	protected void deactivate() {
		RahmenwerkService.service = null;
	}

	/**
	 * liefert den {@link Berechtigungen}-Service.
	 *
	 * @return den Service
	 */
	public Berechtigungen getBerechtigungen() {
		return berechtigungen;
	}

	/**
	 * liefert den {@link Einstellungen}-Service.
	 *
	 * @return den Service
	 */
	public Einstellungen getEinstellungen() {
		return einstellungen;
	}

	/**
	 * Liefert die per Kommandozeilenparameter festgelegte Landesmeldestelle
	 * zur&uuml;ck.
	 *
	 * @return the landesMeldeStelle
	 */
	public TmcLandesMeldeStelle getLandesMeldeStelle() {
		if (null == landesMeldeStelle) {
			Assert.isNotNull(objektFactory, "Objekt-Factory konnte nicht initialisiert werden.");
			if (defaultLmsPid != null) {
				if (objektFactory.isVerbunden()) {
					final SystemObjekt lms = objektFactory.getModellobjekt(defaultLmsPid);
					if (lms instanceof TmcLandesMeldeStelle) {
						landesMeldeStelle = (TmcLandesMeldeStelle) lms;
					}
				}
			}
		}
		return landesMeldeStelle;
	}

	public ObjektFactory getObjektFactory() {
		return objektFactory;
	}

	/**
	 * liefert den {@link Rahmenwerk}-Service.
	 *
	 * @return den Service
	 */
	public Rahmenwerk getRahmenWerk() {
		return rahmenWerk;
	}

	/**
	 * Liefert <code>true</code> zur&uuml;ck, wenn dieses Plug-in ausf&uuml;hrbar
	 * ist.
	 *
	 * <p>
	 * Das Plug-in ist nicht ausf&uuml;hrbar, wenn keine g&uuml;ltige
	 * Landesmeldestelle als Aufrufeparameter &uuml;bergeben wurde.
	 * </p>
	 *
	 * @return <code>true</code>, wenn dieses Plug-in ausf&uuml;hrbar ist, ansonsten
	 *         <code>false</code>.
	 */
	public boolean isAusfuehrbar() {
		return ausfuehrbar;
	}

	/**
	 * entfernt einen {@link Berechtigungen}-Service von der Komponente.
	 *
	 * @param newBerechtigungen der Service
	 */
	protected void unbindBerechtigungen(final Berechtigungen newBerechtigungen) {
		this.berechtigungen = null;
	}

	/**
	 * entfernt einen {@link Einstellungen}-Service von der Komponente.
	 *
	 * @param newEinstellungen der Service
	 */
	protected void unbindEinstellungen(final Einstellungen newEinstellungen) {
		this.einstellungen = null;
	}

	protected void unbindObjektFactory(@SuppressWarnings("unused") final ObjektFactory newObjektFaktory) {
		objektFactory = null;
	}

	/**
	 * entfernt einen Rahmenwerk-Service von der Komponente.
	 *
	 * @param newRahmenWerk der Service
	 */
	protected void unbindRahmenwerk(final Rahmenwerk newRahmenWerk) {
		this.rahmenWerk = null;
	}
}
