/*
 * Rahmenwerk-Plug-in "RDS-Detailanzeige"
 * Spezial-Plug-in VRZ3 - Baden-Württemberg
 *
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */

package de.bsvrz.buv.plugin.verkehrsmeldung.detail.views;

import java.util.Collection;
import java.util.HashSet;

import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.Platform;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.IMenuListener;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.action.Separator;
import org.eclipse.jface.layout.GridDataFactory;
import org.eclipse.jface.layout.GridLayoutFactory;
import org.eclipse.jface.layout.TreeColumnLayout;
import org.eclipse.jface.viewers.AbstractTreeViewer;
import org.eclipse.jface.viewers.ColumnWeightData;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Tree;
import org.eclipse.swt.widgets.TreeColumn;
import org.eclipse.ui.ISelectionListener;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.part.ViewPart;

import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.sys.funclib.bitctrl.modell.tmkexlmstglobal.objekte.RdsMeldung;
import de.bsvrz.sys.funclib.bitctrl.modell.tmkexlmstic3.objekte.TmcTic3infoMeldung;
import de.bsvrz.sys.funclib.bitctrl.modell.tmtmcverkehrsmeldung.objekte.TmcVerkehrsMeldung;

/**
 * Ansicht zur Darstellung der Details einer RDS-Meldung. Die Ansicht besteht
 * aus einer Baumdarstellung in der die Attributgruppe "atg.rdsMeldung" mit den
 * möglichen Aspekten dargestellt wird. Die Darstellung bildet direkt die
 * Datenverteilerstruktur ab und verwendet zur Darstellung die Ausgabefunktionen
 * der Datenverteiler-API.
 *
 * @author BitCtrl Systems GmbH, Uwe Peuker
 *
 */
public class RdsDetailsView extends ViewPart implements ISelectionListener {

	/** interne Aktion zum Zuklappen aller Elemente des dargestellten Baums. */
	private class CollapseAllAction extends Action {

		/** Standardkonstruktor. */
		CollapseAllAction() {
			super("Alles zuklappen");
		}

		@Override
		public boolean isEnabled() {
			return viewer != null;
		}

		@Override
		public void run() {
			if (viewer != null) {
				viewer.collapseAll();
			}
		}
	}

	/** interne Aktion zum Aufklappen aller Elemente des dargestellten Baums. */
	private class ExpandAllAction extends Action {

		/** Standardkonstruktor. */
		ExpandAllAction() {
			super("Alles aufklappen");
		}

		@Override
		public boolean isEnabled() {
			return viewer != null;
		}

		@Override
		public void run() {
			if (viewer != null) {
				viewer.expandAll();
			}
		}
	}

	/**
	 * interne Aktion zum Aufklappen aller Unterelemente des aktuell ausgewählten
	 * Elements.
	 */
	private class ExpandElementAction extends Action {

		/** das Element, dessen Kinder ausgeklappt werden sollen. */
		private final Object element;

		/**
		 * Konstruktor, erzeugt eine Aktion f�r das übergebene Element, dessen Kinder
		 * aufgeklappt werden sollen.
		 *
		 * @param element das Element
		 */
		ExpandElementAction(final Object element) {
			super("Aufklappen");
			this.element = element;
		}

		@Override
		public boolean isEnabled() {
			return (viewer != null) && (element != null);
		}

		@Override
		public void run() {
			if (viewer != null) {
				viewer.expandToLevel(element, AbstractTreeViewer.ALL_LEVELS);
			}
		}
	}

	/** die ID der Ansicht. */
	public static final String VIEW_ID = RdsDetailsView.class.getName();

	/**
	 * der Treeviewer zur Darstellung der Daten der ausgew�hlten RDS-Meldungen.
	 */
	private TreeViewer viewer;

	/**
	 * erzeugt das Kontextmen� des Baumes zur Darstellung der Detaildaten von
	 * RDS-Meldungen. Es sind Aktionen vorgesehen zum:
	 * <ul>
	 * <li>vollst�ndigen Aufklappen des Baumes</li>
	 * <li>vollst�ndigen Zuklappen des Baumes</li>
	 * <li>vollst�ndigen Aufklappen eines Unterbaumes</li>
	 * </ul>
	 */
	private void createContextMenue() {
		final MenuManager mgr = new MenuManager();
		mgr.setRemoveAllWhenShown(true);

		mgr.addMenuListener(new IMenuListener() {
			@Override
			public void menuAboutToShow(final IMenuManager manager) {
				final IStructuredSelection selection = (IStructuredSelection) viewer.getSelection();
				manager.add(new ExpandAllAction());
				manager.add(new CollapseAllAction());
				final Object element = selection.getFirstElement();
				if (element != null) {
					manager.add(new Separator());
					manager.add(new ExpandElementAction(element));
				}
			}
		});

		viewer.getControl().setMenu(mgr.createContextMenu(viewer.getControl()));

	}

	@Override
	public void createPartControl(final Composite parent) {

		final Composite panel = new Composite(parent, SWT.NONE);
		GridLayoutFactory.fillDefaults().applyTo(panel);

		final Composite tabPanel = new Composite(panel, SWT.NONE);
		GridDataFactory.fillDefaults().grab(true, true).applyTo(tabPanel);

		final Tree tree = new Tree(tabPanel, SWT.SINGLE | SWT.FULL_SELECTION);
		tree.setLinesVisible(true);
		tree.setHeaderVisible(true);

		final TreeColumnLayout treeLayout = new TreeColumnLayout();
		TreeColumn column = new TreeColumn(tree, SWT.NONE);
		column.setText("Name");
		treeLayout.setColumnData(column, new ColumnWeightData(1));

		column = new TreeColumn(tree, SWT.NONE);
		column.setText("Wert");
		treeLayout.setColumnData(column, new ColumnWeightData(2));

		tabPanel.setLayout(treeLayout);

		viewer = new TreeViewer(tree);
		viewer.setAutoExpandLevel(4);
		final RdsDetailsViewContentProvider provider = new RdsDetailsViewContentProvider();
		viewer.setContentProvider(provider);
		viewer.setLabelProvider(provider);

		createContextMenue();

		getSite().getWorkbenchWindow().getSelectionService().addSelectionListener(this);
	}

	@Override
	public void dispose() {
		getSite().getWorkbenchWindow().getSelectionService().removeSelectionListener(this);
	}

	/**
	 * ermittelt aus dem übergebenen Objekt eine Systemobjekt vom Typ "RdsMeldung".
	 * Das übergebene Element muss entweder direkt eibn derartiges Systemobjekt sein
	 * bzw. ein Adapter auf den Typ SystemObject muss ein derartiges liefern.
	 *
	 * @param obj das umzuwandelnde Objekt
	 * @return das ermittelte RdsMeldungs-Objekt oder <code>null</code>, wenn eine
	 *         Umwandlung nicht möglich war.
	 */
	private SystemObject getRdsMeldung(final Object obj) {
		SystemObject result = null;
		SystemObject sysObj = null;

		if (obj instanceof RdsMeldung) {
			result = ((RdsMeldung) obj).getSystemObject();
		} else if (obj instanceof TmcTic3infoMeldung) {
			result = ((TmcTic3infoMeldung) obj).getSystemObject();
		} else if (obj instanceof TmcVerkehrsMeldung) {
			result = ((TmcVerkehrsMeldung) obj).getSystemObject();
		} else {
			if (obj instanceof IAdaptable) {
				sysObj = ((IAdaptable) obj).getAdapter(SystemObject.class);
			}
			if ((sysObj == null) && (obj instanceof SystemObject)) {
				sysObj = (SystemObject) obj;
			}
			if (sysObj == null) {
				sysObj = Platform.getAdapterManager().getAdapter(obj, SystemObject.class);
			}
		}

		if ((result == null) && (sysObj != null) && (sysObj.isOfType(RdsMeldung.PID)
				|| sysObj.isOfType(TmcTic3infoMeldung.PID) || sysObj.isOfType(TmcVerkehrsMeldung.PID))) {
			result = sysObj;
		}
		return result;
	}

	@Override
	public void selectionChanged(final IWorkbenchPart part, final ISelection selection) {
		if (selection instanceof IStructuredSelection) {
			final Collection<SystemObject> rdsMeldungen = new HashSet<>();
			for (final Object obj : ((IStructuredSelection) selection).toList()) {
				final SystemObject meldung = getRdsMeldung(obj);
				if (meldung != null) {
					rdsMeldungen.add(meldung);
				}
			}

			viewer.setInput(rdsMeldungen);
		}
	}

	@Override
	public void setFocus() {
		if ((viewer != null) && (!viewer.getControl().isDisposed())) {
			viewer.getControl().setFocus();
		}
	}

}
