/*
 * Rahmenwerk-Plug-in "RDS-Detailanzeige"
 * Spezial-Plug-in VRZ3 - Baden-Württemberg
 *
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */

package de.bsvrz.buv.plugin.verkehrsmeldung.detail.views;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.eclipse.jface.viewers.ITableLabelProvider;
import org.eclipse.jface.viewers.ITreeContentProvider;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.swt.events.DisposeEvent;
import org.eclipse.swt.events.DisposeListener;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.widgets.Display;

import de.bsvrz.buv.plugin.verkehrsmeldung.detail.internal.RahmenwerkService;
import de.bsvrz.buv.rw.basislib.Rahmenwerk;
import de.bsvrz.dav.daf.main.ClientDavInterface;
import de.bsvrz.dav.daf.main.ClientReceiverInterface;
import de.bsvrz.dav.daf.main.Data;
import de.bsvrz.dav.daf.main.DataDescription;
import de.bsvrz.dav.daf.main.ReceiveOptions;
import de.bsvrz.dav.daf.main.ReceiverRole;
import de.bsvrz.dav.daf.main.ResultData;
import de.bsvrz.dav.daf.main.config.Aspect;
import de.bsvrz.dav.daf.main.config.Attribute;
import de.bsvrz.dav.daf.main.config.AttributeGroup;
import de.bsvrz.dav.daf.main.config.AttributeListDefinition;
import de.bsvrz.dav.daf.main.config.DataModel;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.sys.funclib.bitctrl.modell.tmkexlmstglobal.objekte.RdsMeldung;
import de.bsvrz.sys.funclib.bitctrl.modell.tmkexlmstglobal.parameter.PdRdsMeldung;
import de.bsvrz.sys.funclib.bitctrl.modell.tmkexlmstic3.objekte.TmcTic3infoMeldung;
import de.bsvrz.sys.funclib.bitctrl.modell.tmkexlmstic3.onlinedaten.OdTmcTic3infoMeldung;
import de.bsvrz.sys.funclib.bitctrl.modell.tmtmcverkehrsmeldung.objekte.TmcVerkehrsMeldung;
import de.bsvrz.sys.funclib.bitctrl.modell.tmtmcverkehrsmeldung.parameter.PdTmcVerkehrsMeldung;

/**
 * Content-Provider für die Darstellung der Daten von Verkehrsmeldungen im
 * {@link RdsDetailsView}.
 *
 * @author BitCtrl Systems GmbH, Uwe Peuker, Steffen Gieseler
 * 
 */
class RdsDetailsViewContentProvider extends LabelProvider
		implements ITableLabelProvider, ITreeContentProvider, DisposeListener, ClientReceiverInterface {

	/**
	 * interne Klasse zur Zuordnung, der potentiellen unter verschiedenen
	 * Aspekten vorliegenden Daten einer Verkehrsmeldung.
	 */
	static class DataSet {

		/** interne Verwaltung der Daten für die einzelnen Aspekte. */
		private final Map<Aspect, DataSetItem> items = new HashMap<>();

		/**
		 * Konstruktor, erzeugt einen Datensatz für die übergebene
		 * Verkehrsmeldung.
		 *
		 * @param obj
		 *            eine Systemobjekt vom Typ Verkehrsmeldung
		 */
		DataSet(final SystemObject obj) {
			Collection<Aspect> aspekts = null;

			if (obj.isOfType(RdsMeldung.PID)) {
				aspekts = RdsDetailsViewContentProvider.ASPECTS_RDS;
			}

			if (obj.isOfType(TmcTic3infoMeldung.PID)) {
				aspekts = RdsDetailsViewContentProvider.ASPECTS_TIC3;
			}

			if (obj.isOfType(TmcVerkehrsMeldung.PID)) {
				aspekts = RdsDetailsViewContentProvider.ASPECTS_TMC;
			}

			if (aspekts != null) {
				for (final Aspect aspect : aspekts) {
					items.put(aspect, new DataSetItem(aspect));
				}
			}
		}

		/**
		 * liefert die Datensätze zu denen gültige Daten vom Datenverteiler
		 * empfangen wurden.
		 *
		 * @return die Liste der gültigen Datensätze
		 */
		public Collection<DataSetItem> getValidItems() {
			final Collection<DataSetItem> result = new ArrayList<>();
			for (final DataSetItem item : items.values()) {
				if (item.getData() != null) {
					result.add(item);
				}
			}
			return result;
		}

		/**
		 * ermittelt, ob unter mindestens einem Aspekt Daten für die zugeordnete
		 * RDS-Meldung verfügbar sind.
		 *
		 * @return den Zustand
		 */
		public boolean hasData() {
			boolean result = false;
			for (final DataSetItem item : items.values()) {
				if (item.getData() != null) {
					result = true;
					break;
				}
			}
			return result;
		}

		/**
		 * aktualisiert die Daten für den übergebenen Aspekt.
		 *
		 * @param aspect
		 *            der Aspekt
		 * @param data
		 *            die Daten
		 */
		public void update(final Aspect aspect, final Data data) {
			final DataSetItem item = items.get(aspect);
			if (item != null) {
				item.setData(data);
			}
		}
	}

	/** interner Eintrag zur Verwaltung der Daten für einen Aspekt. */
	static class DataSetItem {

		/** der Aspekt, unter dem die Daten betrachtet werden. */
		private final Aspect aspect;
		/** die Daten selbst. */
		private Data data;

		/**
		 * Konstruktor, erzeugt einen Datenspeicher für den übergebenen Aspekt.
		 *
		 * @param aspect
		 *            der Aspekt
		 */
		DataSetItem(final Aspect aspect) {
			this.aspect = aspect;
		}

		/**
		 * liefert die zugeordneten Daten.
		 *
		 * @return die Daten
		 */
		public Data getData() {
			return data;
		}

		/**
		 * setzt die zugeordneten Daten.
		 *
		 * @param data
		 *            die neuen Daten
		 */
		public void setData(final Data data) {
			this.data = data;
		}

		@Override
		public String toString() {
			return aspect.getNameOrPidOrId();
		}
	}

	/** die Aspekte für die RDS-Daten visualisiert werden. */
	private static final Collection<Aspect> ASPECTS_RDS = new ArrayList<>();

	/** die Aspekte für die TIC-Daten visualisiert werden. */
	private static final Collection<Aspect> ASPECTS_TIC3 = new ArrayList<>();

	/** die Aspekte für die TMC-Daten visualisiert werden. */
	private static final Collection<Aspect> ASPECTS_TMC = new ArrayList<>();

	/** die Attributgruppe, die die Daten der RDS-Meldungen enthält. */
	private static AttributeGroup rdsAtg;

	/** die Attributgruppe, die die Daten der TIC3-Meldungen enthält. */
	private static AttributeGroup tic3Atg;

	/** die Attributgruppe, die die Daten der TMC-Meldungen enthält. */
	private static AttributeGroup tmcAtg;

	/** die Menge der darzustellenden Meldungen mit ihren Datensätzen. */
	private final Map<SystemObject, DataSet> meldungen = new HashMap<>();

	/** der Viewer, in dem die Daten visualisiert werden. */
	private TreeViewer viewer;

	/**
	 * stellt die Verbindung der anzuzeigenden RDS-Meldungen mit dem
	 * Datenverteiler her. Es erfolgt eine Empfängeranmeldung für alle
	 * anzuzeigenden Objekte für die Attributgruppe "atg.rdsMeldung" mit allen
	 * potentiell erwarteteten Aspekten.
	 */
	private void connect() {
		final Rahmenwerk rahmenwerk = RahmenwerkService.getService().getRahmenWerk();
		if (rahmenwerk.isOnline()) {
			initTypeVariables(rahmenwerk.getDavVerbindung());
			final Set<SystemObject> objects = meldungen.keySet();

			final List<SystemObject> rdsObjects = new ArrayList<>();
			final List<SystemObject> ticObjects = new ArrayList<>();
			final List<SystemObject> tmcObjects = new ArrayList<>();

			for (final SystemObject obj : objects) {
				if (obj.isOfType(RdsMeldung.PID)) {
					rdsObjects.add(obj);
				}

				if (obj.isOfType(TmcTic3infoMeldung.PID)) {
					ticObjects.add(obj);
				}

				if (obj.isOfType(TmcVerkehrsMeldung.PID)) {
					tmcObjects.add(obj);
				}
			}

			if (!rdsObjects.isEmpty()) {
				for (final Aspect asp : RdsDetailsViewContentProvider.ASPECTS_RDS) {
					rahmenwerk.getDavVerbindung().subscribeReceiver(this, rdsObjects,
							new DataDescription(RdsDetailsViewContentProvider.rdsAtg, asp), ReceiveOptions.normal(),
							ReceiverRole.receiver());
				}
			}

			if (!ticObjects.isEmpty()) {
				for (final Aspect asp : RdsDetailsViewContentProvider.ASPECTS_TIC3) {
					rahmenwerk.getDavVerbindung().subscribeReceiver(this, ticObjects,
							new DataDescription(RdsDetailsViewContentProvider.tic3Atg, asp), ReceiveOptions.normal(),
							ReceiverRole.receiver());
				}
			}

			if (!tmcObjects.isEmpty()) {
				for (final Aspect asp : RdsDetailsViewContentProvider.ASPECTS_TMC) {
					rahmenwerk.getDavVerbindung().subscribeReceiver(this, tmcObjects,
							new DataDescription(RdsDetailsViewContentProvider.tmcAtg, asp), ReceiveOptions.normal(),
							ReceiverRole.receiver());
				}
			}
		}
	}

	/**
	 * trennt die Verbindung der anzuzeigenden RDS-Meldungen mit dem
	 * Datenverteiler. Es werden alle erfolgten Empfängeranmeldungen abgemeldet.
	 */
	private void disconnect() {
		final Rahmenwerk rahmenwerk = RahmenwerkService.getService().getRahmenWerk();
		if (rahmenwerk.isOnline()) {
			final List<SystemObject> rdsObjects = new ArrayList<>();
			final List<SystemObject> ticObjects = new ArrayList<>();
			final List<SystemObject> tmcObjects = new ArrayList<>();

			for (final SystemObject obj : meldungen.keySet()) {
				if (obj.isOfType(RdsMeldung.PID)) {
					rdsObjects.add(obj);
				}

				if (obj.isOfType(TmcTic3infoMeldung.PID)) {
					ticObjects.add(obj);
				}

				if (obj.isOfType(TmcVerkehrsMeldung.PID)) {
					tmcObjects.add(obj);
				}
			}

			if (!rdsObjects.isEmpty() && (RdsDetailsViewContentProvider.rdsAtg != null)) {
				for (final Aspect asp : RdsDetailsViewContentProvider.ASPECTS_RDS) {
					rahmenwerk.getDavVerbindung().unsubscribeReceiver(this, rdsObjects,
							new DataDescription(RdsDetailsViewContentProvider.rdsAtg, asp));
				}
			}

			if (!ticObjects.isEmpty() && (RdsDetailsViewContentProvider.tic3Atg != null)) {
				for (final Aspect asp : RdsDetailsViewContentProvider.ASPECTS_TIC3) {
					rahmenwerk.getDavVerbindung().unsubscribeReceiver(this, ticObjects,
							new DataDescription(RdsDetailsViewContentProvider.tic3Atg, asp));
				}
			}

			if (!tmcObjects.isEmpty() && (RdsDetailsViewContentProvider.tmcAtg != null)) {
				for (final Aspect asp : RdsDetailsViewContentProvider.ASPECTS_TMC) {
					rahmenwerk.getDavVerbindung().unsubscribeReceiver(this, tmcObjects,
							new DataDescription(RdsDetailsViewContentProvider.tmcAtg, asp));
				}
			}
		}
	}

	@Override
	public Object[] getChildren(final Object parentElement) {
		final Collection<Object> result = new ArrayList<>();
		if (parentElement instanceof Collection<?>) {
			result.addAll(meldungen.keySet());
		} else if (parentElement instanceof SystemObject) {
			final DataSet dataSet = meldungen.get(parentElement);
			if (dataSet.hasData()) {
				result.addAll(dataSet.getValidItems());
			}
		} else if (parentElement instanceof DataSetItem) {
			if (((DataSetItem) parentElement).getData() != null) {
				result.add(((DataSetItem) parentElement).getData());
			}
		} else if (parentElement instanceof Data) {
			final Data data = (Data) parentElement;
			if (data.isArray()) {
				final Data.Array array = data.asArray();
				for (int idx = 0; idx < array.getLength(); idx++) {
					result.add(array.getItem(idx));
				}
			} else if (data.isList()) {
				final AttributeListDefinition type = (AttributeListDefinition) data.getAttributeType();

				if (type == null) {
					List<Attribute> attribute = null;
					if (data.getName().equals(PdRdsMeldung.PID)) {
						attribute = RdsDetailsViewContentProvider.rdsAtg.getAttributes();
					}

					if (data.getName().equals(OdTmcTic3infoMeldung.PID)) {
						attribute = RdsDetailsViewContentProvider.tic3Atg.getAttributes();
					}

					if (data.getName().equals(PdTmcVerkehrsMeldung.PID)) {
						attribute = RdsDetailsViewContentProvider.tmcAtg.getAttributes();
					}

					if (attribute != null) {
						for (final Attribute att : attribute) {
							result.add(data.getItem(att.getName()));
						}
					}
				} else {
					for (final Attribute att : type.getAttributes()) {
						result.add(data.getItem(att.getName()));
					}
				}
			}
		}
		return result.toArray();
	}

	@Override
	public Image getColumnImage(final Object element, final int columnIndex) {
		return null;
	}

	@Override
	public String getColumnText(final Object element, final int columnIndex) {
		String result = null;
		switch (columnIndex) {
		case 0:
			if (element instanceof Data) {
				result = ((Data) element).getName();
			}
			break;
		case 1:
			if ((element instanceof Data) && ((Data) element).isPlain()) {
				result = ((Data) element).valueToString();
			}
			if (result == null) {
				result = "";
			}
			break;
		default:
			break;
		}

		if (result == null) {
			result = getText(element);
		}

		// entferne Zeilenumbrüche (aus Meldungstext)
		result = result.replaceAll("\\r\\n|\\r|\\n", " ").trim();

		return result;
	}

	@Override
	public Object[] getElements(final Object inputElement) {
		return getChildren(inputElement);
	}

	@Override
	public Object getParent(final Object element) {
		return null;
	}

	@Override
	public boolean hasChildren(final Object element) {
		return getChildren(element).length > 0;
	}

	/**
	 * initialisiert global die Systemobjekte für die Attributgruppe
	 * "atg.rdsMeldung" und die erwarteten Aspekte für die Daten von
	 * RDS-Meldungen.
	 *
	 * @param dav
	 *            die verwendete Datenverteilerverbindung
	 */
	private void initTypeVariables(final ClientDavInterface dav) {
		if ((RdsDetailsViewContentProvider.rdsAtg == null) && (dav != null)) {
			final DataModel dataModel = dav.getDataModel();
			RdsDetailsViewContentProvider.rdsAtg = dataModel.getAttributeGroup(PdRdsMeldung.PID);
			RdsDetailsViewContentProvider.ASPECTS_RDS
					.add(dataModel.getAspect(PdRdsMeldung.Aspekte.RdsBearbeitet.getPid()));
			RdsDetailsViewContentProvider.ASPECTS_RDS
					.add(dataModel.getAspect(PdRdsMeldung.Aspekte.RdsEmpfangen.getPid()));
			RdsDetailsViewContentProvider.ASPECTS_RDS
					.add(dataModel.getAspect(PdRdsMeldung.Aspekte.RdsGeneriert.getPid()));
			RdsDetailsViewContentProvider.ASPECTS_RDS.add(dataModel.getAspect(PdRdsMeldung.Aspekte.RdsSenden.getPid()));
			RdsDetailsViewContentProvider.ASPECTS_RDS
					.add(dataModel.getAspect(PdRdsMeldung.Aspekte.RdsVersendet.getPid()));
		}

		if ((RdsDetailsViewContentProvider.tic3Atg == null) && (dav != null)) {
			final DataModel dataModel = dav.getDataModel();
			RdsDetailsViewContentProvider.tic3Atg = dataModel.getAttributeGroup(OdTmcTic3infoMeldung.PID);
			RdsDetailsViewContentProvider.ASPECTS_TIC3
					.add(dataModel.getAspect(OdTmcTic3infoMeldung.Aspekte.LmsEmpfangen.getPid()));
		}

		if ((RdsDetailsViewContentProvider.tmcAtg == null) && (dav != null)) {
			final DataModel dataModel = dav.getDataModel();
			RdsDetailsViewContentProvider.tmcAtg = dataModel.getAttributeGroup(PdTmcVerkehrsMeldung.PID);
			RdsDetailsViewContentProvider.ASPECTS_TMC
					.add(dataModel.getAspect(PdTmcVerkehrsMeldung.Aspekte.TmcEmpfangen.getPid()));
			RdsDetailsViewContentProvider.ASPECTS_TMC
					.add(dataModel.getAspect(PdTmcVerkehrsMeldung.Aspekte.TmcBearbeitet.getPid()));
			RdsDetailsViewContentProvider.ASPECTS_TMC
					.add(dataModel.getAspect(PdTmcVerkehrsMeldung.Aspekte.TmcGeneriert.getPid()));
			RdsDetailsViewContentProvider.ASPECTS_TMC
					.add(dataModel.getAspect(PdTmcVerkehrsMeldung.Aspekte.TmcSenden.getPid()));
			RdsDetailsViewContentProvider.ASPECTS_TMC
					.add(dataModel.getAspect(PdTmcVerkehrsMeldung.Aspekte.TmcVersendet.getPid()));
		}
	}

	@Override
	public void inputChanged(final Viewer dataViewer, final Object oldInput, final Object newInput) {

		initTypeVariables(RahmenwerkService.getService().getRahmenWerk().getDavVerbindung());

		if ((dataViewer != null) && (!dataViewer.equals(viewer))) {
			if (viewer != null) {
				viewer.getControl().removeDisposeListener(this);
			}
			viewer = (TreeViewer) dataViewer;
			viewer.getControl().addDisposeListener(this);

		}

		disconnect();
		meldungen.clear();
		if (newInput instanceof Collection<?>) {
			for (final Object obj : (Collection<?>) newInput) {
				if ((((((obj instanceof SystemObject) && ((((SystemObject) obj).isOfType(RdsMeldung.PID))))
						|| ((SystemObject) obj).isOfType(TmcTic3infoMeldung.PID))
						|| ((SystemObject) obj).isOfType(TmcVerkehrsMeldung.PID)))) {
					meldungen.put((SystemObject) obj, new DataSet((SystemObject) obj));
				}
			}
		}
		connect();
	}

	@Override
	public void update(final ResultData[] results) {
		boolean updated = false;
		for (final ResultData result : results) {
			final DataSet dataSet = meldungen.get(result.getObject());
			if (dataSet != null) {
				dataSet.update(result.getDataDescription().getAspect(), result.getData());
				updated = true;
			}
		}

		if (updated) {
			Display.getDefault().asyncExec(new Runnable() {
				@Override
				public void run() {
					if ((viewer != null) && (!viewer.getControl().isDisposed())) {
						viewer.refresh();
					}
				}
			});
		}
	}

	@Override
	public void widgetDisposed(final DisposeEvent e) {
		disconnect();
	}
}
