/*******************************************************************************
 * Basis-Plug-In fr die Darstellung von Charts mit BIRT
 * Copyright (C) 2009 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 ******************************************************************************/
package com.bitctrl.lib.eclipse.birt.chart;

import org.eclipse.birt.chart.device.IDeviceRenderer;
import org.eclipse.birt.chart.device.IUpdateNotifier;
import org.eclipse.birt.chart.exception.ChartException;
import org.eclipse.birt.chart.factory.GeneratedChartState;
import org.eclipse.birt.chart.factory.Generator;
import org.eclipse.birt.chart.model.Chart;
import org.eclipse.birt.chart.model.attribute.Bounds;
import org.eclipse.birt.chart.model.attribute.impl.BoundsImpl;
import org.eclipse.birt.chart.util.PluginSettings;
import org.eclipse.core.runtime.ILog;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.BusyIndicator;
import org.eclipse.swt.events.ControlAdapter;
import org.eclipse.swt.events.ControlEvent;
import org.eclipse.swt.events.PaintEvent;
import org.eclipse.swt.events.PaintListener;
import org.eclipse.swt.graphics.GC;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.graphics.Rectangle;
import org.eclipse.swt.widgets.Canvas;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;

import com.bitctrl.lib.eclipse.birt.BirtPluginActivator;
import com.bitctrl.lib.eclipse.job.BackgroundUIJob;

/**
 * The canvas to show chart.
 *
 * @author Qi Liang
 * @author BitCtrl Systems GmbH, Ch. Hsel
 */
public class ChartCanvas extends Canvas implements IUpdateNotifier {
	private final ILog LOGGER = BirtPluginActivator.getDefault().getLog();

	/**
	 * The device render for rendering chart.
	 */
	protected IDeviceRenderer render = null;

	private boolean doubleBuffering = false;

	/**
	 * The chart instance.
	 */
	protected Chart chart = null;

	/**
	 * The chart state.
	 */
	protected GeneratedChartState state = null;

	private static enum UpdateType {
		a_None, b_Refresh, c_Render, d_Regenerate
	}

	private UpdateType update = UpdateType.d_Regenerate;

	private int painting = 0;

	private String nullMessage = null;

	private Image cachedImage;

	/**
	 * Constructs one canvas containing chart.
	 *
	 * @param parent a composite control which will be the parent of the new
	 *               instance (cannot be null)
	 * @param style  the style of control to construct
	 */
	public ChartCanvas(final Composite parent, final int style) {
		super(parent, style | SWT.NO_FOCUS);
		// initialize the SWT rendering device
		final PaintListener paintListener = new PaintListener() {

			@Override
			public void paintControl(final PaintEvent e) {
				BusyIndicator.showWhile(getDisplay(), new Runnable() {
					@Override
					public void run() {
						synchronized (ChartCanvas.class) {
							if (0 == painting) {
								if (null == render) {
									painting++;
									initRenderer();
									paintMessage(e, "Der Renderer wird initialisiert ...");
									return;
								}
								// default case => no return
							} else {
								paintMessage(e, "Bitte warten ...");
								painting++;
								return;
							}
							painting++;
						}
						UpdateType old_update = UpdateType.d_Regenerate;
						if (null != chart) {
							final GC gc = e.gc;
							try {

								render.setProperty(IDeviceRenderer.GRAPHICS_CONTEXT, gc);

								final Generator gr = Generator.instance();
								final Rectangle rectangle = getClientArea();

								switch (getUpdate()) {
								case d_Regenerate:
									final Point size2 = getSize();
									final Bounds bo = BoundsImpl.create(0, 0, size2.x, size2.y);
									final int resolution = render.getDisplayServer().getDpiResolution();
									bo.scale(72d / resolution);
									state = gr.build(render.getDisplayServer(), chart, bo, null, null, null);
									if (doubleBuffering) {
										drawToCachedImage(rectangle);
									}
									break;
								case b_Refresh:
									gr.refresh(state);
									if (doubleBuffering) {
										drawToCachedImage(rectangle);
									}
								case c_Render:
								default:
									break;
								}

								if (doubleBuffering) {
									gc.drawImage(cachedImage, 0, 0, cachedImage.getBounds().width,
											cachedImage.getBounds().height, 0, 0, rectangle.width, rectangle.height);
								} else {
									// XXX: an dieser Stelle liegt
									// wahrscheinlich
									// der Grund, warum es nach Aussage von
									// Herrn
									// Uhlmann manchmal zu stndigen repaints
									// innerhalb des Charts kommt. als Folge
									// kann
									// man die Oberflche nicht mehr bedienen.
									// Warum auch immer, aber dass gr.render()
									// braucht hier zu lange.
									gr.render(render, state);
								}
								old_update = getUpdate();
								setUpdate(UpdateType.a_None);

							} catch (final ChartException e1) {
								// e1.printStackTrace();
								paintMessage(e, e1.getLocalizedMessage());
							}
						} else {
							paintMessage(e,
									null != nullMessage ? nullMessage : "Es kann kein Diagramm angezeigt werden!");
						}
						synchronized (ChartCanvas.class) {
							painting--;
							if (0 < painting) {
								painting = 0;
								setUpdate(old_update);
								// Paint event while painting => paint restart
								redraw();
							}
						}
					}
				});
			}

			private void paintMessage(final PaintEvent e, final String message) {
				LOGGER.log(new Status(IStatus.WARNING, BirtPluginActivator.PLUGIN_ID, "ChartCanvas:" + message));
				e.gc.setBackground(Display.getDefault().getSystemColor(SWT.COLOR_WHITE));
				e.gc.setForeground(Display.getDefault().getSystemColor(SWT.COLOR_BLACK));
				e.gc.fillRectangle(e.x, e.y, e.width, e.height);
				e.gc.setFont(null);
				e.gc.drawText(message, e.x, e.y);
			}

			private void drawToCachedImage(final Rectangle rectangle) {
				GC gc = null;
				try {
					if (cachedImage != null) {
						cachedImage.dispose();
					}
					cachedImage = new Image(Display.getCurrent(), rectangle);

					gc = new GC(cachedImage);
					render.setProperty(IDeviceRenderer.GRAPHICS_CONTEXT, gc);

					final Generator gr = Generator.instance();
					gr.render(render, state);
				} catch (final ChartException ex) {
					LOGGER.log(new Status(IStatus.WARNING, BirtPluginActivator.PLUGIN_ID,
							"ChartCanvas Double Buffered Rendering:", ex));
				} finally {
					if (gc != null) {
						gc.dispose();
					}
				}

			}

		};
		addPaintListener(paintListener);

		addControlListener(new ControlAdapter() {
			@Override
			public void controlResized(final ControlEvent e) {
				regenerateChart();
			}
		});
	}

	/**
	 * Returns the chart which is contained in this canvas.
	 *
	 * @return the chart contained in this canvas.
	 */
	public final Chart getChart() {
		return chart;
	}

	/**
	 * Sets the chart into this canvas. Note: When the chart is set, the cached
	 * image will be dopped, but this method doesn't reset the flag
	 * <code>cachedImage</code>.
	 *
	 * @param chart the chart to set
	 */
	public void setChart(final Chart chart) {
		this.chart = chart;
		regenerateChart();
	}

	@Override
	public final Chart getDesignTimeModel() {
		return chart;
	}

	@Override
	public final Chart getRunTimeModel() {
		return state.getChartModel();
	}

	@Override
	public final Object peerInstance() {
		return this;
	}

	@Override
	public final void regenerateChart() {
		setUpdate(UpdateType.d_Regenerate);
		if (!isDisposed()) {
			redraw();
		}
	}

	@Override
	public final void repaintChart() {
		setUpdate(UpdateType.c_Render);
		if (!isDisposed()) {
			redraw();
		}
	}

	public final void refreshChart() {
		setUpdate(UpdateType.b_Refresh);
		if (!isDisposed()) {
			redraw();
		}
	}

	public void setNullMessage(final String nullMessage) {
		this.nullMessage = nullMessage;
	}

	private synchronized void setUpdate(final UpdateType newVal) {
		if (UpdateType.a_None == newVal || update.ordinal() < newVal.ordinal()) {
			update = newVal;
		}
	}

	private synchronized UpdateType getUpdate() {
		return update;
	}

	protected void initRenderer() {
		final Job job = new BackgroundUIJob("Bitte warten...") {
			@Override
			protected IStatus runInBGThread(final IProgressMonitor monitor) {
				try {
					monitor.beginTask("Initialisung der Diagramm-Bibliothek", 1);
					final PluginSettings ps = PluginSettings.instance();
					render = ps.getDevice(getRendererDeviceId());
					runUI(monitor);
				} catch (final ChartException e1) {
					e1.printStackTrace();
					painting = 0;
					regenerateChart();
				}
				return Status.OK_STATUS;
			}

			@Override
			protected void runInUIThread(final IProgressMonitor monitor) {
				monitor.done();
				render.setProperty(IDeviceRenderer.UPDATE_NOTIFIER, ChartCanvas.this);
				painting = 0;
				// redraw after dialog close
				getDisplay().asyncExec(new Runnable() {
					@Override
					public void run() {
						regenerateChart();
					}
				});
			}
		};
		job.setPriority(Job.SHORT);
		job.setUser(true);
		job.schedule(); // start as soon as possible
	}

	protected String getRendererDeviceId() {
		return "dv.SWT";
	}

	/**
	 * @return the render
	 */
	public IDeviceRenderer getRender() {
		return render;
	}

	/**
	 * @return the doubleBuffering
	 */
	public final boolean isDoubleBuffering() {
		return doubleBuffering;
	}

	/**
	 * @param doubleBuffering the doubleBuffering to set
	 */
	public final void setDoubleBuffering(final boolean doubleBuffering) {
		this.doubleBuffering = doubleBuffering;
	}
}
