/* Segment 13 Bedienung und Visualisierung, Plug-In Streckenprofil
 * Copyright (C) 2009-2010 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weienfelser Strae 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */

package com.bitctrl.lib.eclipse.birt.chart.print;

import java.io.BufferedOutputStream;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;

import org.eclipse.birt.chart.api.ChartEngine;
import org.eclipse.birt.chart.device.IDeviceRenderer;
import org.eclipse.birt.chart.exception.ChartException;
import org.eclipse.birt.chart.factory.GeneratedChartState;
import org.eclipse.birt.chart.factory.Generator;
import org.eclipse.birt.chart.model.Chart;
import org.eclipse.birt.chart.model.attribute.LineAttributes;
import org.eclipse.birt.chart.model.attribute.impl.BoundsImpl;
import org.eclipse.birt.core.framework.PlatformConfig;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.nebula.paperclips.core.ImagePrint;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.ImageData;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.widgets.Display;

/**
 * Ein Print - Objekt zum drucken von {@link Chart}s.
 *
 * @author BitCtrl Systems GmbH, Hoesel
 * @version $Id: $
 */
public class BirtChartPrint extends ImagePrint {

	/**
	 * Erzeugt ein {@link ImagePrint} - Objekt eines {@link Chart}s.
	 *
	 * @param chart das zu druckende {@link Chart}.
	 */
	public BirtChartPrint(final Chart chart) {
		super(chart2Image(chart));
	}

	/**
	 * Erzeugt aus einem {@link Chart}, ein {@link ImageData}.
	 *
	 * @param chart das {@link Chart}.
	 * @return das {@link ImageData}.
	 */
	private static ImageData chart2Image(final Chart chart) {
		ImageData result = null;
		final Display current = Display.getCurrent();
		try {
			// Initialize.
			final PlatformConfig config = new PlatformConfig();
			config.setProperty("STANDALONE", "true");
			final IDeviceRenderer render = ChartEngine.instance(config).getRenderer("dv.BMP");
			final Generator gr = Generator.instance();

			final Point dpi = current.getDPI();
			final GeneratedChartState state = gr.build(render.getDisplayServer(),
					pimpChart(chart, 72.0 * 2 / (dpi.x + dpi.y)),
					BoundsImpl.create(0, 0, chart.getBlock().getBounds().getWidth() * 72 / dpi.x,
							chart.getBlock().getBounds().getHeight() * 72 / dpi.y),
					null, null, null);
			// Render
			final ByteArrayOutputStream baos = new ByteArrayOutputStream();
			final BufferedOutputStream bos = new BufferedOutputStream(baos);

			render.setProperty("output.format", "BMP");
			render.setProperty(IDeviceRenderer.FILE_IDENTIFIER, bos);

			gr.render(render, state);

			// Get image from stream.
			final ByteArrayInputStream bais = new ByteArrayInputStream(baos.toByteArray());
			final Image image = new Image(current, bais);
			result = image.getImageData();
			image.dispose();

		} catch (final ChartException e) {
			e.printStackTrace();
		}
		return result;
	}

	/**
	 * Anpassen des {@link Chart}s an das Ausgabegert.
	 *
	 * @param chart     das zu druckende Chart.
	 * @param dpiFactor die DPI - Zahl des Ausgabegertes
	 * @return ein an das Ausgabegert angepasstes {@link Chart}.
	 */
	private static Chart pimpChart(final Chart chart, final double dpiFactor) {
		final EcoreUtil.Copier copier = new EcoreUtil.Copier() {
			/**
			 * ID.
			 */
			private static final long serialVersionUID = 1L;

			@Override
			public EObject copy(final EObject object) {
				final EObject copy = super.copy(object);
				if (copy instanceof LineAttributes) {
					final LineAttributes lineAttributes = (LineAttributes) copy;
					if (lineAttributes.isSetThickness()) {
						lineAttributes.setThickness((int) (lineAttributes.getThickness() / dpiFactor));
					}
				}
				return copy;
			}
		};
		final Chart ret = (Chart) copier.copy(chart);
		copier.copyReferences();
		return ret;
	}

}
