package com.bitctrl.lib.eclipse.databinding.converter;

import java.util.HashMap;
import java.util.Map;

import org.eclipse.core.databinding.conversion.IConverter;
import org.eclipse.core.runtime.Assert;

/**
 * Used to map from values to other values. E.g. enum values to their string
 * represenation.
 *
 * @author BitCtrl Systems GmbH, Enrico Schnepel
 * @version $Id$
 */
public class SwitchConverter<F, T> implements IConverter {

	private final Class<F> fromType;
	private final Class<T> toType;
	private final T def;
	private final Map<F, T> map;

	/**
	 * constructor.
	 *
	 * @param fromType the type to be converted.
	 * @param toType   the type converted to
	 * @param from     from values
	 * @param to       to values
	 * @param def      default value
	 */
	public SwitchConverter(final Class<F> fromType, final Class<T> toType, final F[] from, final T[] to, final T def) {
		Assert.isTrue(from.length == to.length, "different array lengths");
		this.fromType = fromType;
		this.toType = toType;
		this.def = def;
		this.map = new HashMap<F, T>();
		for (int i = 0; i < from.length; i++) {
			map.put(from[i], to[i]);
		}
		Assert.isTrue(map.size() == from.length, "duplicate from key");
	}

	/**
	 * constructor.
	 *
	 * @param fromType the type to be converted.
	 * @param toType   the type converted to
	 * @param list     list with alternating from-to value pairs with a optional
	 *                 last default value
	 */
	@SuppressWarnings("unchecked")
	public SwitchConverter(final Class<F> fromType, final Class<T> toType, final Object... list) {
		this.fromType = fromType;
		this.toType = toType;
		this.map = new HashMap<F, T>();
		Object def = null;
		for (int i = 0; i < list.length; i += 2) {
			final Object o = list[i];
			if (i + 1 < list.length) {
				map.put((F) o, (T) list[i + 1]);
			} else {
				def = o;
			}
		}
		this.def = (T) def;
		Assert.isTrue(map.size() == list.length / 2, "duplicate from key");
	}

	/**
	 * constructor.
	 *
	 * @param fromType the type to be converted.
	 * @param toType   the type converted to
	 * @param map      the map of associated objects.
	 * @param def      default value
	 */
	public SwitchConverter(final Class<F> fromType, final Class<T> toType, final Map<F, T> map, final T def) {
		this.fromType = fromType;
		this.toType = toType;
		this.map = map;
		this.def = def;
	}

	@Override
	public T convert(final Object fromObject) {
		if (map.containsKey(fromObject)) {
			return map.get(fromObject);
		} else {
			return def;
		}
	}

	@Override
	public Object getFromType() {
		return fromType;
	}

	@Override
	public Object getToType() {
		return toType;
	}

}
