package com.bitctrl.lib.eclipse.databinding.observables;

import org.eclipse.core.databinding.observable.list.IListChangeListener;
import org.eclipse.core.databinding.observable.list.IObservableList;
import org.eclipse.core.databinding.observable.map.IMapChangeListener;
import org.eclipse.core.databinding.observable.map.IObservableMap;
import org.eclipse.core.databinding.observable.value.IObservableValue;
import org.eclipse.core.databinding.observable.value.IValueChangeListener;
import org.eclipse.swt.events.DisposeEvent;
import org.eclipse.swt.events.DisposeListener;
import org.eclipse.swt.widgets.Widget;

/**
 * Hilfsklasse zur Verwaltung von Observables.
 *
 * @author BitCtrl Systems GmbH, Enrico Schnepel
 * @version $Id$
 */
public class ObservablesUtil {
	/**
	 * Fgt Observables einen {@link IValueChangeListener} hinzu und entfernt ihn,
	 * sobald das Widget disposed wird. Disposed das Observable anschlieend.
	 *
	 * @see ObservablesUtil#addAndRemoveListener(Widget, IValueChangeListener,
	 *      IObservableList...)
	 * @param widget      das Widget
	 * @param listener    der zu hinzufgende Listener
	 * @param observables die Observables, denen der Listener hinzugefgt werden
	 *                    soll
	 */
	public static void addAndDisposeListener(final Widget widget, final IValueChangeListener listener,
			final IObservableValue... observables) {
		for (final IObservableValue observable : observables) {
			observable.addValueChangeListener(listener);
		}
		widget.addDisposeListener(new DisposeListener() {
			@Override
			public void widgetDisposed(final DisposeEvent e) {
				for (final IObservableValue observable : observables) {
					observable.removeValueChangeListener(listener);
					observable.dispose();
				}
			}
		});
	}

	/**
	 * Fgt Observables einen {@link IMapChangeListener} hinzu und entfernt ihn,
	 * sobald das Widget disposed wird. Disposed das Observable anschlieend.
	 *
	 * @see ObservablesUtil#addAndRemoveListener(Widget, IMapChangeListener,
	 *      IObservableList...)
	 * @param widget      das Widget
	 * @param listener    der zu hinzufgende Listener
	 * @param observables die Observables, denen der Listener hinzugefgt werden
	 *                    soll
	 */
	public static void addAndDisposeListener(final Widget widget, final IMapChangeListener listener,
			final IObservableMap... observables) {
		for (final IObservableMap observable : observables) {
			observable.addMapChangeListener(listener);
		}
		widget.addDisposeListener(new DisposeListener() {
			@Override
			public void widgetDisposed(final DisposeEvent e) {
				for (final IObservableMap observable : observables) {
					observable.removeMapChangeListener(listener);
					observable.dispose();
				}
			}
		});
	}

	/**
	 * Fgt Observables einen {@link IListChangeListener} hinzu und entfernt ihn,
	 * sobald das Widget disposed wird. Disposed das Observable anschlieend.
	 *
	 * @see ObservablesUtil#addAndRemoveListener(Widget, IListChangeListener,
	 *      IObservableList...)
	 * @param widget      das Widget
	 * @param listener    der zu hinzufgende Listener
	 * @param observables die Observables, denen der Listener hinzugefgt werden
	 *                    soll
	 */
	public static void addAndDisposeListener(final Widget widget, final IListChangeListener listener,
			final IObservableList... observables) {
		for (final IObservableList observable : observables) {
			observable.addListChangeListener(listener);
		}
		widget.addDisposeListener(new DisposeListener() {
			@Override
			public void widgetDisposed(final DisposeEvent e) {
				for (final IObservableList observable : observables) {
					observable.removeListChangeListener(listener);
					observable.dispose();
				}
			}
		});
	}

	/**
	 * Fgt Observables einen {@link IValueChangeListener} hinzu und entfernt ihn,
	 * sobald das Widget disposed wird. Das Observable wird nicht disposed.
	 *
	 * @see ObservablesUtil#addAndDisposeListener(Widget, IValueChangeListener,
	 *      IObservableList...)
	 * @param widget      das Widget
	 * @param listener    der zu hinzufgende Listener
	 * @param observables die Observables, denen der Listener hinzugefgt werden
	 *                    soll
	 */
	public static void addAndRemoveListener(final Widget widget, final IValueChangeListener listener,
			final IObservableValue... observables) {
		for (final IObservableValue observable : observables) {
			observable.addValueChangeListener(listener);
		}
		widget.addDisposeListener(new DisposeListener() {
			@Override
			public void widgetDisposed(final DisposeEvent e) {
				for (final IObservableValue observable : observables) {
					observable.removeValueChangeListener(listener);
				}
			}
		});
	}

	/**
	 * Fgt Observables einen {@link IMapChangeListener} hinzu und entfernt ihn,
	 * sobald das Widget disposed wird. Das Observable wird nicht disposed.
	 *
	 * @see ObservablesUtil#addAndDisposeListener(Widget, IMapChangeListener,
	 *      IObservableList...)
	 * @param widget      das Widget
	 * @param listener    der zu hinzufgende Listener
	 * @param observables die Observables, denen der Listener hinzugefgt werden
	 *                    soll
	 */
	public static void addAndRemoveListener(final Widget widget, final IMapChangeListener listener,
			final IObservableMap... observables) {
		for (final IObservableMap observable : observables) {
			observable.addMapChangeListener(listener);
		}
		widget.addDisposeListener(new DisposeListener() {
			@Override
			public void widgetDisposed(final DisposeEvent e) {
				for (final IObservableMap observable : observables) {
					observable.removeMapChangeListener(listener);
				}
			}
		});
	}

	/**
	 * Fgt Observables einen {@link IListChangeListener} hinzu und entfernt ihn,
	 * sobald das Widget disposed wird. Das Observable wird nicht disposed.
	 *
	 * @see ObservablesUtil#addAndDisposeListener(Widget, IListChangeListener,
	 *      IObservableList...)
	 * @param widget      das Widget
	 * @param listener    der zu hinzufgende Listener
	 * @param observables die Observables, denen der Listener hinzugefgt werden
	 *                    soll
	 */
	public static void addAndRemoveListener(final Widget widget, final IListChangeListener listener,
			final IObservableList... observables) {
		for (final IObservableList observable : observables) {
			observable.addListChangeListener(listener);
		}
		widget.addDisposeListener(new DisposeListener() {
			@Override
			public void widgetDisposed(final DisposeEvent e) {
				for (final IObservableList observable : observables) {
					observable.removeListChangeListener(listener);
				}
			}
		});
	}
}
