/*******************************************************************************
 * Copyright (c) 2008 Ketan Padegaonkar and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Ketan Padegaonkar - initial API and implementation
 *     Enrico Schnepel   - modifications for Running in Realm
 *******************************************************************************/
package com.bitctrl.lib.eclipse.databinding.realm;

import java.util.ArrayList;

import org.eclipse.core.databinding.observable.Realm;
import org.eclipse.swt.widgets.Display;

import com.bitctrl.lib.eclipse.databinding.realm.results.ArrayResult;
import com.bitctrl.lib.eclipse.databinding.realm.results.Result;
import com.bitctrl.lib.eclipse.databinding.realm.results.VoidResult;

/**
 * Performs operations in the Realm. If the {@link #run()} method of this class
 * is called from an non-Realm, the instance ensures that it runs in the Realm
 * by invoking {@link Realm#exec(Runnable)} with a extra sync, else it executes
 * in the Realm. All operations are blocking operations.
 *
 * @author Ketan Padegaonkar &lt;KetanPadegaonkar [at] gmail [dot] com&gt;
 * @author schnepel
 * @version $Id$
 */
public abstract class RealmRunnable implements Runnable {

	/** the display on which runnables must be executed. */
	protected final Realm realm;
	/**
	 * A flag to denote if the runnable should execute asynchronously.
	 */
	private final boolean async;

	/**
	 * Runs synchronously in the Realm.
	 *
	 * @param realm The Realm to be used.
	 */
	private RealmRunnable(final Realm realm) {
		this(realm, false);
	}

	/**
	 * A private contructor use to create this object.
	 *
	 * @param realm The Realm to use.
	 * @param async if the thread should run asynchronously or not.
	 * @see Display#syncExec(Runnable)
	 * @see Display#asyncExec(Runnable)
	 */
	private RealmRunnable(final Realm realm, final boolean async) {
		this.realm = realm;
		this.async = async;
	}

	/**
	 * This method is intelligent to execute in the Realm.
	 */
	@Override
	public void run() {
		if (realm == null) {
			return;
		}

		if (!realm.isCurrent()) {
			if (async) {
				realm.asyncExec(runnable(null));
			} else {
				synchronized (realm) {
					realm.exec(runnable(realm));
					try {
						realm.wait();
					} catch (final InterruptedException e) {
						e.printStackTrace();
					}
				}
			}
		} else {
			runnable(null).run();
		}
	}

	/**
	 * A runnable instance that is used internally.
	 *
	 * @return The runnable instance.
	 */
	private Runnable runnable(final Object syncronizedObject) {
		Runnable runnable;
		if (null != syncronizedObject) {
			runnable = new Runnable() {
				@Override
				public void run() {
					synchronized (syncronizedObject) {
						doRun();
						dispatchAllEvents();
						syncronizedObject.notifyAll();
					}
				}
			};
		} else {
			runnable = new Runnable() {
				@Override
				public void run() {
					doRun();
					dispatchAllEvents();
				}
			};
		}
		return runnable;
	}

	/**
	 * This dispatched events in the Display.
	 * <p>
	 * This must be called in the Realm only. This method does not execute in a
	 * syncexec/asyncexec block
	 * </p>
	 */
	private void dispatchAllEvents() {
		Display.getDefault().wake();
		// while (true)
		// if (!display.readAndDispatch())
		// break;
	}

	/**
	 * Performs the run in the Realm.
	 * <p>
	 * This MUST be invoked in the Realm.
	 * </p>
	 */
	protected abstract void doRun();

	/**
	 * Executes the {@code toExecute} on the Realm, and blocks the calling thread.
	 *
	 * @param           <T> the type of the result.
	 * @param toExecute the runnable to execute.
	 * @return the result of executing result on the Realm.
	 */
	public static <T> T syncExec(final Result<T> toExecute) {
		return syncExec(Realm.getDefault(), toExecute);
	}

	/**
	 * Executes the {@code toExecute} on the Realm, and blocks the calling thread.
	 *
	 * @param           <T> the type of the result.
	 * @param realm     the Realm on which toExecute must be executed.
	 * @param toExecute the runnable to execute.
	 * @return the object result of execution on the Realm.
	 */
	@SuppressWarnings("unchecked")
	public static <T> T syncExec(final Realm realm, final Result<T> toExecute) {
		final ArrayList<Object> arrayList = new ArrayList<Object>(2);
		new RealmRunnable(realm) {
			@Override
			protected void doRun() {
				try {
					arrayList.add(toExecute.run());
				} catch (final Throwable t) {
					arrayList.add(null);
					arrayList.add(t);
				}
			}
		}.run();
		if (2 == arrayList.size()) {
			throw new RuntimeException((Throwable) arrayList.get(1));
		}
		return (T) arrayList.get(0);
	}

	/**
	 * Executes the {@code toExecute} on the Realm thread, and blocks the calling
	 * thread.
	 *
	 * @param           <T> the type of the result.
	 * @param toExecute the runnable to execute.
	 * @return the object result of execution on the Realm.
	 */
	public static <T> T[] syncExec(final ArrayResult<T> toExecute) {
		return syncExec(Realm.getDefault(), toExecute);
	}

	/**
	 * Executes the {@code toExecute} on the Realm thread, and blocks the calling
	 * thread.
	 *
	 * @param           <T> the type of the result.
	 * @param realm     the Realm on which toExecute must be executed.
	 * @param toExecute the runnable to execute.
	 * @return the object result of execution on the Realm.
	 */
	public static <T> T[] syncExec(final Realm realm, final ArrayResult<T> toExecute) {
		final ArrayList<T[]> arrayList = new ArrayList<T[]>();
		new RealmRunnable(realm) {
			@Override
			protected void doRun() {
				final T[] run = toExecute.run();
				arrayList.add(run);
			}
		}.run();
		return arrayList.get(0);
	}

	/**
	 * Executes the {@code toExecute} on the Realm, and blocks the calling thread.
	 *
	 * @param toExecute the runnable to execute.
	 * @since 1.0
	 */
	public static void syncExec(final VoidResult toExecute) {
		syncExec(Realm.getDefault(), toExecute);
	}

	/**
	 * Executes the {@code toExecute} on the Realm, and blocks the calling thread.
	 *
	 * @param realm     the Realm on which toExecute must be executed.
	 * @param toExecute the runnable to execute.
	 */
	public static void syncExec(final Realm realm, final VoidResult toExecute) {
		new RealmRunnable(realm) {
			@Override
			protected void doRun() {
				toExecute.run();
			}
		}.run();
	}

	/**
	 * Executes the {@code toExecute} on the Realm asynchronously, and does not
	 * block the calling thread.
	 *
	 * @param toExecute the runnable to execute.
	 * @since 1.0
	 */
	public static void asyncExec(final VoidResult toExecute) {
		asyncExec(Realm.getDefault(), toExecute);
	}

	/**
	 * Executes the {@code toExecute} on the Realm asynchronously, and does not
	 * block the calling thread.
	 *
	 * @param realm     the Realm on which toExecute must be executed.
	 * @param toExecute the runnable to execute.
	 */
	public static void asyncExec(final Realm realm, final VoidResult toExecute) {
		new RealmRunnable(realm, true) {
			@Override
			protected void doRun() {
				toExecute.run();
			}
		}.run();
	}

}
