/*
 * Rahmenwerk-Plug-in "Baustellen- und Unfalleingabe"
 * Copyright (C) 2009 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package com.bitctrl.lib.eclipse.databinding.validator;

import org.eclipse.core.databinding.validation.IValidator;
import org.eclipse.core.runtime.Assert;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.swt.widgets.Control;
import org.eclipse.ui.forms.IMessageManager;

/**
 * Validator, der einen oder mehrere andere Validatoren kapselt und die
 * zurckgegebenen Fehler an einen Messagemanager weiterreicht.
 *
 * @author BitCtrl Systems GmbH, Andreas Meissner
 * @author BitCtrl Systems GmbH, Enrico Schnepel
 * @version $Id$
 */
public class MessageValidator implements IValidator {
	private IValidator[] wrapped = null;
	private final Object key;
	private final Control control;
	private final int invalidStateMask;

	private final Status2MessageManagerConnector connector;

	/**
	 * Der Konstruktor, der alle Stati auer {@link IStatus#OK} behandelt.
	 *
	 * @param key            der Key
	 * @param messageManager der MessageManager
	 * @param control        das dazugehrige Control zum dekorieren
	 * @param wrapped        {@link IValidator IValidatoren} die dekoriert werden
	 *                       sollen. kann entfallen, wenn stattdessen
	 *                       {@link #add(IValidator)} verwendet wird.
	 */
	public MessageValidator(final Object key, final IMessageManager messageManager, final Control control,
			final IValidator... wrapped) {
		this(key, messageManager, control, IStatus.INFO | IStatus.WARNING | IStatus.ERROR | IStatus.CANCEL, wrapped);
	}

	/**
	 * Der Konstruktor.
	 *
	 * @param key              der Key
	 * @param messageManager   der MessageManager
	 * @param control          das dazugehrige Control zum dekorieren
	 * @param invalidStateMask welche {@link IStatus} Werte sollen behandelt werden?
	 * @param wrapped          {@link IValidator IValidatoren} die dekoriert werden
	 *                         sollen. kann entfallen, wenn stattdessen
	 *                         {@link #add(IValidator)} verwendet wird.
	 */
	public MessageValidator(final Object key, final IMessageManager messageManager, final Control control,
			final int invalidStateMask, final IValidator... wrapped) {
		Assert.isNotNull(key, "key");
		Assert.isNotNull(wrapped, "wrapped");
		Assert.isTrue(null != control || null != messageManager, "control oder messageManager");
		Assert.isTrue(null == control || null != messageManager, "TODO: handle ControlDecoration");

		if (wrapped != null && wrapped.length > 0) {
			for (final IValidator w : wrapped) {
				Assert.isNotNull(w);
			}
			this.wrapped = wrapped;
		} else {
			this.wrapped = new IValidator[0];
		}

		this.key = key;
		this.control = control;
		this.invalidStateMask = invalidStateMask;
		if (messageManager != null) {
			connector = new Status2MessageManagerConnector(messageManager);
		} else {
			connector = null;
		}
	}

	@Override
	public IStatus validate(final Object value) {
		final IStatus merged = StatusUtils.mergeStatus(value, wrapped);
		if (null != connector) {
			connector.setMessage(merged, control);
		}
		return merged;
	}

	/**
	 * Fgt einen Validator zur Liste der zu prfenden Validatoren hinzu.
	 *
	 * @param wrapped der Validator.
	 */
	public void add(final IValidator wrapped) {
		Assert.isNotNull(wrapped);
		final int len = this.wrapped.length;
		final IValidator[] newAr = new IValidator[len + 1];
		System.arraycopy(this.wrapped, 0, newAr, 0, len);
		newAr[len] = wrapped;
		this.wrapped = newAr;
	}
}
