/*
 * Rahmenwerk-Plug-in "Baustellen- und Unfalleingabe"
 * Copyright (C) 2009 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package com.bitctrl.lib.eclipse.databinding.validator;

import org.eclipse.core.databinding.observable.Observables;
import org.eclipse.core.databinding.observable.value.IObservableValue;
import org.eclipse.core.databinding.validation.IValidator;
import org.eclipse.core.runtime.Assert;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;

/**
 * Abstrakter Validator, welcher ein Number-Objekt validiert.
 *
 * @author BitCtrl Systems GmbH, Andreas Meissner
 * @author BitCtrl Systems GmbH, Enrico Schnepel
 * @version $Id$
 */
public abstract class RangeValidator<T> implements IValidator {
	/**
	 * Spezifiert die Art des Intervalls.
	 *
	 * @author schnepel
	 */
	public enum IntervallType {
	Offen(true, true), LinksOffen(true, false), RechtsOffen(false, true), Geschlossen(false, false);

		private final boolean linksOffenFlag;
		private final boolean rechtsOffenFlag;

		private IntervallType(final boolean linksOffen, final boolean rechtsOffen) {
			this.linksOffenFlag = linksOffen;
			this.rechtsOffenFlag = rechtsOffen;
		}
	}

	private final IObservableValue minValue;
	private final IObservableValue maxValue;
	private final String errorText;
	private final Class<T> typeClass;
	private final IntervallType intervall;
	private final boolean nullAllowed;
	private int severity = IStatus.ERROR;

	/**
	 * Der Konstruktor.
	 *
	 * @param key       der Key
	 * @param minValue  der minimal gltige Wert
	 * @param maxValue  der maximal gltige Wert
	 * @param errorText der Fehlertext, welcher bei ungltiger Eingabe angegeben
	 *                  werden soll
	 */
	public RangeValidator(final Class<T> typeClass, final IntervallType intervall, final T minValue, final T maxValue,
			final boolean nullAllowed, final String errorText) {
		this(typeClass, intervall, null != minValue ? Observables.constantObservableValue(minValue, typeClass) : null,
				null != maxValue ? Observables.constantObservableValue(maxValue, typeClass) : null, nullAllowed,
				errorText);
	}

	/**
	 * Der Konstruktor.
	 *
	 * @param key               der Key
	 * @param minValue          der minimal gltige Wert als
	 *                          {@link IObservableValue}
	 * @param maxValue          der maximal gltige Wert als
	 *                          {@link IObservableValue}
	 * @param errorText         der Fehlertext, welcher bei ungltiger Eingabe
	 *                          angegeben werden soll
	 * @param controlDecoration die dazugehrige ControlDecoration
	 * @param messageManager    der MessageManager
	 */
	public RangeValidator(final Class<T> typeClass, final IntervallType intervall, final IObservableValue minValue,
			final IObservableValue maxValue, final boolean nullAllowed, final String errorText) {
		Assert.isNotNull(typeClass, "typeClass");
		Assert.isNotNull(intervall, "intervall");
		Assert.isTrue(null != minValue || null != maxValue, "minValue oder maxValue");
		Assert.isNotNull(errorText, "errorText");

		this.typeClass = typeClass;
		this.intervall = intervall;
		this.minValue = minValue;
		this.maxValue = maxValue;
		this.nullAllowed = nullAllowed;
		this.errorText = errorText;
	}

	@Override
	@SuppressWarnings("unchecked")
	public IStatus validate(final Object value) {
		boolean fehler = false;
		if (typeClass.isInstance(value)) {
			final T wert = typeClass.cast(value);
			if (null != minValue) {
				if (null != minValue.getValue()) {
					if (intervall.linksOffenFlag) {
						fehler = compare(wert, (T) minValue.getValue()) <= 0;
					} else {
						fehler = compare(wert, (T) minValue.getValue()) < 0;
					}
				} else {
					fehler = !nullAllowed;
				}
			}
			if (!fehler && null != maxValue) {
				if (null != maxValue.getValue()) {
					if (intervall.rechtsOffenFlag) {
						fehler = compare(wert, (T) maxValue.getValue()) >= 0;
					} else {
						fehler = compare(wert, (T) maxValue.getValue()) > 0;
					}
				} else {
					fehler = !nullAllowed;
				}
			}
		} else {
			fehler = !nullAllowed;
		}
		if (fehler) {
			return StatusUtils.failureMessage(severity, errorText);
		}
		return Status.OK_STATUS;
	}

	public final RangeValidator<T> setSeverity(final int severity) {
		this.severity = severity;
		return this;
	}

	/**
	 * Vergleichsfunktion. Muss berschrieben werden mit "return wertA - wertB;".
	 *
	 * @param wertA
	 * @param wertB
	 * @return
	 */
	protected abstract double compare(T wertA, T wertB);

	@Override
	public String toString() {
		final String links = intervall.linksOffenFlag ? "(" : "[";
		final String rechts = intervall.rechtsOffenFlag ? ")" : "]";
		return getClass().getName() + "[ type = " + typeClass.getName() + ", severity = " + severity + ", intervall = "
				+ links + minValue + ", " + maxValue + rechts + " ]";
	}
}
