package com.bitctrl.lib.eclipse.databinding.validator;

import java.util.Map;
import java.util.WeakHashMap;

import org.eclipse.core.databinding.observable.value.IValueChangeListener;
import org.eclipse.core.databinding.observable.value.ValueChangeEvent;
import org.eclipse.core.databinding.validation.IValidator;
import org.eclipse.core.databinding.validation.MultiValidator;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.jface.fieldassist.FieldDecorationRegistry;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.resource.JFaceResources;
import org.eclipse.jface.viewers.BaseLabelProvider;
import org.eclipse.jface.viewers.DecoratingLabelProvider;
import org.eclipse.jface.viewers.DecorationOverlayIcon;
import org.eclipse.jface.viewers.IBaseLabelProvider;
import org.eclipse.jface.viewers.IDecoration;
import org.eclipse.jface.viewers.ILabelDecorator;
import org.eclipse.jface.viewers.ILabelProvider;
import org.eclipse.jface.viewers.LabelProviderChangedEvent;
import org.eclipse.swt.graphics.Image;

import com.bitctrl.lib.eclipse.resources.WidgetImageRegistry;

/**
 * Dekoriert die Eintrge in einer SWT List oder Table mit einem Icon passend
 * zum validierten {@link IStatus}.
 *
 * @author BitCtrl Systems GmbH, Enrico Schnepel
 * @version $Id: ValidationDecoratorLabelProvider.java 21700 2010-02-23
 *          12:24:32Z schnepel $
 */
public class ValidationDecoratorLabelProvider extends DecoratingLabelProvider implements IBaseLabelProvider {

	private static class Decorator extends BaseLabelProvider implements ILabelDecorator {
		static {
			/* initialize JFace image registry */
			FieldDecorationRegistry.getDefault();
		}
		private static final String IMG_DEC_FIELD_ERROR = "org.eclipse.jface.fieldassist.IMG_DEC_FIELD_ERROR";

		private static final String IMG_DEC_FIELD_WARNING = "org.eclipse.jface.fieldassist.IMG_DEC_FIELD_WARNING"; //$NON-NLS-1$

		private static final String IMG_DEC_FIELD_INFO = "org.eclipse.jface.fieldassist.IMG_DEC_FIELD_INFO"; //$NON-NLS-1$

		private final Map<Object, MultiValidator> valMap = new WeakHashMap<Object, MultiValidator>();

		private final IValidator[] validators;
		private final ILabelProvider provider;

		public Decorator(final ILabelProvider provider, final IValidator... validators) {
			this.provider = provider;
			this.validators = validators;
		}

		@Override
		public Image decorateImage(final Image image, final Object element) {
			final MultiValidator iValidator = getValidator(element);
			final IStatus iStatus = (IStatus) iValidator.getValidationStatus().getValue();
			String decoImage;
			switch (iStatus.getSeverity()) {
			case IStatus.ERROR:
				decoImage = IMG_DEC_FIELD_ERROR;
				break;
			case IStatus.WARNING:
				decoImage = IMG_DEC_FIELD_WARNING;
				break;
			case IStatus.INFO:
				decoImage = IMG_DEC_FIELD_INFO;
				break;
			default:
				return image;
			}
			final ImageDescriptor decoImageDesc = JFaceResources.getImageRegistry().getDescriptor(decoImage);
			return WidgetImageRegistry.getInstance().getImage(image == null ? decoImageDesc
					: new DecorationOverlayIcon(image, decoImageDesc, IDecoration.TOP_RIGHT) {
					});
		}

		@Override
		public String decorateText(final String text, final Object element) {
			return null;
		}

		private MultiValidator getValidator(final Object element) {
			MultiValidator iValidator = valMap.get(element);
			if (null == iValidator) {
				iValidator = new MultiValidator() {
					@Override
					protected IStatus validate() {
						return StatusUtils.mergeStatus(element, validators);
					}
				};
				valMap.put(element, iValidator);
				iValidator.getValidationStatus().addValueChangeListener(new IValueChangeListener() {

					@Override
					public void handleValueChange(final ValueChangeEvent event) {
						fireLabelProviderChanged(new LabelProviderChangedEvent(provider, element));
					}
				});
			}
			return iValidator;
		}
	}

	public ValidationDecoratorLabelProvider(final ILabelProvider provider, final IValidator... validators) {
		super(provider, new Decorator(provider, validators));
	}
}
