package com.bitctrl.lib.eclipse.databinding.widgets;

import org.eclipse.core.databinding.DataBindingContext;
import org.eclipse.core.databinding.observable.list.IObservableList;
import org.eclipse.core.databinding.observable.value.IObservableValue;
import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;

import com.bitctrl.lib.eclipse.resources.WidgetImageRegistry;

public abstract class AbstractListModificationComposite extends Composite {

	private final WidgetImageRegistry imageRegistry;

	private IObservableList elementList;
	private IObservableValue currentElement;

	private DataBindingContext dbc = null;

	/**
	 * The constructor. The {@link #init()} method needs to be called at the end of
	 * the overriding constructor.
	 *
	 * @param parent         the parent composite
	 * @param style          style for the buttons | the orientation for the
	 *                       component ( {@link SWT#HORIZONTAL} /
	 *                       {@link SWT#VERTICAL} )
	 * @param elementList    the observable list of elements, might be
	 *                       <code>null</code> but expects a later assignment (see:
	 *                       {@link #setElementList(IObservableList)})
	 * @param currentElement the current selected element, might be
	 *                       <code>null</code> but expects a later assignment (see:
	 *                       {@link #setCurrentElement(IObservableValue)})
	 * @param focusElement   an control to be focused on button click or null
	 */
	protected AbstractListModificationComposite(final Composite parent, final int style,
			final IObservableList elementList, final IObservableValue currentElement, final Control focusElement) {
		super(parent, style & ((SWT.VERTICAL | SWT.VERTICAL) ^ -1));

		this.elementList = elementList;
		this.currentElement = currentElement;

		final FillLayout layout = new FillLayout(style & (SWT.VERTICAL | SWT.VERTICAL));
		layout.marginHeight = 0;
		layout.marginWidth = 0;
		layout.spacing = 5;
		setLayout(layout);

		imageRegistry = WidgetImageRegistry.getInstance(this);
	}

	/**
	 * Needs to be called at the end of the overriding constructor.
	 */
	protected final void init() {
		if (elementList != null && currentElement != null) {
			dbc = new DataBindingContext();
			setupBindings(dbc);
		}
	}

	/**
	 * Sets the observable value and list.
	 *
	 * @param elementList    the observable list of elements
	 * @param currentElement the current selected element
	 */
	public final void setObservables(final IObservableList elementList, final IObservableValue currentElement) {
		unsetBindings();
		this.elementList = elementList;
		this.currentElement = currentElement;
		init();
	}

	/**
	 * Returns the image registry.
	 *
	 * @return the imageRegistry
	 */
	public WidgetImageRegistry getImageRegistry() {
		return imageRegistry;
	}

	/**
	 * Returns the observable element list.
	 *
	 * @return the observable element list
	 */
	public IObservableList getElementList() {
		return elementList;
	}

	/**
	 * Returns the current selected element.
	 *
	 * @return the current selected element
	 */
	public IObservableValue getCurrentElement() {
		return currentElement;
	}

	/**
	 * Used to setup some bindings which depend on {@link #getElementList()} or
	 * {@link #getCurrentElement()}.
	 */
	protected abstract void setupBindings(DataBindingContext dbc);

	private void unsetBindings() {
		if (null != dbc) {
			dbc.dispose();
			dbc = null;
		}
	}

	@Override
	public void dispose() {
		unsetBindings();
		super.dispose();
	}
}
