package com.bitctrl.lib.eclipse.databinding.widgets;

import org.eclipse.core.databinding.DataBindingContext;
import org.eclipse.core.databinding.UpdateValueStrategy;
import org.eclipse.core.databinding.observable.Observables;
import org.eclipse.core.databinding.observable.Realm;
import org.eclipse.core.databinding.observable.value.ComputedValue;
import org.eclipse.core.databinding.observable.value.IObservableValue;
import org.eclipse.jface.databinding.swt.SWTObservables;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.MouseAdapter;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;

import com.bitctrl.lib.eclipse.util.InternalEclipseImageProvider;

/**
 * Helper to create a button which resets an observable value to a given
 * default.
 *
 * @author schnepel
 */
public class ObservableClearButton {
	private ObservableClearButton() {
	}

	private static UpdateValueStrategy never = new UpdateValueStrategy(UpdateValueStrategy.POLICY_NEVER);
	private static UpdateValueStrategy onRequest = new UpdateValueStrategy(UpdateValueStrategy.POLICY_ON_REQUEST);

	public static Label create(final Composite parent, final int style, final IObservableValue observableValue,
			final Object[] defaultValues) {
		return create(parent, style, new DataBindingContext(observableValue.getRealm()), observableValue,
				defaultValues);
	}

	public static Label create(final Composite parent, final int style, final DataBindingContext dbc,
			final IObservableValue observableValue, final Object[] defaultValues) {
		return create(parent, style, dbc, observableValue,
				constantObservableValues(observableValue.getRealm(),
						null != defaultValues ? defaultValues : new Object[] { observableValue.getValue() },
						observableValue.getValueType()));
	}

	public static Label create(final Composite parent, final int style, final IObservableValue observableValue,
			final IObservableValue[] defaultValues) {
		return create(parent, style, new DataBindingContext(observableValue.getRealm()), observableValue,
				defaultValues);
	}

	public static Label create(final Composite parent, final int style, final DataBindingContext dbc,
			final IObservableValue observableValue, final IObservableValue defaultValue,
			final IObservableValue enabledModel, final Runnable valueSetter) {
		return create(parent, style, dbc, observableValue, new IObservableValue[] { defaultValue }, enabledModel,
				valueSetter);
	}

	public static Label create(final Composite parent, final int style, final DataBindingContext dbc,
			final IObservableValue observableValue, final IObservableValue defaultValue) {
		return create(parent, style, dbc, observableValue, new IObservableValue[] { defaultValue });
	}

	public static Label create(final Composite parent, final int style, final DataBindingContext dbc,
			final IObservableValue observableValue, final IObservableValue[] defaultValues) {
		final ComputedValue enabledModel = new ComputedValue() {
			@Override
			protected Object calculate() {
				if (defaultValues.length > 1) {
					return true;
				}
				final Object v = observableValue.getValue();
				final Object dv = defaultValues[0].getValue();
				return v != dv || v != null && !v.equals(dv);
			}
		};
		return create(parent, style, dbc, observableValue, defaultValues, enabledModel, null);
	}

	public static Label create(final Composite parent, final int style, final DataBindingContext dbc,
			final IObservableValue observableValue, final IObservableValue[] defaultValues,
			final IObservableValue enabledModel, final Runnable valueSetter) {
		final Label label = new Label(parent, SWT.NONE);
		label.setImage(InternalEclipseImageProvider.getClearImage());
		label.addMouseListener(new MouseAdapter() {
			@Override
			public synchronized void mouseUp(final org.eclipse.swt.events.MouseEvent e) {
				observableValue.getRealm().exec(new Runnable() {
					@Override
					public void run() {
						observableValue.setValue(calculate());
					}
				});
			}

			final int dvLength = defaultValues.length;
			int lastIndex = 0;

			private Object calculate() {
				if (dvLength > 1) {
					final Object v = observableValue.getValue();
					final Object dv = defaultValues[lastIndex].getValue();
					if (v == dv || v != null && v.equals(dv)) {
						lastIndex = (lastIndex + 1) % dvLength;
						return defaultValues[lastIndex].getValue();

					}
				}
				if (dvLength >= 1 && null != defaultValues[0]) {
					return defaultValues[0].getValue();
				} else {
					return null;
				}
			}
		});
		dbc.bindValue(SWTObservables.observeVisible(label), enabledModel, never, null);
		return label;
	}

	private static IObservableValue[] constantObservableValues(final Realm realm, final Object[] defaultValues,
			final Object valueType) {
		if (null == defaultValues) {
			return new IObservableValue[0];
		}
		final IObservableValue[] ret = new IObservableValue[defaultValues.length];
		int i = 0;
		for (final Object value : defaultValues) {
			ret[i++] = Observables.constantObservableValue(value, valueType);
		}
		return ret;
	}
}
