package com.bitctrl.lib.eclipse.databinding.widgets;

import org.eclipse.core.databinding.DataBindingContext;
import org.eclipse.core.databinding.observable.list.IObservableList;
import org.eclipse.core.databinding.observable.value.ComputedValue;
import org.eclipse.core.databinding.observable.value.IObservableValue;
import org.eclipse.core.databinding.observable.value.WritableValue;
import org.eclipse.jface.databinding.swt.SWTObservables;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.MouseAdapter;
import org.eclipse.swt.events.MouseEvent;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Label;

import com.bitctrl.lib.eclipse.BitCtrlEclipseCommonPlugin;
import com.bitctrl.lib.eclipse.Images;

/**
 * Manages automatically an observable list of instances. A "+" and a "-" button
 * are put into a composite handling the addition to / removal from the list.
 * This class needs to be extended to implement {@link #createNewObject()}.
 *
 * @param <T> the object type instances of are managed
 *
 * @author schnepel
 */
public abstract class PlusMinus<T> extends AbstractListModificationComposite {

	protected final IObservableValue deletionAllowed = new WritableValue(true, Boolean.TYPE);
	protected final IObservableValue additionAllowed = new WritableValue(true, Boolean.TYPE);

	private final Label plus;
	private final Label minus;

	/**
	 * The constructor.
	 *
	 * @param parent       the parent composite
	 * @param style        style for the buttons | the orientation for the component
	 *                     ( {@link SWT#HORIZONTAL} / {@link SWT#VERTICAL} )
	 * @param focusElement an control to be focused on button click or null
	 */
	public PlusMinus(final Composite parent, final int style, final Control focusElement) {
		this(parent, style, null, null, focusElement);
	}

	/**
	 * The constructor.
	 *
	 * @param parent         the parent composite
	 * @param style          style for the buttons | the orientation for the
	 *                       component ( {@link SWT#HORIZONTAL} /
	 *                       {@link SWT#VERTICAL} )
	 * @param elementList    the observable list of elements, might be
	 *                       <code>null</code> but expects a later assignment (see:
	 *                       {@link #setElementList(IObservableList)})
	 * @param currentElement the current selected element which get possibly removed
	 *                       and it will be set on addition, might be
	 *                       <code>null</code> but expects a later assignment (see:
	 *                       {@link #setCurrentElement(IObservableValue)})
	 * @param focusElement   an control to be focused on button click or null
	 */
	public PlusMinus(final Composite parent, final int style, final IObservableList elementList,
			final IObservableValue currentElement, final Control focusElement) {
		super(parent, style, elementList, currentElement, focusElement);

		plus = new Label(this, style);
		plus.setImage(getImageRegistry()
				.getImage(BitCtrlEclipseCommonPlugin.getDefault().getImageDescriptor(Images.IMG_ETOOL_LIST_ADD)));
		plus.addMouseListener(new MouseAdapter() {
			@Override
			public void mouseUp(final MouseEvent e) {
				if (getElementList() == null || getCurrentElement() == null) {
					return;
				}
				final T newT = createNewObject();
				getElementList().add(newT);
				getCurrentElement().setValue(newT);
				if (null != focusElement) {
					focusElement.setFocus();
				}
			}
		});
		plus.setToolTipText("Fgt ein neues Element hinzu");

		minus = new Label(this, style);
		minus.setImage(getImageRegistry()
				.getImage(BitCtrlEclipseCommonPlugin.getDefault().getImageDescriptor(Images.IMG_ETOOL_LIST_REMOVE)));
		minus.addMouseListener(new MouseAdapter() {
			@Override
			public void mouseUp(final MouseEvent e) {
				if (getElementList() == null || getCurrentElement() == null) {
					return;
				}
				final int idx = getElementList().indexOf(getCurrentElement().getValue());
				getElementList().remove(idx);
				if (null != focusElement) {
					focusElement.setFocus();
				}
				if (!getElementList().isEmpty()) {
					getCurrentElement().setValue(getElementList().get(getElementList().size() == idx ? idx - 1 : idx));
				}
			}
		});
		minus.setToolTipText("Lscht das ausgewhlte Element.");

		init();
	}

	@Override
	public void setupBindings(final DataBindingContext dbc) {
		dbc.bindValue(SWTObservables.observeEnabled(minus), new ComputedValue() {
			@Override
			protected Object calculate() {
				final Boolean value = (Boolean) deletionAllowed.getValue();
				return value && !getElementList().isEmpty();
			}
		});
		dbc.bindValue(SWTObservables.observeEnabled(plus), additionAllowed);
	}

	/**
	 * Creates a new object of type <T>. Needs to be implemented!
	 *
	 * @return a new instance of type <T>
	 */
	protected abstract T createNewObject();

	/**
	 * Setting the deletion allowed flag controls the enabled state of the "-"
	 * button. Deletion from an empty is never allowed.
	 *
	 * @return an IObservableValue
	 */
	public IObservableValue getDeletionAllowed() {
		return deletionAllowed;
	}

	/**
	 * Setter for the deletion allowed flag.
	 *
	 * @param deletionAllowed the new value
	 */
	public void setDeletionAllowed(final boolean deletionAllowed) {
		this.deletionAllowed.setValue(deletionAllowed);
	}

	/**
	 * Setting the addition allowed flag controls the enabled state of the "+"
	 * button.
	 *
	 * @return an IObservableValue
	 */
	public IObservableValue getAdditionAllowed() {
		return additionAllowed;
	}

	/**
	 * Setter for the deletion allowed flag.
	 *
	 * @param deletionAllowed the new value
	 */
	public void setAdditionAllowed(final boolean additionAllowed) {
		this.additionAllowed.setValue(additionAllowed);
	}
}
