package com.bitctrl.lib.eclipse.databinding.widgets;

import org.eclipse.core.databinding.DataBindingContext;
import org.eclipse.core.databinding.observable.list.IObservableList;
import org.eclipse.core.databinding.observable.value.ComputedValue;
import org.eclipse.core.databinding.observable.value.IObservableValue;
import org.eclipse.core.databinding.observable.value.WritableValue;
import org.eclipse.jface.databinding.swt.SWTObservables;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.MouseAdapter;
import org.eclipse.swt.events.MouseEvent;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Label;

import com.bitctrl.lib.eclipse.BitCtrlEclipseCommonPlugin;
import com.bitctrl.lib.eclipse.Images;

/**
 * Manages automatically an observable list of instances. An "Up" and a "Down"
 * button are put into a composite handling the relocation of a selected element
 * of the list.
 *
 * @author Andreas Meissner
 */
public class UpDown extends AbstractListModificationComposite {

	protected final IObservableValue upAllowed = new WritableValue(true, Boolean.TYPE);
	protected final IObservableValue downAllowed = new WritableValue(true, Boolean.TYPE);

	private final Label up;
	private final Label down;

	/**
	 * The constructor.
	 *
	 * @param parent       the parent composite
	 * @param style        style for the buttons | the orientation for the component
	 *                     ( {@link SWT#HORIZONTAL} / {@link SWT#VERTICAL} )
	 * @param focusElement an control to be focused on button click or null
	 */
	public UpDown(final Composite parent, final int style, final Control focusElement) {
		this(parent, style, null, null, focusElement);
	}

	/**
	 * The constructor.
	 *
	 * @param parent         the parent composite
	 * @param style          style for the buttons | the orientation for the
	 *                       component ( {@link SWT#HORIZONTAL} /
	 *                       {@link SWT#VERTICAL} )
	 * @param elementList    the observable list of elements, might be
	 *                       <code>null</code> but expects a later assignment (see:
	 *                       {@link #setObservables(IObservableList, IObservableValue)})
	 * @param currentElement the current selected element which get possibly moved
	 *                       up or down in the list, might be <code>null</code> but
	 *                       expects a later assignment (see:
	 *                       {@link #setObservables(IObservableList, IObservableValue)})
	 * @param focusElement   an control to be focused on button click or null
	 */
	public UpDown(final Composite parent, final int style, final IObservableList elementList,
			final IObservableValue currentElement, final Control focusElement) {
		super(parent, style, elementList, currentElement, focusElement);

		up = new Label(this, style);
		up.setImage(getImageRegistry()
				.getImage(BitCtrlEclipseCommonPlugin.getDefault().getImageDescriptor(Images.IMG_ETOOL_LIST_UP)));
		up.addMouseListener(new MouseAdapter() {
			@Override
			public void mouseUp(final MouseEvent e) {
				if (getElementList() == null || getCurrentElement() == null) {
					return;
				}

				final Object selectedElement = getCurrentElement().getValue();
				if (selectedElement == null) {
					return;
				}

				final int oldIndex = getElementList().indexOf(selectedElement);
				if (oldIndex <= 0) {
					return;
				}
				final int newIndex = oldIndex - 1;
				getElementList().move(oldIndex, newIndex);
				if (null != focusElement) {
					focusElement.setFocus();
				}
			}
		});
		up.setToolTipText("Verschiebt ein ausgewhltes Element nach oben");

		down = new Label(this, style);
		down.setImage(getImageRegistry()
				.getImage(BitCtrlEclipseCommonPlugin.getDefault().getImageDescriptor(Images.IMG_ETOOL_LIST_DOWN)));
		down.addMouseListener(new MouseAdapter() {
			@Override
			public void mouseUp(final MouseEvent e) {
				if (getElementList() == null || getCurrentElement() == null) {
					return;
				}

				final Object selectedElement = getCurrentElement().getValue();
				if (selectedElement == null) {
					return;
				}

				final int oldIndex = getElementList().indexOf(selectedElement);
				if (oldIndex == -1 || oldIndex >= getElementList().size() - 1) {
					return;
				}
				final int newIndex = oldIndex + 1;

				getElementList().move(oldIndex, newIndex);
				if (null != focusElement) {
					focusElement.setFocus();
				}
			}
		});
		down.setToolTipText("Verschiebt ein ausgewhltes Element nach unten");

		init();
	}

	@Override
	protected void setupBindings(final DataBindingContext dbc) {
		dbc.bindValue(SWTObservables.observeEnabled(up), new ComputedValue() {
			@Override
			protected Object calculate() {
				final Boolean value = (Boolean) upAllowed.getValue();
				final Object selectedElement = getCurrentElement().getValue();
				if (value && selectedElement != null && getElementList().indexOf(selectedElement) != 0) {
					return true;
				}
				return false;
			}
		});
		dbc.bindValue(SWTObservables.observeEnabled(down), new ComputedValue() {
			@Override
			protected Object calculate() {
				final Boolean value = (Boolean) downAllowed.getValue();
				final Object selectedElement = getCurrentElement().getValue();
				if (value && selectedElement != null
						&& getElementList().indexOf(selectedElement) != getElementList().size() - 1) {
					return true;
				}
				return false;
			}
		});
	}

	/**
	 * Setting the up allowed flag controls the enabled state of the button. Moving
	 * up of the first element is never allowed.
	 *
	 * @return an IObservableValue
	 */
	public IObservableValue getUpAllowed() {
		return upAllowed;
	}

	/**
	 * Setter for the up allowed flag.
	 *
	 * @param upAllowed the new value
	 */
	public void setUpAllowed(final boolean upAllowed) {
		this.upAllowed.setValue(upAllowed);
	}

	/**
	 * Setting the down allowed flag controls the enabled state of the button.
	 * Moving down the last element is never allowed.
	 *
	 * @return an IObservableValue
	 */
	public IObservableValue getDownAllowed() {
		return downAllowed;
	}

	/**
	 * Setter for the down allowed flag.
	 *
	 * @param downAllowed the new value
	 */
	public void setDownAllowed(final boolean downAllowed) {
		this.downAllowed.setValue(downAllowed);
	}
}
