package com.bitctrl.lib.eclipse.gef;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import javax.swing.event.EventListenerList;

import org.eclipse.draw2d.FreeformLayeredPane;
import org.eclipse.draw2d.IFigure;
import org.eclipse.draw2d.LayeredPane;
import org.eclipse.gef.editparts.ScalableFreeformRootEditPart;

import com.bitctrl.lib.eclipse.draw2d.ITransparentFigure;

/**
 * Erweitert das Root Edit Part um die Fähigkeit von benutzerdefinierten
 * Layern. Diese Layer lassen sich sichtbar und unsichtbar schalten sowie deren
 * Deckkraft und Reihenfolge ändern.
 *
 * @author BitCtrl Systems GmbH, Falko Schumann
 * @version $Id$
 */
public class LayeredScalableFreeformRootEditPart
		extends ScalableFreeformRootEditPart implements ILayerManager {

	/** Konstante für die benutzerdefinierten Layer. */
	public static final String CUSTOM_LAYERS = "CUSTOM_LAYERS"; //$NON-NLS-1$

	private final EventListenerList listenerList = new EventListenerList();
	private final Map<Object, List<LayerListener>> layerListenerList = new HashMap<Object, List<LayerListener>>();

	private final List<Object> layerKeys = new LinkedList<Object>();
	private FreeformLayeredPane customLayers;

	@Override
	public void activate() {
		super.activate();

		getViewer().setProperty(ILayerManager.class.getName(), this);
	}

	@Override
	public void deactivate() {
		getViewer().setProperty(ILayerManager.class.getName(), null);

		super.deactivate();
	}

	/**
	 * Gibt die vorhandenen benutzerdefinierten Layer zurück.
	 *
	 * @return die benutzerdefinierten Layer.
	 */
	protected LayeredPane getCustomLayers() {
		if (customLayers == null) {
			customLayers = new FreeformLayeredPane();
		}
		return customLayers;
	}

	/**
	 * Erzeugt im druckbaren Layer den Container für die benutzerdefinierten
	 * Layer.
	 */
	@Override
	protected void createLayers(final LayeredPane layeredPane) {
		super.createLayers(layeredPane);
		getPrintableLayers().add(getCustomLayers(), CUSTOM_LAYERS);
	}

	@Override
	public List<?> getCustomLayerKeys() {
		return layerKeys;
	}

	/**
	 * Prüft zusätzlich auf einen benutzerdefinierten Layer.
	 */
	@Override
	public IFigure getLayer(final Object key) {
		final IFigure layer = getCustomLayers().getLayer(key);
		if (layer != null) {
			return layer;
		}

		return super.getLayer(key);
	}

	@Override
	public void addCustomLayer(final Object key, final IFigure layer) {
		getCustomLayers().add(layer, key);
		layerKeys.add(key);

		fireLayerEvent(LayerEvent.ADDED, key);
	}

	@Override
	public void addCustomLayer(final Object key, final IFigure layer,
			final int index) {
		getCustomLayers().add(layer, key, index);
		layerKeys.add(index, key);

		fireLayerEvent(LayerEvent.ADDED, key);
	}

	@Override
	public void removeCustomLayer(final Object key) {
		getCustomLayers().removeLayer(key);
		layerKeys.remove(key);

		fireLayerEvent(LayerEvent.REMOVED, key);
	}

	@Override
	public boolean isVisble(final Object key) {
		return getLayer(key).isVisible();
	}

	@Override
	public void setVisble(final Object key, final boolean visible) {
		getLayer(key).setVisible(visible);

		fireLayerEvent(LayerEvent.VISIBLE, key);
	}

	@Override
	public boolean canMove(final Object key, final Direction direction) {
		final int layerCount = getCustomLayerKeys().size();
		final int index = getCustomLayerKeys().indexOf(key);
		switch (direction) {
		case Up:
		case Top:
			return index > 0;
		case Down:
		case Bottom:
			return index < layerCount - 1;
		}

		throw new IllegalStateException("Unkown direction: " + direction); //$NON-NLS-1$
	}

	@Override
	public void move(final Object key, final Direction direction) {
		final int layerCount = getCustomLayerKeys().size();
		final int index = getCustomLayerKeys().indexOf(key);
		final IFigure layer = getLayer(key);
		switch (direction) {
		case Up:
			if (canMove(key, direction)) {
				Collections.swap(getCustomLayerKeys(), index, index - 1);
				getCustomLayers().removeLayer(key);
				getCustomLayers().add(layer, key, layerCount - index);
			}
			break;
		case Top:
			if (canMove(key, direction)) {
				Collections.swap(getCustomLayerKeys(), index, 0);
				getCustomLayers().removeLayer(key);
				getCustomLayers().add(layer, key, layerCount - 1);
			}
			break;
		case Down:
			if (canMove(key, direction)) {
				Collections.swap(getCustomLayerKeys(), index, index + 1);
				getCustomLayers().removeLayer(key);
				getCustomLayers().add(layer, key, layerCount - index - 2);
			}
			break;
		case Bottom:
			if (canMove(key, direction)) {
				Collections.swap(getCustomLayerKeys(), index, layerCount - 1);
				getCustomLayers().removeLayer(key);
				getCustomLayers().add(layer, key, 0);
			}
			break;
		}

		fireLayerEvent(LayerEvent.MOVED, key);
	}

	@Override
	public int getAlpha(final Object key) {
		final IFigure layer = getLayer(key);
		if (layer instanceof ITransparentFigure) {
			return ((ITransparentFigure) layer).getAlpha();
		}
		return -1;
	}

	@Override
	public void setAlpha(final Object key, final int alpha) {
		final IFigure layer = getLayer(key);
		if (layer instanceof ITransparentFigure) {
			((ITransparentFigure) layer).setAlpha(alpha);
		}

		fireLayerEvent(LayerEvent.ALPHA, key);
	}

	@Override
	public void addLayerListener(final LayerListener l) {
		listenerList.add(LayerListener.class, l);
	}

	@Override
	public void addLayerListener(final LayerListener l, final Object key) {
		if (!layerListenerList.containsKey(key)) {
			layerListenerList.put(key, new ArrayList<LayerListener>());
		}
		layerListenerList.get(key).add(l);
	}

	@Override
	public void removeLayerListener(final LayerListener l) {
		listenerList.remove(LayerListener.class, l);
	}

	@Override
	public void removeLayerListener(final LayerListener l, final Object key) {
		layerListenerList.get(key).remove(l);
	}

	protected synchronized void fireLayerEvent(final int eventType,
			final Object key) {
		final LayerEvent e = new LayerEvent(this, eventType, key);
		e.setVisible(isVisble(key));
		e.setAlpha(getAlpha(key));
		for (final LayerListener l : listenerList
				.getListeners(LayerListener.class)) {
			l.layerChanged(e);
		}

		if (layerListenerList.containsKey(key)) {
			for (final LayerListener l : layerListenerList.get(key)) {
				l.layerChanged(e);
			}
		}
	}

}
