/*
 * Copyright 2018 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.bsvrz.dav.dav.
 *
 * de.bsvrz.dav.dav is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.dav.dav is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.dav.dav.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Martin-Luther-Straße 14
 * 52062 Aachen, Germany
 * phone: +49 241 4090 436
 * mail: <info@kappich.de>
 */

package de.bsvrz.dav.dav.main;

import de.bsvrz.dav.daf.communication.lowLevel.ConnectionInterface;
import de.bsvrz.dav.daf.communication.lowLevel.LowLevelCommunication;
import de.bsvrz.dav.daf.communication.lowLevel.ServerConnectionInterface;
import de.bsvrz.dav.daf.communication.tcpCommunication.TCP_IP_ServerCommunication;
import de.bsvrz.dav.daf.main.ConnectionException;
import de.bsvrz.dav.dav.communication.appProtocol.T_A_HighLevelCommunication;
import de.bsvrz.sys.funclib.debug.Debug;

import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

/**
 * Verwaltung für ausgehende Datenverteilerverbindungen
 *
 * @author Kappich Systemberatung
 * @version $Revision$
 */
public class OutgoingApplicationConnections extends AbstractConnections {

	private static final Debug _debug = Debug.getLogger();

	/**
	 * Threadpool, der nicht etablierte Verbindungen aufbaut
	 */
	private final ScheduledExecutorService _applicationReconnectService = Executors.newScheduledThreadPool(1);

	private final HighLevelApplicationManager _applicationManager;
	/**
	 * Verwaltung aller offenen Verbindungen
	 */
	private final ApplicationConnectionMap _connections;

	/**
	 * Factory zum Aufbauen von Verbindungen (z.B. {@link TCP_IP_ServerCommunication})
	 */
	private final ServerConnectionInterface _serverConnection;

	/**
	 * Wartezeit in ms, bevor versucht wird, eine abgebrochene Verbindung wiederherzustellen
	 */
	private final int _reconnectionDelay;
	
	/**
	 * Ist die Konfiguration verfügbar?
	 */
	private boolean _configurationAvailable = false;

	/**
	 * Konstruktor
	 * @param serverConnection Factory zum Aufbauen von Verbindungen (z.B. {@link TCP_IP_ServerCommunication})
	 * @param reconnectionDelay Wiederverbindungswartezeit nach Verbindungsabbruch
	 * @param serverDavParameters  Server-Parameter
	 * @param lowLevelConnectionsManager Low-Level-Verbindungsverwaltung
	 * @param applicationManager High-Level-Verwaltung für Dav-Dav-Verbindungen (Anmeldelisten usw.)
	 * @param connectionMap Verwaltung für alle offenen Verbindungen pro Datenverteiler
	 */
	public OutgoingApplicationConnections(final ServerConnectionInterface serverConnection, final int reconnectionDelay, final ServerDavParameters serverDavParameters, final LowLevelConnectionsManagerInterface lowLevelConnectionsManager, final HighLevelApplicationManager applicationManager, final ApplicationConnectionMap connectionMap) {
		super(serverDavParameters, lowLevelConnectionsManager);
		_serverConnection = serverConnection;
		_reconnectionDelay = reconnectionDelay;
		_applicationManager = applicationManager;
		_connections = connectionMap;
	}

	public void start() {
		for(final ApplicationInfo info : _connections.getAllInfos()) {
			if(!connectToApplication(info, 1000)) {
				scheduleApplicationConnect(info, _reconnectionDelay, TimeUnit.MILLISECONDS);
			}
		}
	}

	/**
	 * Trigger, dass die Konfiguration verfügbar ist.
	 */
	public void setConfigurationAvailable() {
		_configurationAvailable = true;
	}


	private T_A_HighLevelCommunication startApplicationConnection(
			final ApplicationInfo applicationInfo)
			throws ConnectionException {
		final ConnectionInterface connection = _serverConnection.getPlainConnection();
		final LowLevelCommunication lowLevelCommunication = createLowLevelCommunication(connection, false);
		final ServerConnectionProperties properties = new ServerConnectionProperties(
				lowLevelCommunication, _lowLevelConnectionsManager.getLowLevelAuthentication().getAuthenticationComponent(), _serverDavParameters
		);
		lowLevelCommunication.connect(applicationInfo.getAddress(), applicationInfo.getSubAddress());

		return new T_A_HighLevelCommunication(
				properties, _applicationManager, _lowLevelConnectionsManager, false, !_configurationAvailable
		);
	}

	/**
	 * Startet den Verbindungsaufbau zwischen zwei Datenverteilern. Falls keine Verbindung etabliert werden konnte, wird eine entsprechende Exception gefangen
	 *
	 * @param applicationInfo Information zum Datenverteiler
	 * @param time            Zeitspanne in der versucht werden soll eine Verbindung aufzubauen, in Millisekunden. Minimale Wartezeit eine Sekunde.
	 * @return true, wenn Verbindung hergestellt werden konnte; false, wenn Verbindung nicht hergestellt werden konnte.
	 * @see #connectToApplication(ApplicationInfo)
	 */
	private boolean connectToApplication(final ApplicationInfo applicationInfo, final long time) {
		long waitingTime = time;
		if(waitingTime < 1000) {
			waitingTime = 1000;
		}
		final long startTime = System.currentTimeMillis();
		do {
			try {
				connectToApplication(applicationInfo);
				return true;
			}
			catch(ConnectionException ex) {
				_debug.fine("Applikationsverbindung zum " + applicationInfo + " konnte nicht aufgebaut werden", ex);
				try {
					Thread.sleep(_reconnectionDelay);
				}
				catch(InterruptedException e) {
					return false;
				}
				waitingTime -= (System.currentTimeMillis() - startTime);
			}
		}
		while(waitingTime > 0);
		return false;
	}

	/**
	 * Etabliert Verbindung zwischen zwei Datenverteilern. Falls keine Verbindung aufgebaut werden konnte, wird eine entsprechende Ausnahme geworfen.
	 *
	 * @param applicationInfo Informationen zum Datenverteiler.
	 * @throws ConnectionException wenn eine Verbindung nicht aufgebaut werden konnte
	 */
	private void connectToApplication(final ApplicationInfo applicationInfo)
			throws ConnectionException {
		if(_serverConnection == null) {
			throw new IllegalArgumentException("Die Verwaltung ist nicht richtig initialisiert.");
		}

		for(final T_A_HighLevelCommunication applicationConnection : _connections.getAllConnections()) {
			if(applicationConnection != null && !applicationConnection.isClosed()) {
				final String address = applicationConnection.getRemoteAdress();
				final int subAddress = applicationConnection.getRemoteSubadress();
				if((address != null) && (address.equals(applicationInfo.getAddress())) && (applicationInfo.getSubAddress() == subAddress)) {
					// Verbindung mit einem Datenverteiler mit der angegebenen Adresse besteht bereits
					return;
				}
			}
		}

		final T_A_HighLevelCommunication highLevelCommunication = startApplicationConnection(applicationInfo);
		_connections.addUnsortedConnection(highLevelCommunication);

		_debug.info("Verbindungsaufbau zum " + applicationInfo + " war erfolgreich");
	}

	/**
	 * Terminiert die Versuche, neue Verbindungen aufzubauen
	 */
	public void close() {
		_applicationReconnectService.shutdown();
	}

	@Override
	public String toString() {
		return "OutgoingApplicationConnections{" +
				"_configurationAvailable=" + _configurationAvailable +
				'}';
	}

	class ApplicationReconnectionTask implements Runnable {

		private final ApplicationInfo _applicationConnectionInfo;

		protected ApplicationReconnectionTask(ApplicationInfo applicationConnectionInfo) {
			_applicationConnectionInfo = applicationConnectionInfo;
		}

		/**
		 * Behandelt den Verbindungsaufbau mit einem entfernten Datenverteiler (Application)
		 */
		@Override
		public final void run() {
			if(_lowLevelConnectionsManager.isClosing()) return;

			if(_applicationConnectionInfo != null) {
				if(!connectToApplication(_applicationConnectionInfo, 1000)) {
					// Nach ein paar Sekunden neuen Verbindungsversuch starten.
					scheduleApplicationConnect(_applicationConnectionInfo, _reconnectionDelay, TimeUnit.MILLISECONDS);
				}
			}
		}
	}

	void scheduleApplicationConnect(final ApplicationInfo applicationConnectionInfo, final int delay, final TimeUnit timeUnit) {
		if(_lowLevelConnectionsManager.isClosing()) return;
		_applicationReconnectService.schedule(new ApplicationReconnectionTask(applicationConnectionInfo), delay, timeUnit);
	}
}
