/*
 * Rahmenwerk-Plug-in "Darstellungsobjekte"
 * Copyright (C) 2023 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.darstellung.actions;

import java.text.DecimalFormat;
import java.text.NumberFormat;

import org.eclipse.draw2d.geometry.Point;
import org.eclipse.draw2d.geometry.PrecisionPoint;
import org.eclipse.gef.EditPart;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.swt.events.MouseEvent;
import org.eclipse.swt.events.MouseMoveListener;
import org.eclipse.ui.IWorkbenchPage;

import de.bsvrz.buv.plugin.darstellung.editparts.AnsichtEditPart;
import de.bsvrz.buv.plugin.darstellung.editparts.DarstellungEditPart;
import de.bsvrz.buv.plugin.darstellung.model.MassstaeblicheDarstellung;
import de.bsvrz.buv.plugin.dobj.internal.RahmenwerkService;
import de.bsvrz.buv.plugin.dobj.model.DoModel;
import de.bsvrz.buv.plugin.dobj.util.Projektion;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.sys.funclib.bitctrl.modell.ObjektFactory;
import de.bsvrz.sys.funclib.bitctrl.modell.SystemObjekt;
import de.bsvrz.sys.funclib.bitctrl.modell.tmgeoreferenzierungglobal.konfigurationsdaten.KdPunktKoordinaten;
import de.bsvrz.sys.funclib.bitctrl.modell.tmgeoreferenzierungglobal.objekte.PunktXY;

/**
 * Ein Status Item welches die Position des Cursors anzeigt.
 *
 * @author BitCtrl Systems GmbH, Falko Schumann
 *
 */
public class PositionStatus extends DarstellungStatus implements MouseMoveListener, ISelectionChangedListener {

	private static final NumberFormat FORMATER;
	static {
		FORMATER = new DecimalFormat("###.######");
		FORMATER.setMinimumFractionDigits(6);
		FORMATER.setMaximumFractionDigits(6);
	}

	// private static final String COMMA = FORMATER.format(1.1).substring(1, 2);

	private Point mouseLocation;

	private Point primaryLocation;

	/**
	 * Initialisiert das Status Item.
	 *
	 * @param page
	 *            die Workbench Page für den das Item bestimmt ist.
	 */
	public PositionStatus(final IWorkbenchPage page) {
		super(page, PositionStatus.class.toString(), 35);
	}

	@Override
	protected void activate() {
		getViewer().getControl().addMouseMoveListener(PositionStatus.this);
		getViewer().addSelectionChangedListener(this);
	}

	@Override
	protected void deactivate() {
		getViewer().removeSelectionChangedListener(this);
		getViewer().getControl().removeMouseMoveListener(PositionStatus.this);
	}

	@Override
	public void mouseMove(final MouseEvent e) {
		if (getZoomManager() == null) {
			return;
		}

		int offset = getZoomManager().getViewport().getHorizontalRangeModel().getValue();
		final double x = (offset + e.x) / getZoomManager().getZoom();
		offset = getZoomManager().getViewport().getVerticalRangeModel().getValue();
		final double y = (offset + e.y) / getZoomManager().getZoom();

		mouseLocation = new Point((int) x, (int) y);

		updateAnzeige();
	}

	@Override
	public void selectionChanged(final SelectionChangedEvent event) {
		primaryLocation = null;

		if (event.getSelection() instanceof StructuredSelection) {
			final EditPart primSel = getPrimarySelectionOf((StructuredSelection) event.getSelection());
			if (primSel != null) {
				primaryLocation = getLocationOf(primSel);
				if (primaryLocation == null && !(primSel instanceof AnsichtEditPart)
						&& !(primSel instanceof DarstellungEditPart<?>)) {
					primaryLocation = mouseLocation;
				}
			}
		}

		updateAnzeige();
	}

	private void updateAnzeige() {
		if (getViewer() != null && getViewer().getProperty(Projektion.class.toString()) != null) {
			final Projektion projektion = (Projektion) getViewer().getProperty(Projektion.class.toString());
			String text = "";

			if (primaryLocation != null) {
				if (getDarstellung() instanceof MassstaeblicheDarstellung) {
					if (primaryLocation instanceof PrecisionPoint) {
						text = getWGSString(primaryLocation);
					} else {
						text = getWGSString(projektion.umrechnenModellNachWelt(primaryLocation));
					}
				} else {
					final NumberFormat formatter = NumberFormat.getIntegerInstance();
					text = formatter.format(primaryLocation.x) + ", " + formatter.format(primaryLocation.y);
				}
			} else if (mouseLocation != null) {
				if (getDarstellung() instanceof MassstaeblicheDarstellung) {
					text = getWGSString(projektion.umrechnenModellNachWelt(mouseLocation));
				} else {
					final NumberFormat formatter = NumberFormat.getIntegerInstance();
					text = formatter.format(mouseLocation.x) + ", " + formatter.format(mouseLocation.y);
				}
			}

			setText(text);
		}
	}

	private static String getWGSString(final Point wgsPoint) {
		String wgsString = "";

		if (wgsPoint.preciseY() < 0) {
			wgsString = FORMATER.format(Math.abs(wgsPoint.preciseY())) + "° S";
		} else {
			wgsString = FORMATER.format(wgsPoint.preciseY()) + "° N";
		}

		wgsString += " / ";

		if (wgsPoint.preciseX() < 0) {
			wgsString += FORMATER.format(Math.abs(wgsPoint.preciseX())) + "° W";
		} else {
			wgsString += FORMATER.format(wgsPoint.preciseX()) + "° O";
		}

		return wgsString;
	}

	/**
	 * Erfragt die Position eines EditParts.
	 *
	 * @param ep
	 *            ein EditPart.
	 * @return die Position des uebergebenen EditParts, oder <code>null</code>,
	 *         wenn dieser nicht verortbar ist.
	 */
	private Point getLocationOf(final EditPart ep) {
		if (ep != null) {
			if (!(ep instanceof AnsichtEditPart) && ep.getModel() instanceof DoModel) {
				final DoModel dom = (DoModel) ep.getModel();

				if (getDarstellung() instanceof MassstaeblicheDarstellung) {
					final PrecisionPoint editPartLocation = getSystemObjectLocation(dom.getSystemObject());
					if (editPartLocation != null) {
						return editPartLocation;
					}
				}

				return dom.getLocation();
			}
		}

		return null;
	}

	/**
	 * Erfragt die konfigurierte Position eines Systemobjekts.
	 *
	 * @param sysObj
	 *            ein Systemobjekt.
	 * @return die konfigurierte Position des uebergebenen Systemobjekts, oder
	 *         <code>null</code>, wenn dieses nicht verortbar ist.
	 */
	private static PrecisionPoint getSystemObjectLocation(final SystemObject sysObj) {
		PrecisionPoint location = null;

		if (sysObj != null) {
			final ObjektFactory of = RahmenwerkService.getService().getObjektFactory();
			final SystemObjekt bcSysObj = of.getModellobjekt(sysObj.getPid());
			if (bcSysObj != null) {
				if (bcSysObj instanceof final PunktXY punkt) {
					final KdPunktKoordinaten.Daten daten = punkt.getKdPunktKoordinaten().getDatum();
					if (daten != null && daten.getX() != null && daten.getY() != null) {
						location = new PrecisionPoint(daten.getX().doubleValue(), daten.getY().doubleValue());
					}
				}
			}
		}

		return location;
	}

	/**
	 * Erfragt die primaere Selektion als EditPart.
	 *
	 * @param sel
	 *            eine Selektion.
	 * @return die primaere Selektion als EditPart oder <code>null</code>, wenn
	 *         kein EditPart selektiert wurde.
	 */
	private static EditPart getPrimarySelectionOf(final StructuredSelection sel) {
		EditPart primarySelectedEditPart = null;

		if (!sel.isEmpty()) {
			for (final Object element : sel.toArray()) {
				if (element instanceof final EditPart ep) {
					if (ep.getSelected() == EditPart.SELECTED_PRIMARY) {
						primarySelectedEditPart = ep;
						break;
					}
				}
			}
		}

		return primarySelectedEditPart;
	}

}
