/*
 * Rahmenwerk-Plug-in "Darstellungsobjekte"
 * Copyright (C) 2023 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.darstellung.actions;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import org.eclipse.draw2d.geometry.Rectangle;
import org.eclipse.gef.EditPart;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.ActionContributionItem;
import org.eclipse.jface.action.IMenuCreator;
import org.eclipse.jface.dialogs.InputDialog;
import org.eclipse.jface.window.Window;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.swt.widgets.MenuItem;
import org.eclipse.ui.IWorkbenchPart;

import com.bitctrl.lib.eclipse.emf.eclipse.model.EclipsePackage;
import com.bitctrl.lib.eclipse.emf.gef.commands.AddCommand;
import com.bitctrl.lib.eclipse.emf.gef.commands.RemoveCommand;
import com.bitctrl.lib.eclipse.emf.gef.commands.SetCommand;

import de.bsvrz.buv.plugin.darstellung.dialogs.AusschnittAnlegenDialog;
import de.bsvrz.buv.plugin.darstellung.dialogs.AusschnitteVerwaltenDialog;
import de.bsvrz.buv.plugin.darstellung.editparts.AnsichtEditPart;
import de.bsvrz.buv.plugin.darstellung.internal.DarstellungMessages;
import de.bsvrz.buv.plugin.darstellung.model.Ansicht;
import de.bsvrz.buv.plugin.darstellung.model.Ausschnitt;
import de.bsvrz.buv.plugin.darstellung.model.DarstellungFactory;
import de.bsvrz.buv.plugin.darstellung.model.DarstellungPackage;
import de.bsvrz.buv.plugin.darstellung.util.DarstellungIcons;
import de.bsvrz.buv.plugin.dobj.actions.DobjWorkbenchPartAction;

/**
 * Aktiviert und deaktiviert das Synchronisieren der Selektion zwischen
 * Workbench Parts.
 *
 * @author BitCtrl Systems GmbH, Falko Schumann
 *
 */
public class SelectAusschnittAction extends DobjWorkbenchPartAction implements IMenuCreator {

	private class AusschnittAnzeigenAction extends Action {

		private final Ansicht ansicht;
		private final Ausschnitt ausschnitt;

		AusschnittAnzeigenAction(final Ansicht ansicht, final Ausschnitt ausschnitt) {
			super(ausschnitt.getName());
			this.ansicht = ansicht;
			this.ausschnitt = ausschnitt;
			if (ausschnitt.equals(ansicht.getAusschnitt())) {
				setText(ausschnitt.getName() + " (Standardausschnitt)");
			}
		}

		@Override
		public boolean isEnabled() {
			return ausschnitt != null && ansicht != null;
		}

		@Override
		public void run() {
			if (isEnabled()) {
				getZoomManager().setZoom(ausschnitt.getZoomLevel());
				getZoomManager().getViewport().setHorizontalLocation(ausschnitt.getBounds().x);
				getZoomManager().getViewport().setVerticalLocation(ausschnitt.getBounds().y);
			}
		}
	}

	private class DefaultAnsichtAction extends Action {

		private final Ansicht ansicht;

		DefaultAnsichtAction(final Ansicht ansicht) {
			super("Gesamtansicht anzeigen");
			this.ansicht = ansicht;
		}

		@Override
		public boolean isEnabled() {
			return ansicht != null;
		}

		@Override
		public void run() {
			if (isEnabled()) {
				getZoomManager().setZoom(1 / getZoomManager().getUIMultiplier());
				getZoomManager().getViewport().setHorizontalLocation(0);
				getZoomManager().getViewport().setVerticalLocation(0);
			}
		}

	}

	private class AusschnitteVerwaltenAction extends Action {
		private final Ansicht ansicht;

		AusschnitteVerwaltenAction(final Ansicht ansicht) {
			super("Ausschnitte verwalten");
			this.ansicht = ansicht;
		}

		@Override
		public boolean isEnabled() {
			return ansicht != null && !ansicht.getAusschnitte().isEmpty();
		}

		@Override
		public void run() {
			final AusschnitteVerwaltenDialog dialog = new AusschnitteVerwaltenDialog(
					getWorkbenchPart().getSite().getShell(), ansicht);
			if (dialog.open() == Window.OK) {
				final Map<Ausschnitt, String> result = dialog.getResult();
				final List<Ausschnitt> original = new ArrayList<>(ansicht.getAusschnitte());
				final SetCommand defaultCommand = new SetCommand(ansicht,
						DarstellungPackage.Literals.ANSICHT__AUSSCHNITT, dialog.getDefaultAusschnitt());
				getCommandStack().execute(defaultCommand);

				for (final Ausschnitt ausschnitt : original) {
					if (!result.containsKey(ausschnitt)) {
						final RemoveCommand command = new RemoveCommand(ansicht,
								DarstellungPackage.Literals.ANSICHT__AUSSCHNITTE, ausschnitt);
						getCommandStack().execute(command);
					} else {
						final String name = result.get(ausschnitt);
						if (!ausschnitt.getName().equals(name)) {
							final SetCommand command = new SetCommand(ausschnitt, EclipsePackage.Literals.NAMED__NAME,
									name);
							getCommandStack().execute(command);
						}
					}
				}
			}
		}
	}

	private class AusschnittSichernAction extends Action {

		private final Ansicht ansicht;

		AusschnittSichernAction(final Ansicht ansicht) {
			super("Aktuellen Ausschnitt sichern");
			this.ansicht = ansicht;
		}

		@Override
		public boolean isEnabled() {
			return ansicht != null;
		}

		@Override
		public void run() {
			if (isEnabled()) {
				final InputDialog dialog = new AusschnittAnlegenDialog(ansicht);
				if (dialog.open() == Window.OK) {
					final Rectangle clientArea = getZoomManager().getViewport().getClientArea();
					final double zoom = getZoomManager().getZoom();
					final String name = dialog.getValue();

					Ausschnitt ausschnitt = null;
					for (final Ausschnitt item : ansicht.getAusschnitte()) {
						if (item.getName().equals(name)) {
							ausschnitt = item;
							break;
						}
					}

					if (ausschnitt == null) {
						ausschnitt = DarstellungFactory.eINSTANCE.createAusschnitt();
						ausschnitt.setZoomLevel(zoom);
						ausschnitt.setBounds(clientArea);
						ausschnitt.setName(name);
						final AddCommand command = new AddCommand(ansicht,
								DarstellungPackage.Literals.ANSICHT__AUSSCHNITTE, ausschnitt);
						getCommandStack().execute(command);
					} else {
						SetCommand command = new SetCommand(ausschnitt, DarstellungPackage.Literals.AUSSCHNITT__BOUNDS,
								clientArea);
						getCommandStack().execute(command);
						command = new SetCommand(ausschnitt, DarstellungPackage.Literals.AUSSCHNITT__ZOOM_LEVEL, zoom);
						getCommandStack().execute(command);
					}
				}
			}
		}
	}

	private Menu menu;

	/**
	 * Initialisiert die Aktion.
	 *
	 * @param part
	 *            der Workbench Part dessen Selektion mit anderen
	 *            synchronisieren werden soll.
	 */
	public SelectAusschnittAction(final IWorkbenchPart part) {
		super(part, AS_DROP_DOWN_MENU);

		setText(DarstellungMessages.SelectAusschnitt_Label);
		setImageDescriptor(DarstellungIcons.ActionSelectAusschnitt.getImageDescriptor());
		setToolTipText(DarstellungMessages.SelectAusschnitt_Tooltip);

		setId(DarstellungActionFactory.SELECT_AUSSCHNITT.getCommandId());
		setActionDefinitionId(DarstellungActionFactory.SELECT_AUSSCHNITT.getCommandId());
	}

	@Override
	public IMenuCreator getMenuCreator() {
		return this;
	}

	@Override
	public void run() {
		new AusschnittSichernAction(getAnsicht()).run();
	}

	@Override
	public void dispose() {
		if (menu != null) {
			menu.dispose();
		}
	}

	@Override
	public Menu getMenu(final Control parent) {
		if (menu != null) {
			menu.dispose();
		}

		final Ansicht ansicht = getAnsicht();
		menu = new Menu(parent);
		if (ansicht != null) {
			addActionToMenu(menu, new AusschnittSichernAction(ansicht));
			addActionToMenu(menu, new AusschnitteVerwaltenAction(ansicht));
			new MenuItem(menu, SWT.SEPARATOR);
			addActionToMenu(menu, new DefaultAnsichtAction(ansicht));
			for (final Ausschnitt ausschnitt : ansicht.getAusschnitte()) {
				addActionToMenu(menu, new AusschnittAnzeigenAction(ansicht, ausschnitt));
			}
		}

		return menu;
	}

	private Ansicht getAnsicht() {
		final EditPart content = getViewer().getContents();
		if (content instanceof AnsichtEditPart) {
			final Object ansicht = content.getModel();
			if (ansicht instanceof Ansicht) {
				return (Ansicht) ansicht;
			}
		}
		return null;
	}

	@Override
	public Menu getMenu(final Menu parent) {
		return null;
	}

	protected void addActionToMenu(final Menu parent, final Action action) {
		final ActionContributionItem item = new ActionContributionItem(action);
		item.fill(parent, -1);
	}
}
